<?php
/*
 * Copyright 2012 Sébastien Raud
 *
 * This file is part of beCms.
 *
 * beCms is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * beCms is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with beCms.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * Specific back-office application dispatcher.
 *
 * @return void
 */
function back_dispatch()
{
    init_messages();
    // @todo fix it
    load_language(get_user_property('lang', 'fr'));

    $b_is_ajax = 1 == get_request_parameter('ajax');

    if (get_configuration('administration/is_secure') && !current_uri_is_securised())
    {
        disconnection();
        set_message('error', __('ErrSecurity'));
        flash_messages_and_redirect('login');
    }

    set_configuration('server/session/duration', 3600); // 1 hour

    if (!is_connected())
    {
        if ($b_is_ajax)
        {
            set_message('error', __('ErrSecurity'));
            send_ajax(array('html' => '<div class="ui-message-box error">' . __('ErrSecurity') . '</div>'));
        }
        if (!in_array(get_current_uri(true), array(get_back_uri('home'), get_back_uri('login')))) {
            setcookie('redirect', get_current_uri(), time() + 900, '/');
            flash_messages_and_redirect('home');
        }
        set_configuration('app/page', 'login');
    }
    else
    {
        // refresh cookie time
        if (!connection(get_user_id()))
        {
            disconnection();
            redirect('login');
        }
        else
        {
            // extract page from parameters
            $a_parameters = get_uri_parameters();

            if (!isset($a_parameters['page']))
            {
                set_message('error', __('ErrPageNotFound'));
                flash_messages_and_redirect('home');
            }

            $a_page_path = explode('/', $a_parameters['page']);
            array_shift($a_page_path); // remove first token (administration route)
            if (!count($a_page_path)) redirect('home');
            else
            {
                set_configuration('app/page', $a_page_path[0]);
                array_shift($a_page_path);
                set_configuration('app/page-path', $a_page_path);
            }
        }
    }

    // prevent CSRF attacks with post action
    if ('login' !== get_configuration('app/page') && ('POST' === get_request_method() || get_request_parameter('action')) && !has_session_token())
    {
        disconnection();
        set_message('error', __('ErrSecurity'));
        flash_messages_and_redirect('login');
    }

    include get_configuration('app/path/back').'controllers.php';

    if (!function_exists($s_controller = get_configuration('app/page', 'login').'_controller'))
    {
        set_message('error', __('ErrPageNotFound'));
        flash_messages_and_redirect('home');
    }

    $r = array('controller' => $s_controller);
    module_chain(__FUNCTION__, $r);

    // continue
    call_user_func($s_controller);
    flash_message_to_messages();
    if ($b_is_ajax) send_ajax(get_page_vars());
    else display_page();
}

/**
 * Indicates if the request contains the session token.
 *
 * @return boolean
 */
function has_session_token()
{
    return isset($_COOKIE['session_token']) && $_COOKIE['session_token'] === get_request_parameter('session_token');
}

/**
 * Returns the session token tag for html forms.
 *
 * @param  boolean  $b_rand_id  Adds a random id, default : false.
 * @return string
 */
function get_session_token_tag($b_rand_id = false)
{
    return sprintf('<input type="hidden" id="session_token'.($b_rand_id ? '_'.str_replace('.', '_', generate_id()) : '').'" name="session_token" value="%s" />', $_COOKIE['session_token']);
}

/**
 * Tests if user is connected.
 *
 * @return boolean
 */
function is_connected()
{
    $a_users = get_users();

    return (get_user_id() &&
            get_user_properties() &&
            isset($_COOKIE['session_token']) &&
            isset($_COOKIE['end_session_time']) &&
            $_COOKIE['end_session_time'] > time() &&
            isset($_COOKIE['http_user_agent']) &&
            $_SERVER['HTTP_USER_AGENT'] === $_COOKIE['http_user_agent']);
}

/**
 * Makes connection for user.
 *
 * @param  string  $s_user  User id.
 * @return boolean
 */
function connection($s_user)
{
    if (!$s_user)
        return false;

    $s_session_token = isset($_COOKIE['session_token']) ? $_COOKIE['session_token'] : md5(uniqid(srand(), TRUE));

    $i_end_session_time = time() + get_configuration('server/session/duration', 3600); // 1 hour
    return  setcookie('user', $s_user, $i_end_session_time, '/') &&
            setcookie('end_session_time', $i_end_session_time, $i_end_session_time, '/') &&
            setcookie('http_user_agent', $_SERVER['HTTP_USER_AGENT'], $i_end_session_time, '/') && // prevent session hijacking
            setcookie('session_token', $s_session_token, $i_end_session_time, '/');
}

/**
 * Makes disconnection for current user.
 *
 * @return boolean
 */
function disconnection()
{
    $i_time = time();
    return  setcookie('user', '', $i_time - 3600, '/') &&
            setcookie('end_session_time', '', $i_time - 3600, '/') &&
            setcookie('http_user_agent', '', $i_time - 3600, '/') &&
            setcookie('session_token', '', $i_time - 3600, '/');
}

function include_view($s_view, $a_vars = array())
{
    if (null !== ($a_user_vars = get_application_parameter('_page/_template/_vars'))) extract($a_user_vars);
    if (count($a_vars)) extract($a_vars);

    $s_view_path = get_configuration('app/path/contents').'back' . DIRECTORY_SEPARATOR . 'view.';

    if (file_exists($s_file = $s_view_path.$s_view.'.php'))
        include $s_file;
}

function get_view($s_view, $a_vars = array())
{
    if (null !== ($a_user_vars = get_application_parameter('_page/_template/_vars'))) extract($a_user_vars);
    if (count($a_vars)) extract($a_vars);

    $s_view_path = get_configuration('app/path/contents').'back' . DIRECTORY_SEPARATOR . 'view.';

    if (file_exists($s_file = $s_view_path.$s_view.'.php'))
    {
        ob_start();
        include $s_file;
        return ob_get_clean();
    }
    return '';
}


/**
 * Returns the users list.
 *
 * @return array
 */
function get_users()
{
    $s_users_file = get_configuration('app/path/data').'users.php';

    if (!file_exists($s_users_file))
    {
        echo __('ErrFileNotRead', $s_users_file);
        exit;
    }

    return include $s_users_file;
}

/**
 * Returns user name from cookie.
 *
 * @return string
 */
function get_user_id()
{
    return isset($_COOKIE['user']) ? $_COOKIE['user'] : null;
}

/**
 * Returns a property for a user.
 *
 * @param  string  $s_key  Property key.
 * @return mixed
 */
function get_user_property($s_key, $m_default = null)
{
    $a_user_properties = get_user_properties();
    return array_key_exists($s_key, $a_user_properties) ? $a_user_properties[$s_key] : $m_default;
}

/**
 * Returns user properties.
 *
 * @return array
 */
function & get_user_properties()
{
    static $a_user_properties = array();
    static $b_loaded = false;

    if (!$b_loaded)
    {
        if ($s_user_name = get_user_id())
        {
            $a_users = get_users();
            if (isset($a_users[$s_user_name]))
                $a_user_properties = $a_users[$s_user_name];
        }
    }

    return $a_user_properties;
}

/**
 * Sets the flash messages (messages as cookie) and redirect to a page.
 *
 * @param  string   $s_page         Page key.
 * @param  integer  $i_status_code  Redirection status code, default : 302.
 * @param  array    $a_parameters   Uri parameters.
 * @param  string   $s_application  Application name (APP_BACK, APP_FRONT).
 * @return void
 */
function flash_messages_and_redirect($s_page, $i_status_code = 302, $a_parameters = array(), $s_application = null)
{
    $a_messages = get_application_parameter('messages');
    foreach ($a_messages as $s_type => $a_messages)
        if (count($a_messages))
            set_flash_var('messages/'.$s_type, $a_messages);

    if (1 != get_request_parameter('ajax'))
        redirect($s_page, $i_status_code, $a_parameters, $s_application);
    else
    {
        send_ajax(array('html' => '<div class="ui-message-box error">' . __('ErrSecurity') . '</div>'));
        exit;
    }
}

/**
 * Initialize messages in application parameters.
 *
 * @return void
 */
function init_messages()
{
    set_application_parameter('messages', array('success' => array(), 'error' => array(), 'warning' => array(), 'info' => array()));
}

/**
 * Clears messages.
 *
 * @param  string  $s_type  Message type (success, error, warning or info). Clear these type if specified.
 * @return void
 */
function clear_message($s_type = null)
{
    if (!$s_type) init_messages();
    else set_application_parameter('messages/' . $s_type, array());
}

/**
 * Returns a specific message list.
 *
 * @param  string  $s_type  Message type (success, error, warning or info).
 * @return array
 */
function get_message($s_type)
{
    return get_application_parameter('messages/' . $s_type);
}

/**
 * Adds a message.
 *
 * @param  string  $s_type     Message type (success, error, warning or info).
 * @param  mixed   $m_message  string / array. Message(s).
 * @return void
 */
function set_message($s_type, $m_message)
{
    $a_message = get_application_parameter('messages/' . $s_type, array());
    if (is_array($m_message)) $a_message = array_merge($a_message, $m_message);
    else $a_message[] = $m_message;
    set_application_parameter('messages/' . $s_type, $a_message);
}

/**
 * Adds a flash var (cookie removed when a page is reloaded).
 *
 * @param  string  $s_key  Variable name.
 * @param  mixed   $m_var  Value.
 * @return void
 */
function set_flash_var($s_key, $m_var = null)
{
    setcookie('flash/var/'.$s_key, ($m_var ? serialize($m_var) : ''), time() + 1 * 360, '/');
}

/**
 * Returns a flash var.
 *
 * @param  string  $s_key      Variable name.
 * @param  mixed   $m_default  Return value if $sk_ey not exists.
 * @return mixed
 */
function get_flash_var($s_key, $m_default = null)
{
    $s_val = has_flash_var($s_key) ? unserialize($_COOKIE['flash/var/'.$s_key]) : $m_default;
    setcookie('flash/var/'.$s_key, false, time(), '/');
    return $s_val;
}

/**
 * Indicates if a flash var exists.
 *
 * @param  string  $s_key      Variable name.
 * @return boolean
 */
function has_flash_var($s_key)
{
    return isset($_COOKIE['flash/var/'.$s_key]);
}


/**
 * Adds a message.
 *
 * @param  string  $s_type     error, warning, success.
 * @param  string  $s_message  Message.
 * @return void
 */
function set_flash_message($s_type, $s_message)
{
    $a_message = get_flash_var('messages/'.$s_type, array());
    $a_message[] = $s_message;
    set_flash_var('messages/'.$s_type, $a_message);
}

/**
 * Add flash messages to application messages.
 *
 * @return void
 */
function flash_message_to_messages()
{
    foreach (array('error', 'warning', 'success', 'info') as $s_type_message)
    {
        if (has_flash_var('messages/'.$s_type_message))
        {
            set_message($s_type_message, get_flash_var('messages/'.$s_type_message, array()));
            set_flash_var($s_type_message);
        }
    }
}

/**
 * Sends a standardized ajax web response.
 *
 * @param  array  $a_values  Values to sends.
 * @return void
 */
function send_ajax($a_values = array())
{
    flash_message_to_messages();
    header('Content-type: text/xml');

    echo '<?xml version="1.0" encoding="UTF-8"?>';
    echo '<data>';
    echo messages_to_xml();
    /*
    $i_end = microtime_float();
    $i_end_memory = memory_get_usage();
    global $i_base_memory;
    global $i_start;
    echo array_to_xml(array(
        'memory' => ($i_end_memory / 1024).' ko '.(($i_end_memory - $i_base_memory) / 1024) . 'ko',
        'time' => (($i_end - $i_start) * 1000).' µs'
    ));
    */
    echo array_to_xml($a_values);
    echo '</data>';
    exit;
}

/**
 * Returns a CDATA string.
 *
 * @param  string  $s  String to convert.
 * @return string
 */
function xml_cdata($s)
{
    return '<![CDATA['.$s.']]>';
}

/**
 * Returns an xml string with messages.
 *
 * @return string
 */
function messages_to_xml()
{
    $a_messages = get_application_parameter('messages');
    return '<messages>'.array_to_xml($a_messages).'</messages>';
}

/**
 * Returns an xml string from array.
 *
 * @param  array  $a  Array to convert.
 * @return string
 */
function array_to_xml($a)
{
    $s_xml = '';
    if (!count($a)) return '';

    foreach ($a as $m_key => $m)
    {
        $s_xml .= '<'. (is_int($m_key) ? 'array_index_' : '') . $m_key .'>';
        $s_xml .= (is_array($m) ? array_to_xml($m) : '<![CDATA['.str_replace(array('<![CDATA[', ']]>'), array('$$CDATA_START$$', '$$CDATA_END$$'), $m).']]>'); // preserve inner cdata sections
        $s_xml .= '</'. (is_int($m_key) ? 'array_index_' : '') . $m_key .'>';
    }

    return $s_xml;
}
