<?php
/*
 * Copyright 2012 Sébastien Raud
 *
 * This file is part of beCms.
 *
 * beCms is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * beCms is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with beCms.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * Login controller.
 *
 * @return void
 */
function login_controller()
{
    if ('POST' === get_request_method())
    {
        $a_users = include get_configuration('app/path/data').'users.php';
        $a_form_values = get_request_parameter('connection', array());

        if ($a_form_values && array_key_exists('login', $a_form_values) && array_key_exists('password', $a_form_values))
        {
            include get_configuration('app/path/back').'users.php';
            if (array_key_exists($a_form_values['login'], $a_users) && !strcmp(user_password_encode($a_form_values['password']), $a_users[$a_form_values['login']]['password']))
            {
                if (connection($a_form_values['login']))
                {
                    $s_page = isset($_COOKIE['redirect']) ? $_COOKIE['redirect'] : 'home';
                    setcookie('redirect', null, time() - 3600, '/');
                    flash_messages_and_redirect($s_page);
                }
            }
        }

        set_message('error', __('ErrLoginPassword'));
    }

    disconnection();
}

/**
 * Quit controller.
 *
 * @return void
 */
function quit_controller()
{
    disconnection();
    set_message('info', __('MsgLoggoutOk'));
    flash_messages_and_redirect('login');
}

/**
 * Home controller.
 *
 * @return void
 */
function home_controller()
{
}

/**
 * Users controller.
 *
 * @return void
 */
function users_controller()
{
    redirect_if_not_admin();

    // action save : POST only
    // actions edit and delete uses id
    $b_is_ajax = (1 == get_request_parameter('ajax'));
    set_page_var('b_is_ajax', $b_is_ajax);
    $b_is_success = true;

    if (!($s_action = get_request_parameter('action'))) $s_action = 'list';
    if ('save' === $s_action && ('POST' !== get_request_method())) { $b_is_success = false; set_message('error', __('ErrRequestPostOnly')); }
    if (in_array($s_action, array('edit', 'delete')) && !($s_id = get_request_parameter('id'))) { $b_is_success = false; set_message('error', __('ErrRequestData', 'id')); }

    include get_configuration('app/path/back').'users.php';

    if ($b_is_success)
    {
        switch ($s_action)
        {
            case 'add':
                set_page_var('a_edit', array_merge(array('login' => ''), user_get_default()));
                set_page_var('s_type_action', 'add');
                break;

            case 'delete':
                $b_is_success = user_delete(array('login' => $s_id));
                break;

            case 'list':
                $a_users = users_get_list();
                set_page_var('a_users', $a_users);
                break;

            case 'save':
                $s_type_action = get_request_parameter('type_action', 'save');
                $a_user = get_request_parameter('user');

                if ('save' === $s_type_action)
                    $b_is_success = user_modify($a_user);
                else
                    $b_is_success = user_new($a_user);

                $a_user['password'] = user_password_encode($a_user['password']);

                set_page_var('a_edit', $a_user);
                set_page_var('s_type_action', $b_is_success ? 'save' : $s_type_action);

                break;

            case 'edit':
                $a_users = get_users();
                if (!$s_id) set_page_var('a_edit', array_merge(array('login' => ''), user_get_default()));
                else
                {
                    if (!($b_is_success = array_key_exists($s_id, $a_users))) set_message('error', __('ErrUserNotExists', $s_id));
                    else set_page_var('a_edit', array_merge(array('login' => $s_id), $a_users[$s_id]));
                }
                set_page_var('s_type_action', $s_id ? 'save' : 'add');
                break;

        }
    }

    if ($b_is_success && !in_array($s_action, array('list', 'edit', 'add'))) set_message('success', __('MsgOperationOk'));
    if ($b_is_ajax)
    {
        if (in_array($s_action, array('add', 'edit', 'save')))
        {
            set_page_var('html', get_page_content());
            delete_page_var(array('a_edit', 's_type_action'));
            if ('save' === $s_action && $b_is_success)
            {
                // update users list ui
                set_page_var('a_users', users_get_list());
                set_page_var('html_list', get_page_content());
                delete_page_var('a_users');
            }
        }
        else if ('delete' === $s_action)
        {
            set_page_var('a_users', users_get_list());
            set_page_var('html_list', get_page_content());
            delete_page_var('a_users');
        }
        return true;
    }

    if ('delete' === $s_action) flash_messages_and_redirect('users');
    if ('save' === $s_action && $b_is_success) flash_messages_and_redirect('users', 302, array('action' => 'edit', 'id' => $a_user['login']));
}

/**
 * Configuration controller.
 *
 * @return void
 */
function configuration_controller()
{
    redirect_if_not_admin();
    $b_is_ajax = (1 == get_request_parameter('ajax'));

    if ('POST' === get_request_method())
    {
        $a_configuration = get_request_parameter('configuration');
        $s_action = get_request_parameter('action');
        $b_is_success = true;

        include get_configuration('app/path/back').'configuration.php';

        if ('save' === $s_action)
        {
            $a_configuration['server']['url_rewriting'] = array_key_exists('url_rewriting', $a_configuration['server']) && 'on' === $a_configuration['server']['url_rewriting'] ? true : false;
            $a_configuration['administration']['is_secure'] = array_key_exists('is_secure', $a_configuration['administration']) && 'on' === $a_configuration['administration']['is_secure'] ? true : false;
            if ($b_is_success = configuration_control_values($a_configuration))
                if ($b_is_success = write_data($a_configuration, 'config'))
                    set_message('success', __('MsgOperationOk'));
        }

        if ($b_is_success)
        {
            load_configuration();

            if (!$b_is_ajax)
                flash_messages_and_redirect('configuration');
        }
    }

    include get_configuration('app/path/back').'external_files.php';

    if ($b_is_ajax)
    {
        set_page_var(array('layouts' => array_map(create_function('$e', 'return basename($e, \'.php\');'), external_files_get_layout()),
                        'css' => array_map(create_function('$e', 'return str_replace(array(get_configuration(\'app/path/web\'), \'\\\\\'), array(\'/\', \'/\'), $e);'), external_files_get_css()),
                        'js'  => array_map(create_function('$e', 'return str_replace(array(get_configuration(\'app/path/web\'), \'\\\\\'), array(\'/\', \'/\'), $e);'), external_files_get_js()),
                        'configuration' => get_data('config'), 'url' => get_back_uri('configuration')));
        return true;
    }

    set_page_var('a_layouts', external_files_get_layout());
    set_page_var('a_css', external_files_get_css());
    set_page_var('a_js', external_files_get_js());
}

/**
 * Structure controller.
 *
 * @return void
 */
function structure_controller()
{
    // display structure
    if (!($s_action = get_request_parameter('action'))) return true;

    // actions edit and add, up, down, clear-cache and delete : GET or POST, save : POST only
    // actions add, up, down and delete uses path_id, edit and clear-cache uses id
    $b_is_success = true;
    $b_is_ajax = (1 == get_request_parameter('ajax'));

    // test parameters, methods and session_token
    if (in_array($s_action, array('add', 'clear-cache', 'delete', 'down', 'edit', 'up')) && !($s_id = get_request_parameter('id'))) { $b_is_success = false; set_message('error', __('ErrRequestData', 'id')); }
    else if ('save' === $s_action && ('POST' !== get_request_method())) { $b_is_success = false; set_message('error', __('ErrRequestPostOnly')); }

    if ($b_is_success)
    {
        include get_configuration('app/path/back').'structure.php';

        switch ($s_action)
        {
            case 'add':
                set_page_var(array(
                    'a_edit_element' => ($a_edit_element = structure_get_defaults()),
                    's_parent_id' => $s_id
                ));
                break;

            case 'clear-cache':
                $s_cache_file = get_configuration('app/path/cache').str_replace('/', DIRECTORY_SEPARATOR, preg_replace('~(https?)://~i', '\1/', trim(get_front_uri($s_id), '/'))).'.cache.php';
                if (file_exists($s_cache_file)) unlink($s_cache_file);
                $b_is_success = true;
                break;

            case 'delete':
                $b_is_success = structure_delete_element($s_id);
                break;

            case 'down':
                $b_is_success = structure_down_element($s_id);
                break;

            case 'edit':
                if (!($a_edit_element = structure_get_element($s_id))) $a_edit_element = array();
                set_page_var(array(
                    'a_edit_element' => $a_edit_element,
                ));
                break;

            case 'save':
                $a_edit_element = get_request_parameter('structure');
                // clean checkbox
                foreach (array('show_slug', 'state', 'index_menu', 'is_secure') as $s_key)
                    $a_edit_element[$s_key] = array_key_exists($s_key, $a_edit_element) && 'on' === $a_edit_element[$s_key] ? true : false;

                if (empty($a_edit_element['id']))
                {
                    if (!$b_is_success = structure_add_child($a_edit_element['parent_id'], $a_edit_element)) $a_edit_element['id'] = '';
                }
                else
                    $b_is_success = structure_modify_element($a_edit_element['id'], $a_edit_element);
                $s_id = $a_edit_element['id'];

                if (!$b_is_success) { set_page_var('a_edit_element', $a_edit_element); set_page_var('s_parent_id', $a_edit_element['parent_id']); }
                break;

            case 'up':
                $b_is_success = structure_up_element($s_id);
                break;
        }

        if ($b_is_success && !in_array($s_action, array('add', 'edit'))) set_message('success', __('MsgOperationOk'));

        if (!$b_is_ajax)
        {
            if (in_array($s_action, array('down', 'up'))) { flash_messages_and_redirect('structure', 302, array('action' => 'edit', 'id' => $s_id)); }
            if ('clear-cache' === $s_action) flash_messages_and_redirect('structure', 302, array('action' => 'edit', 'id' => $s_id));
            elseif ('delete' === $s_action) flash_messages_and_redirect('structure');
            elseif ('save' === $s_action && $b_is_success) flash_messages_and_redirect('structure');
        }
        else
        {
            delete_page_vars();
            switch ($s_action)
            {
                case 'down':
                case 'up':
                    if (!($a_edit_element = structure_get_element($s_id))) $a_edit_element = array();

                case 'add':
                case 'edit':
                    set_page_var('html', get_view('structure_element', array('a_edit_element' => $a_edit_element, 's_parent_id' => $s_id)));
                    break;

                case 'save':
                    if (empty($a_edit_element['parent_id']))
                    {
                        $s_title = $a_edit_element['title'];
                        $a_edit_element['title'] = 'http://'.get_configuration('server/base_url');
                    }

                    $s_element = get_view('structure_treeview_element', array('a_properties' => $a_edit_element, 's_parent_id' => $a_edit_element['parent_id']));

                    if (!$a_edit_element['parent_id']) $a_edit_element['title'] = $s_title;

                    set_page_var(array(
                        'html' => get_view('structure_element', array('a_edit_element' => $a_edit_element, 's_parent_id' => $a_edit_element['parent_id'])), //get_structure_edit_element($a_edit_element, $a_edit_element['parent_id']),
                        'line' => $s_element
                    ));
                    break;
            }
        }
    }
}

/**
 * Edit controller.
 *
 * @return void
 */
function edit_controller()
{
    if (!$s_page_id = get_request_parameter('id'))
    {
        set_message('warning', __('MsgSelectPageToEdit'));
        return ;
    }

    // action save and publish by POST
    // actions lock, unlock and edit by GET or POST
    // actions publish, delete and copy exists only for draft mode, by GET or POST
    $b_is_success = true;
    $b_is_draft = 'draft' === get_request_parameter('type');
    $b_is_ajax = (1 == get_request_parameter('ajax'));

    if (!$s_action = get_request_parameter('action')) $s_action = 'edit';
    if (in_array($s_action, array('delete', 'copy', 'publish')) && !$b_is_draft) { set_message('error', __('ErrMustBeDraft')); $b_is_success = false; } // only with draft
    if (in_array($s_action, array('save', 'publish')) && ('POST' !== get_request_method())) { $b_is_success = false; set_message('error', __('ErrRequestPostOnly')); }
    if (in_array($s_action, array('delete', 'copy', 'lock', 'unlock')) && !has_session_token()) {  $b_is_success = false; set_message('error', __('ErrSecurity')); }

    include get_configuration('app/path/back').'page.php';

    if ((in_array($s_action, array('copy', 'delete', 'lock', 'save', 'publish'))) &&
        ($a_properties = page_get_properties($s_page_id, $b_is_draft)) && is_array($a_properties['locked']))
    {
        if (get_user_id() !== $a_properties['locked']['id'])
        {
            set_message('error', __('ErrPageLocked'));
            $s_action = 'edit';
            $b_is_success = false;
        }
    }

    // contains page informations for template
    $a_editable_page = array('is_root' => ($s_page_id === key(get_structure())), 'is_draft' => $b_is_draft, 'has_draft' => !$b_is_draft ? file_exists(get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.draft.php') : true);

    if ('edit' === $s_action)
    {
        if (!$b_is_ajax)
        {
            if (($a_tmp_editable_page = !$b_is_draft ? get_editable_page($s_page_id) : get_editable_page_draft($s_page_id)))
            {
                $a_editable_page = array_merge($a_editable_page, $a_tmp_editable_page);
                $a_editable_page['id'] = $s_page_id;
                set_page_var('a_page', $a_editable_page);
            }
            else
                $b_is_success = false;
        }
    }
    else if ($b_is_success)
    {
        if (in_array($s_action, array('save', 'publish')))
        {
            $a_page = get_request_parameter('page');
            $a_page['state'] = intval($a_page['state']);
            $a_page['css_files'] = array_key_exists('css_files', $a_page) ? trim($a_page['css_files']) : array();
            $a_page['js_files'] = array_key_exists('js_files', $a_page) ? trim($a_page['js_files']) : array();

            $a_page['css_files'] = ($a_page['css_files'] && strlen($a_page['css_files'])) ? explode(";", $a_page['css_files']) : array();
            $a_page['js_files'] = ($a_page['js_files'] && strlen($a_page['js_files'])) ? explode(";", $a_page['js_files']) : array();

        }

        switch ($s_action)
        {
            case 'copy':
                $b_success = page_copy_draft_content($s_page_id);
                break;

            case 'delete':
                    if (file_exists($s_content_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.draft.php')) unlink($s_content_file);
                    if (file_exists($s_content_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.draft.properties.php')) unlink($s_content_file);
                    $b_is_draft = false;
                    $a_editable_page['has_draft'] = false;
                    break;

            case 'lock':
                $a_page_properties = page_get_properties($s_page_id);
                $a_user = array('id' => get_user_id(), 'firstname' => get_user_property('firstname'), 'lastname' => get_user_property('lastname'));
                $b_is_success = page_write_properties($s_page_id, array_merge($a_page_properties, array('locked' => $a_user)), $b_is_draft);
                break;

            case 'publish':
                if (array_key_exists('content', $a_page))
                {
                    if (!($b_is_success = $b_is_success && page_write_content($s_page_id, $a_page['content'])))
                        set_message('error', __('ErrPageContentNotWrite', $a_page['title'], $a_page['id']));

                    if (file_exists($s_properties_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.properties.php'))
                    {
                        unset($a_page['locked']);
                        $a_page = array_merge(include $s_properties_file, $a_page);
                    }

                    if (!($b_is_success = $b_is_success &&  page_write_properties($s_page_id, $a_page) && page_update_structure($s_page_id, $a_page)))
                        set_message('error', __('ErrPagePropertiesNotWrite', $a_page['title'], $a_page['id']));
                }
                break;

            case 'save':
                if (array_key_exists('content', $a_page))
                {
                    if (!($b_is_success = $b_is_success && page_write_content($s_page_id, $a_page['content'], $b_is_draft)))
                        set_message('error', __('ErrPageContentNotWrite', $a_page['title'], $a_page['id']));
                }

                if (!($b_is_success = $b_is_success && page_update_structure($s_page_id, $a_page, $b_is_draft) && page_write_properties($s_page_id, $a_page, $b_is_draft)))
                    set_message('error', __('ErrPagePropertiesNotWrite', $a_page['title'], $a_page['id']));

                if (!$b_is_draft && page_is_cached($s_page_id))
                    set_message('info', __('MsgPageIsCached', $a_page['title']));
                break;

            case 'unlock':
                $a_page_properties = page_get_properties($s_page_id);
                $a_user = array('id' => get_user_id(), 'firstname' => get_user_property('firstname'), 'lastname' => get_user_property('lastname'));
                $b_is_success = page_write_properties($s_page_id, array_merge($a_page_properties, array('locked' => false)), $b_is_draft);
                break;
        }
    }

    if ($b_is_success && 'edit' !== $s_action) set_message('success', __('MsgOperationOk'));
    if (!$b_is_ajax)
    {
        if ('edit' === $s_action) return ;
        if ($b_is_draft && 'publish' !== $s_action) flash_messages_and_redirect('edit', 302, array('action' => 'edit', 'id' => $s_page_id, 'type' => 'draft'));
        else flash_messages_and_redirect('edit', 302, array('action' => 'edit', 'id' => $s_page_id));
    }
    else
    {
        $a_editable_page['is_draft'] = $b_is_draft;
        $a_misc = array();
        if ($a_tmp_editable_page = !$b_is_draft ? get_editable_page($s_page_id) : get_editable_page_draft($s_page_id))
        {
            $a_editable_page = array_merge($a_editable_page, $a_tmp_editable_page);
            include_once get_configuration('app/path/back').'external_files.php';

            $a_editable_page['id'] = $s_page_id;
            $a_editable_page['css_files'] = implode(';', $a_editable_page['css_files']);

            // css files
            $a_css_files = array();
            foreach (external_files_get_css() as $s_file)
                $a_css_files[] = str_replace(array(get_configuration('app/path/web'), '\\'), array('/', '/'), $s_file);
            $a_editable_page['css_files_list'] = implode(';', $a_css_files);

            $a_editable_page['js_files'] = implode(';', $a_editable_page['js_files']);

            // js files
            $a_js_files = array();
            foreach (external_files_get_js() as $s_file)
                $a_js_files[] = str_replace(array(get_configuration('app/path/web'), '\\'), array('/', '/'), $s_file);
            $a_editable_page['js_files_list'] = implode(';', $a_js_files);

            $a_misc['title'] = __('LblEditEditTitle', $a_editable_page['title']).($b_is_draft ? '('.__('LblEditDraftMode').')' : '');

            $a_misc['buttons'] = get_view('edit_buttons', array('a_page' => $a_editable_page));
            $a_misc['locked'] = (is_array($a_editable_page['locked']) ? __('LblEditLockedBy',  $a_editable_page['locked']['firstname'] .' '. $a_editable_page['locked']['lastname'].' (' .$a_editable_page['locked']['id'] . ')') : null);
            $a_misc['preview'] = ($b_is_draft ?
                                    '<a href="'.get_back_uri('page_preview', array('view' => $s_page_id, 'type' => 'draft')).'" title="'.__('LblEditRoutePreview').'" rel="external">'.__('LblEditRoutePreview').'</a>' :
                                    ((STRUCTURE_STATE_VISIBLE === intval($a_editable_page['state']) && (empty($a_editable_page['redirection']))) ?
                                        ((false !== ($s_url = get_front_uri($s_page_id))) ?
                                            '<a href="'.$s_url.'" title="'.__('LblEditRouteShow').'" rel="external">'.$s_url.'</a>' :
                                            __('LblEditNoRoute')
                                        ) :
                                        '<a href="'.get_back_uri('page_preview', array('view' => $s_page_id)).'" title="'.__('LblEditRoutePreview').'" rel="external">'.__('LblEditRoutePreview').'</a>'
                                    ));
        }
        delete_page_vars();
        set_page_var(array('page' => $a_editable_page, 'misc' => $a_misc));
    }
}

/**
 * Preview controller.
 *
 * @return void
 */
function page_preview_controller()
{
    if (!($s_page_id = get_request_parameter('view')))
    {
        set_message('error', 'ErrNoPage');
        flash_messages_and_redirect('home');
    }

    if ($b_is_draft = ($b_is_draft = get_request_parameter('type')) && ($b_is_draft) && (strtolower($b_is_draft) === 'draft'))
        $s_page_id .= '.draft';

    $s_page_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.php';
    $s_page_properties = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.properties.php';

    if (!($b_page_file = is_file($s_page_file)) || !($b_properties_file = is_file($s_page_properties)))
    {
        set_message('error', __('ErrLoadingPage') . ' ' . __('ErrFileNotRead', ($b_page_file ? $s_page_properties : $s_page_file)));
        flash_messages_and_redirect('home');
    }

    set_configuration('app/name', APP_FRONT);
    set_configuration('app/page', $s_page_id);

    $s_layout_file = get_configuration('app/path/layouts').APP_FRONT.DIRECTORY_SEPARATOR.get_page_property('layout', 'layout').'.php';

    /*if (is_file($s_layout_file))
        include $s_layout_file;
    else
        echo get_page_content();*/
    if (is_file($s_layout_file)) {
        ob_start();
        include $s_layout_file;
        echo fix_css_js(ob_get_clean());
    }
    else echo get_page_content();
    exit;
}

/**
 * Files controller.
 *
 * @return void
 */
function files_controller()
{
    $b_is_success = true;
    $b_is_ajax = (1 == get_request_parameter('ajax'));
    set_page_var('b_is_ajax', $b_is_ajax);

    if (!$s_action = get_request_parameter('action')) $s_action = 'list';
    if (in_array($s_action, array('delete', 'edit', 'modify')) && !($s_id = get_request_parameter('id'))) { $b_is_success = false; set_message('error', __('ErrRequestData', 'id')); }
    if (in_array($s_action, array('save', 'modify')) && 'POST' !== get_request_method()) { $b_is_success = false; set_message('error', __('ErrRequestPostOnly')); }
    if ('delete' === $s_action && !has_session_token()) {  $b_is_success = false; set_message('error', __('ErrSecurity')); }
    if (in_array($s_action, array('delete', 'delete_dir', 'edit', 'save', 'modify')) && !($s_type = get_request_parameter('type'))) { $b_is_success = false; set_message('error', __('ErrRequestData', 'type')); }

    if ($b_is_success  && 'list' !== $s_action)
    {
        include get_configuration('app/path/back').'files.php';
        include get_configuration('app/path/back').'urlize.php';

        switch ($s_action)
        {
            case 'save':
                $a_files = array_key_exists('files', $_FILES) ? $_FILES['files'] : array();
                $a_parameters = get_request_parameter('files');
                if (false !== ($b_is_success = (UPLOAD_ERR_OK === $a_files['error']['file'])))
                {
                    if (false === ($b_is_success = call_user_func('files_new_'.$s_type, array('name' => $a_files['name']['file'], 'tmp_name' => $a_files['tmp_name']['file']), $a_parameters)))
                        set_message('error', __('ErrFileNotWrite', 'images'));
                }
                else if ('' !== trim($a_parameters['new_dir']))
                    $b_is_success = files_new_dir($s_type, $a_parameters['dir'], $a_parameters['new_dir']);
                else
                    set_message('error', __('ErrUpload'.$a_files['error']['file']));
                break;

            case 'edit':
                $a_properties = get_data($s_type.'s', array());
                if (!($b_is_success = array_key_exists($s_id, $a_properties)))
                    set_message('error', __('ErrFilesFileNotFound', $s_id));
                else
                    set_page_var('a_edit', array_merge($a_properties[$s_id], array('type' => $s_type, 'name' => $s_id)));
                break;

            case 'delete_dir':
                $a_parameters = get_request_parameter('files');
                $b_is_success = files_remove_dir($s_type, $a_parameters['dir']);
                break;

            case 'delete':
                $b_is_success = call_user_func('file_'.$s_action.'_'.$s_type, $s_id);
                break;
            case 'modify':
                $b_is_success = call_user_func('file_'.$s_action.'_'.$s_type, $s_id, get_request_parameter('file'));
                break;
        }
    }

    if (!in_array($s_action, array('edit', 'list')) && $b_is_success) set_message('success', __('MsgOperationOk'));

    if ($b_is_ajax)
    {
        switch ($s_action)
        {
            case 'modify':
                $a_properties = get_data($s_type.'s', array());
                set_page_var('a_edit', array_merge($a_properties[$s_id], array('type' => $s_type, 'name' => $s_id)));

            case 'edit':
                set_page_var(array('html' => get_page_content()));
                delete_page_var('a_edit');
                break;

            case 'list':
                include get_configuration('app/path/back').'external_files.php';
                $s_type = get_request_parameter('type');
                $a_list_dir = get_request_parameter($s_type.'_list_dir', '');
                set_page_var('a_files', array($s_type => call_user_func('external_files_get_' . $s_type, $a_list_dir)));
                send_ajax(array(
                    'html' => get_view('files_' . $s_type)
                ));
                break;
        }
        return true;
    }

    if ('modify' === $s_action) flash_messages_and_redirect('files', 302, array_merge(array('action' => 'edit', 'id' => $s_id, 'type' => $s_type), ($b_is_ajax ? array('ajax' => 1) : array())));

    // list
    if ('edit' !== $s_action || !$b_is_success)
    {

        include get_configuration('app/path/back').'external_files.php';

        $a_list_dirs = array('documents' => get_request_parameter('documents_list_dir', get_flash_var('files/documents_list_dir', '')),
                             'images' => get_request_parameter('images_list_dir', get_flash_var('files/images_list_dir', '')));

        $a_files = array('documents' =>  external_files_get_documents($a_list_dirs['documents']), 'images' => external_files_get_images($a_list_dirs['images']));

        $a_new_files = array('new_document' => array('title' => '', 'dir' => '', 'new_dir' => ''), 'new_image' => array('title' => '', 'thumbnail' => 'on', 'dir' => '', 'new_dir' => ''));

        $a_dirs = array('documents' => array('' => ''), 'images' => array('' => ''));

        $s_doc_dir = get_configuration('app/path/web') . 'files' . DIRECTORY_SEPARATOR . 'documents' . DIRECTORY_SEPARATOR;
        foreach (external_files_get_documents_dirs() as $i_index => $s_dir)
        {
            $s_id = str_replace($s_doc_dir, '', $s_dir);
            $a_id = explode(DIRECTORY_SEPARATOR, $s_id);
            $s_name = '';
            for ($i = 0; $i < count($a_id); $i++) $s_name .= '&nbsp;&nbsp;';
            $s_name .= array_pop($a_id);

            $a_dirs['documents'][$s_id] = $s_name;
        }

        $s_img_dir = get_configuration('app/path/web') . 'files' . DIRECTORY_SEPARATOR . 'images' . DIRECTORY_SEPARATOR;
        foreach (external_files_get_images_dirs() as $i_index => $s_dir)
        {
            $s_id = str_replace($s_img_dir, '', $s_dir);
            $a_id = explode(DIRECTORY_SEPARATOR, $s_id);
            $s_name = '';
            for ($i = 0; $i < count($a_id); $i++) $s_name .= '&nbsp;&nbsp;';
            $s_name .= array_pop($a_id);

            $a_dirs['images'][$s_id] = $s_name;
        }

        set_page_var(array(
            'a_display_dir' => $a_list_dirs,
            'a_new_files' => $a_new_files,
            'a_files' => $a_files,
            'a_dirs' => $a_dirs,
            'a_list_dirs' => $a_list_dirs
        ));
    }

    //if (in_array($s_action, array('save', 'delete_dir'))) flash_messages_and_redirect('files', 302, array('type' => $s_type, 'dir' => isset($a_parameters['dir']) ? $a_parameters['dir'] : ''));
    if ('save' === $s_action) {
        set_flash_var('files/' . $s_type.'s_list_dir', trim((isset($a_parameters['dir']) ? $a_parameters['dir'] : '') .'/'.(isset($a_parameters['new_dir']) ? $a_parameters['new_dir'] : ''), '/'));
        flash_messages_and_redirect('files');
    }
    if ('delete_dir' === $s_action) {
        flash_messages_and_redirect('files');
    }

    load_helpers('defaults');
    $a_filesize = octets_to_multiples(get_configuration('server/filesize', 3000000));
    set_message('info', __('LblFilesServerMaxFileSize', $a_filesize[0], $a_filesize[1]));
}

/**
 * Modules controller.
 *
 * @return void
 */
function modules_controller()
{
    $b_is_success = true;
    $b_is_ajax = (1 == get_request_parameter('ajax'));

    // sub dispatch
    $a_page_path = get_configuration('app/page-path', array());
    $b_dispatch = (count($a_page_path) >= 1);

    include get_configuration('app/path/back').'modules.php';

    $a_modules_installed = modules_get_installed_modules();
    $a_modules_availables = modules_get_availables_modules($a_modules_installed);

    if ($b_dispatch)
        return modules_dispatch($a_page_path, $a_modules_installed, $a_modules_availables);

    if ($s_action = get_request_parameter('action'))
    {
        // all actions are availables for admin user only
        if (USER_ADMIN != get_user_property('rights')) { $b_is_success = false; set_message('error', __('ErrUserRightsAction')); }
        if (in_array($s_action, array('install', 'uninstall', 'up', 'down')) && !has_session_token()) {  $b_is_success = false; set_message('error', __('ErrSecurity')); }
        if (in_array($s_action, array('install', 'uninstall', 'up', 'down')) && !($s_id = get_request_parameter('id'))) { $b_is_success = false; set_message('error', __('ErrRequestData', 'id')); }
    }

    if ($b_is_success)
    {
        switch ($s_action)
        {
            case 'install':
                $b_is_success = module_install($s_id);
                break;

            case 'uninstall':
                $b_is_success = module_uninstall($s_id);
                break;

            case 'up':
                $b_is_success = module_up($s_id);
                break;

            case 'down':
                $b_is_success = module_down($s_id);
                break;
        }

        if ($b_is_success && $s_action) set_message('success', __('MsgOperationOk'));

        if ($b_is_ajax)
        {
            $a_modules_installed = modules_get_installed_modules();
            $a_modules_availables = modules_get_availables_modules($a_modules_installed);

            set_page_var(array(
                'html_modules_installed' => get_view('modules_installed', array('a_modules_installed' => $a_modules_installed)),
                'html_modules_availables' => get_view('modules_availables', array('a_modules_availables' => $a_modules_availables)),
                'module_installed' => (count($a_modules_installed) ? array_keys($a_modules_installed) : array())));
            return true;
        }

        if ($s_action)
            flash_messages_and_redirect('modules');

    }
    set_page_var('a_modules_installed', $a_modules_installed);
    set_page_var('a_modules_availables', $a_modules_availables);
}

/**
 * Shows data from CMS in xml format.
 *
 * @return void
 */
function data_controller()
{
    $a_types = explode(';', get_request_parameter('type'));
    $b_full_path = get_request_parameter('fullpath', false);
    $b_has_type = count($a_types) > 1;
    $a_return = array('data' => null);

    include get_configuration('app/path/back').'external_files.php';

    foreach ($a_types as $s_type)
    {
        $a_return_temp = array();
        switch($s_type)
        {
            case 'pages':
                /*foreach (get_data('front_routes') as $s_route => $a_values) {
                    $a_return_temp[] = array('level' => (count(explode('/', $s_route))) - 1, 'title' => $a_values['title'], 'id' => $a_values['id']);
                }*/
                $a_return_temp = external_files_get_routes();
                break;

            case 'documents':
            case 'images':
                $s_protocol = get_configuration('request/protocol');
                $s_ressources_path = get_configuration('server/base_uri_ressources_path');
                foreach (get_data($s_type) as $s_name => $a_values) {
                    $a_return_temp[] = array('title' => $a_values['title'],
                                             'name' => $s_name,
                                             'url' => ($b_full_path ? $s_protocol.'://'.$s_ressources_path : '').$a_values['url'],
                                             'thumbnail_url' => array_key_exists('thumbnail_url', $a_values) && !empty($a_values['thumbnail_url']) ? ($b_full_path ? $s_protocol.'://'.$s_ressources_path : '').$a_values['thumbnail_url'] : '',
                                             'dir' => rtrim(str_replace(array(DIRECTORY_SEPARATOR, '/files/'.$s_type, basename($a_values['url'])), array('/', '', ''), $a_values['url']), '/')
                                            );
                }
                break;

            case 'css':
            case 'js':
                $s_protocol = get_configuration('request/protocol');
                $s_base_url = get_configuration('server/base_url');
                $s_web_dir = get_configuration('app/path/web');
                foreach (call_user_func('external_files_get_'.$s_type) as $s_file) {
                    $a_return_temp[] = sprintf('{ "filename": "%s", "url": "%s" }', str_replace('\\', '/', basename($s_file)), ($b_full_path ? $s_protocol.'://'.$s_base_url : '').str_replace(array($s_web_dir, '\\'), array('/', '/'), $s_file));
                }
                break;

            case 'layout':
                foreach (external_files_get_layout() as $s_file) {
                    $a_return_temp[] = sprintf('{ "name": "%s" }', basename($s_file, '.php'));
                }
                break;

            case 'ref':
                $s_page = get_request_parameter('ref-page');
                if (is_file($s_file = get_configuration('app/path/contents') . APP_FRONT . DIRECTORY_SEPARATOR . $s_page . '.php')) {
                    $a_tags = array();
                    if ($i_tags = preg_match_all('@<(\w+)(\s[^>]+)?\s*/?>@xsi', file_get_contents($s_file), $a_tags)) {
                        for ($i_tag = 0; $i_tag < $i_tags; $i_tag++) {
                            if ('' !== $a_tags[2][$i_tag]) {
                                $a_attributes = array();
                                if (preg_match('@id\s*=\s*((\"(.*?)\")|(\'(.*?)\'))(?:\s+|$)@xsi', $a_tags[2][$i_tag], $a_attributes) && count($a_attributes))
                                    $a_return_temp[] = array('id' => $a_attributes[3], 'title' => $a_attributes['3']);
                            }
                        }
                    }
                }
                break;

            /*case 'templates':
                $a_config = array('templates' => array());
                $a_templates_config = bfglob(get_configuration('app/path/base').'modules'.DIRECTORY_SEPARATOR, 'templates.config.php', 2, -1);
                foreach ($a_templates_config as $s_template_config)
                {
                    $a_conf = include $s_template_config;
                    $a_conf['path'] = dirname($s_template_config);
                    $s_module_name = basename($a_conf['path']);
                    if (!array_key_exists('icon', $a_conf)) $a_conf['icon'] = null;
                    if (null != $a_conf['icon']) $a_conf['icon'] = 'http://'.get_configuration('server/base_uri_ressources_path').'/images/modules/'.$s_module_name.'/'.$a_conf['icon'];
                    $a_config['templates'][$s_module_name] = $a_conf;
                }
                send_ajax($a_config);
                exit;*/
        }

        $a_return['data'][$s_type] = $a_return_temp;
        //else $a_return['data'] = $a_return_temp;
    }

    send_ajax($a_return);
}

/**
 * Makes a redirection to home page with message if user not have admin rights.
 *
 * @return void
 */
function redirect_if_not_admin()
{
    if (USER_ADMIN !== get_user_property('rights'))
    {
        set_message('error', __('ErrUserAccess'));
        flash_messages_and_redirect('home');
    }
}