<?php
/*
 * Copyright 2012 Sébastien Raud
 *
 * This file is part of beCms.
 *
 * beCms is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * beCms is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with beCms.  If not, see <http://www.gnu.org/licenses/>.
 */
/*
 * @todo files_control_values() tester unicité nom de fichier
 * @todo sauvegarde dans tableau
 */

/**
 * Creates a new document.
 *
 * @param  array  $a_files       List of documents.
 * @param  array  $a_properties  New document properties.
 * @return boolean
 */
function files_new_document($a_files, $a_properties)
{
    $a_properties['type'] = 'documents';
    if (!files_control_values($a_files, $a_properties, true))
        return false;

    if ('' !== $a_properties['new_dir'] && !files_new_document_dir($a_properties['dir'], $a_properties['new_dir'])) return false;
    $s_sub_dir = ('' !== $a_properties['dir'] ? $a_properties['dir'] . DIRECTORY_SEPARATOR : '') . ('' !== $a_properties['new_dir'] ? $a_properties['new_dir'] . DIRECTORY_SEPARATOR : '');

    $s_file_path = get_configuration('app/path/web').'files'.DIRECTORY_SEPARATOR.'documents'.DIRECTORY_SEPARATOR.$s_sub_dir.$a_properties['name'];

    if (!file_save_uplaoded($a_files['tmp_name'], $s_file_path, $a_files['name']))
        return false;

    $a_files = get_data('documents', array());
    $a_files[md5($s_file_path)] = array('title' => $a_properties['title'], 'url' => '/files/documents/'.str_replace(DIRECTORY_SEPARATOR, '/', $s_sub_dir).$a_properties['name']);

    uasort($a_files, create_function('$a, $b', 'return strnatcasecmp($a[\'title\'], $b[\'title\']);'));
    return write_data($a_files, 'documents');
}

/**
 * Creates a new image.
 *
 * @param  array  $a_files       List of images.
 * @param  array  $a_properties  New image properties.
 * @return boolean
 */
function files_new_image($a_files, $a_properties)
{
    $a_properties['type'] = 'images';
    if (!files_control_values($a_files, $a_properties, true))
        return false;

    if ('' !== $a_properties['new_dir'] && !files_new_image_dir($a_properties['dir'], $a_properties['new_dir'])) return false;
    $s_sub_dir = ('' !== $a_properties['dir'] ? $a_properties['dir'] . DIRECTORY_SEPARATOR : '') . ('' !== $a_properties['new_dir'] ? $a_properties['new_dir'] . DIRECTORY_SEPARATOR : '');

    $s_file_path = get_configuration('app/path/web').'files'.DIRECTORY_SEPARATOR.'images'.DIRECTORY_SEPARATOR.$s_sub_dir.$a_properties['name'];
    $s_thumbnail_path = get_configuration('app/path/web').'files'.DIRECTORY_SEPARATOR.'images'.DIRECTORY_SEPARATOR.'thumbnails'.DIRECTORY_SEPARATOR.$s_sub_dir.$a_properties['name'];

    if (!file_save_uplaoded($a_files['tmp_name'], $s_file_path, $a_files['name']))
        return false;

    if (($b_has_thumbnail = (array_key_exists('thumbnail', $a_properties) && 'on' === $a_properties['thumbnail'] && function_exists('imagecreatefromjpeg'))))
        make_thumbnail($s_file_path, $s_thumbnail_path, $a_properties['thumb_width'], $a_properties['thumb_height']);

    $a_files = get_data('images', array());
    $a_files[md5($s_file_path)] = array(
        'url' => '/files/images/'.str_replace(DIRECTORY_SEPARATOR, '/', $s_sub_dir).$a_properties['name'],
        'title' => $a_properties['title'],
        'thumbnail_url'  => ($b_has_thumbnail ? '/files/images/thumbnails/'.str_replace(DIRECTORY_SEPARATOR, '/', $s_sub_dir).$a_properties['name'] : ''));

    uasort($a_files, create_function('$a, $b', 'return strnatcasecmp($a[\'title\'], $b[\'title\']);'));
    return write_data($a_files, 'images');
}

/**
 * Updates a document.
 *
 * @param  string  $s_name        Name of document.
 * @param  array   $a_properties  Properties of document.
 * @return boolean
 */
function file_modify_document($s_name, $a_properties)
{
    $a_files = get_data('documents', array());

    if (!array_key_exists($s_name, $a_files))
        return false;

    $a_document = array_merge($a_files[$s_name], $a_properties);

    if (!files_control_values(array_merge($a_document, array('name' => basename($a_document['url']))), $a_document))
        return false;

    $a_files[$s_name] = $a_document;
    ksort($a_files);
    return write_data($a_files, 'documents');
}

/**
 * Removes a document.
 *
 * @param  string  $s_name        Name of document.
 * @return boolean
 */
function file_delete_document($s_name)
{
    $a_files = get_data('documents', array());

    if (!array_key_exists($s_name, $a_files))
        return false;

    $s_path = get_configuration('app/path/web').str_replace('/', DIRECTORY_SEPARATOR, $a_files[$s_name]['url']);
    if (is_file($s_path)) {
        if (!unlink($s_path))
            set_message('error', __('ErrFileDelete', $s_path));
    }
    unset($a_files[$s_name]);
    ksort($a_files);
    return write_data($a_files, 'documents');
}

/**
 * Updates an image.
 *
 * @param  string  $s_name        Name of image.
 * @param  array   $a_properties  Properties of image.
 * @return boolean
 */
function file_modify_image($s_name, $a_properties)
{
    $a_files = get_data('images', array());

    if (!array_key_exists($s_name, $a_files))
        return false;

    $a_image = array_merge($a_files[$s_name], $a_properties);

    if (!files_control_values(array_merge($a_image, array('name' => basename($a_image['url']))), $a_image))
        return false;

    $a_files[$s_name] = $a_image;
    ksort($a_files);
    return write_data($a_files, 'images');
}

/**
 * Removes an image.
 *
 * @param  string  $s_name        Name of image.
 * @return boolean
 */
function file_delete_image($s_name)
{
    $a_files = get_data('images', array());

    if (!array_key_exists($s_name, $a_files))
        return false;

    $s_path = get_configuration('app/path/web').str_replace('/', DIRECTORY_SEPARATOR, $a_files[$s_name]['url']);

    if (is_file($s_path)) {
        if (!unlink($s_path))
            set_message('error', __('ErrFileDelete', $s_path));
    }

    $s_thumb_path = get_configuration('app/path/web').str_replace('/', DIRECTORY_SEPARATOR, $a_files[$s_name]['thumbnail_url']);
    if (is_file($s_thumb_path)) {
        if (!unlink($s_thumb_path))
            set_message('error', __('ErrFileDelete', $s_thumb_path));
    }

    unset($a_files[$s_name]);
    ksort($a_files);
    return write_data($a_files, 'images');
}

/**
 * Saves an uploaded file.
 *
 * @param  string  $s_tmp_name   Name in uploaded files.
 * @param  string  $s_path_to    Path to save.
 * @param  string  $s_real_name  Name to save.
 * @return boolean
 */
function file_save_uplaoded($s_tmp_name, $s_path_to, $s_real_name)
{
    if (!move_uploaded_file($s_tmp_name, $s_path_to))
    {
        set_message('error', __('ErrFileNotWrite', $s_real_name));
        return false;
    }
    chmod($s_path_to, 0666);

    return true;
}

/**
 * Check data files.
 *
 * @param  array    $a_files       Files lists.
 * @param  array    $a_properties  File properties.
 * @param  boolean  $b_urlize      Indicates if the name of file must be urlized.
 * @return boolean
 */
function files_control_values($a_files, & $a_properties, $b_urlize = false)
{
    $a_errors = array();

    if (!array_key_exists('title', $a_properties) || !strlen($a_properties['title']))
        $a_errors[] = __('MsgRequiredFileTitle', $a_files['name']);

    $s_extension = (false !== ($i_pos = strrpos($a_files['name'], '.'))) ? substr($a_files['name'], $i_pos + 1) : null;
    if (!$s_extension || !strlen($s_extension))
        $a_errors[] = __('MsgCheckFileExtension', $a_files['name']);
    else if ('images' === $a_properties['type'] && !in_array(strtolower($s_extension), array('png', 'gif', 'jpg', 'jpeg', 'jpe')))
        $a_errors[] = __('ErrCheckFileImageType');

    if (array_key_exists('thumbnail', $a_properties)) {
        $a_properties['thumb_width'] = trim($a_properties['thumb_width']);
        $a_properties['thumb_height'] = trim($a_properties['thumb_height']);

        if (!preg_match('/^[\d]+$/', $a_properties['thumb_width']) && !empty($a_properties['thumb_width']))
            $a_errors[] = __('MsgCheckInteger', __('LblThumbsWidth'));

        if (!preg_match('/^[\d]+$/', $a_properties['thumb_height']) && !empty($a_properties['thumb_height']))
            $a_errors[] = __('MsgCheckInteger', __('LblThumbsHeight'));

        if (empty($a_properties['thumb_width']) && empty($a_properties['thumb_height']))
        {
            $a_properties['thumb_width'] = get_configuration('misc/images/thumb_width', '100');
            $a_properties['thumb_height'] = get_configuration('misc/images/thumb_height', '100');
        }
    }

    $a_properties['new_dir'] = isset($a_properties['new_dir']) ? trim($a_properties['new_dir']) : '';
    if ('' !== $a_properties['new_dir']) $a_properties['new_dir'] = urlize($a_properties['new_dir']);

    if ($b_urlize) {
        $a_properties['name'] = strtolower(urlize(str_replace(
            array('%title%', '%date%', '%random%'),
            array(urlize($a_properties['title']), date('YmdHis'), generate_id()),
            get_configuration('misc/'.$a_properties['type'].'/name_format', '%title%-%date%'))).'.'.$s_extension);
    }

    if (count($a_errors)) {
        set_message('error', $a_errors);
        return false;
    }

    return true;
}

/**
 * Creates a new directory for documents or images.
 *
 * @param  string  $s_type     document / image.
 * @param  string  $s_dir      Parent directory.
 * @param  string  $s_new_dir  New directory name.
 * @return boolean
 */
function files_new_dir($s_type, $s_dir, $s_new_dir)
{
    if ('' === trim($s_new_dir)) return false;
    return ('image' === $s_type) ? files_new_image_dir($s_dir, $s_new_dir) : files_new_document_dir($s_dir, $s_new_dir);
}

/**
 * Creates a new directory for documents.
 *
 * @param  string  $s_dir      Parent directory.
 * @param  string  $s_new_dir  New directory name.
 * @return boolean
 */
function files_new_document_dir($s_dir, $s_new_dir)
{
    $s_base_dir = get_configuration('app/path/web').'files'.DIRECTORY_SEPARATOR.'documents'.DIRECTORY_SEPARATOR;
    $s_new_dir = urlize($s_new_dir);

    $s_mk_dir = $s_base_dir . ('' !== $s_dir ? $s_dir . DIRECTORY_SEPARATOR : '') . $s_new_dir;

    if (is_dir($s_mk_dir))
        return true;

    return make_dir($s_mk_dir);
}

/**
 * Creates a new directory for images.
 *
 * @param  string  $s_dir      Parent directory.
 * @param  string  $s_new_dir  New directory name.
 * @return boolean
 */
function files_new_image_dir($s_dir, $s_new_dir)
{
    $s_base_dir = get_configuration('app/path/web').'files'.DIRECTORY_SEPARATOR.'images'.DIRECTORY_SEPARATOR;
    $s_new_dir = urlize($s_new_dir);

    $s_mk_dir = $s_base_dir . ('' !== $s_dir ? $s_dir . DIRECTORY_SEPARATOR : '') . $s_new_dir;
    $s_mk_thumb_dir = $s_base_dir . 'thumbnails' . DIRECTORY_SEPARATOR . ('' !== $s_dir ?  $s_dir . DIRECTORY_SEPARATOR : '') . $s_new_dir;

    $b_is_success = false;

    if (!is_dir($s_mk_dir) && !make_dir($s_mk_dir))
        return false;

    if (!is_dir($s_mk_thumb_dir) && !make_dir($s_mk_thumb_dir))
    {
        rmdir($s_mk_dir);
        return false;
    }

    return true;
}

/**
 * Removes a directory for documents or images.
 *
 * @param  string  $s_type     document / image.
 * @param  string  $s_dir      Parent directory.
 * @return boolean
 */
function files_remove_dir($s_type, $s_dir)
{
    if ('' === $s_dir) {
        set_message('error', __('ErrDirDelete', $s_type . 's'));
        return false; // not delete the base directory
    }

    return ('image' === $s_type) ? files_remove_image_dir($s_dir) : files_remove_document_dir($s_dir);
}

/**
 * Removes a directory for documents.
 *
 * @param  string  $s_dir      Parent directory.
 * @return boolean
 */
function files_remove_document_dir($s_dir)
{
    if ('' === $s_dir) return false; // not delete the base directory

    $s_dir = get_configuration('app/path/web').'files'.DIRECTORY_SEPARATOR.'documents'.DIRECTORY_SEPARATOR.$s_dir;

    $s_web_dir = get_configuration('app/path/web');
    $a_deleted_files = array();

    if (!is_dir($s_dir)) return true;

    foreach (bfglob($s_dir, '*', 2, -1) as $s_file)
    {
        if (!unlink($s_file))
        {
            set_message('error', __('ErrFileDelete', $s_file));
            files_remove_documents_files($a_deleted_files);
            return false;
        }
        $a_deleted_files[] = '/' . str_replace($s_web_dir, '', $s_file);
    }

    files_remove_documents_files($a_deleted_files);

    foreach (bfglob($s_dir, '*', 1, -1) as $s_sub_dir)
    {
        if (!rmdir($s_sub_dir))
        {
            set_message('error', __('ErrDirDelete', $s_sub_dir));
            return false;
        }
    }

    if (!rmdir($s_dir))
    {
        set_message('error', __('ErrDirDelete', $s_sub_dir));
        return false;
    }

    return true;
}

/**
 * Removes documents files when a document directory is removed.
 *
 * @param  array  $a_files  Files to remove.
 * @return void
 */
function files_remove_documents_files($a_files)
{
    $a_documents = get_data('documents');
    $a_delete = array();

    foreach ($a_documents as $s_key => $a_properties) {
        if (in_array($a_properties['url'], $a_files))
            $a_delete[] = $s_key;
    }

    foreach ($a_delete as $s_key)
        unset($a_documents[$s_key]);

    ksort($a_documents);
    return write_data($a_documents, 'documents');
}

/**
 * Removes a directory for images.
 *
 * @param  string  $s_dir      Parent directory.
 * @return boolean
 */
function files_remove_image_dir($s_dir)
{
    if ('' === $s_dir) return false; // not delete the base directory

    $s_img_dir = get_configuration('app/path/web').'files'.DIRECTORY_SEPARATOR.'images'.DIRECTORY_SEPARATOR.$s_dir;
    $s_thumb_dir = get_configuration('app/path/web').'files'.DIRECTORY_SEPARATOR.'images'.DIRECTORY_SEPARATOR.'thumbnails'.DIRECTORY_SEPARATOR.$s_dir;

    if (!is_dir($s_img_dir) && !is_dir($s_thumb_dir)) return true;

    $s_web_dir = get_configuration('app/path/web');
    $a_deleted_files = array();

    foreach (array($s_img_dir, $s_thumb_dir) as $i_index => $s_dir)
    {
        foreach (bfglob($s_dir, '*', 2, -1) as $s_file)
        {
            if (!unlink($s_file))
            {
                set_message('error', __('ErrFileDelete', $s_file));
                if (!$i_index) files_remove_images_files($a_deleted_files);
                return false;
            }
            if (!$i_index) {
                $a_deleted_files[] = '/' . str_replace($s_web_dir, '', $s_file);
            }
        }

        if (!$i_index) files_remove_images_files($a_deleted_files);

        foreach (bfglob($s_dir, '*', 1, -1) as $s_sub_dir)
        {
            if (!rmdir($s_sub_dir))
            {
                set_message('error', __('ErrDirDelete', $s_sub_dir));
                return false;
            }
        }

        if (!rmdir($s_dir))
        {
            set_message('error', __('ErrDirDelete', $s_sub_dir));
            return false;
        }
    }

    return true;
}

/**
 * Removes images files when an image directory is removed.
 *
 * @param  array  $a_files  Files to remove.
 * @return void
 */
function files_remove_images_files($a_files)
{
    $a_images = get_data('images');
    $a_delete = array();

    foreach ($a_images as $s_key => $a_properties) {
        if (in_array($a_properties['url'], $a_files))
            $a_delete[] = $s_key;
    }

    foreach ($a_delete as $s_key)
        unset($a_images[$s_key]);

    ksort($a_images);
    return write_data($a_images, 'images');
}

/**
 * Generates a thumbnail.
 *
 * @param  string   $s_file_path   Original image path.
 * @param  string   $s_thumb_path  Thumbnail path.
 * @param  integer  $i_width       Thumbnail width.
 * @param  integer  $i_height      Thumbnail height.
 * @return boolean
 */
function make_thumbnail($s_file_path, $s_thumb_path, $i_width, $i_height)
{
    $a_thumb_size = array($i_width, $i_height);
    $a_src_info = getimagesize($s_file_path);
    $s_save_function = null;

    switch ($a_src_info[2])
    {
        case IMAGETYPE_PNG:
            $r_src_img = imagecreatefrompng($s_file_path);
            $s_save_function = 'imagepng';
            break;

        case IMAGETYPE_GIF:
            $r_src_img = imagecreatefromgif($s_file_path);
            $s_save_function = 'imagegif';
            break;

        case IMAGETYPE_JPEG:
        case IMAGETYPE_JPEG2000:
            $r_src_img = imagecreatefromjpeg($s_file_path);
            $s_save_function = 'imagejpeg';
            break;

        default:
            return false;
    }

    $i_src_width = $a_src_info[0];
    $i_src_height = $a_src_info[1];
    $i_ratio = $i_src_width / $i_src_height;

    if (empty($a_thumb_size[0])) $a_thumb_size[0] = ($i_ratio > 0 ? $a_thumb_size[1] * $i_ratio : $a_thumb_size[1] / $i_ratio);
    if (empty($a_thumb_size[1])) $a_thumb_size[1] = ($i_ratio < 0 ? $a_thumb_size[0] * $i_ratio : $a_thumb_size[0] / $i_ratio);

    if (($i_src_width / $a_thumb_size[0]) >= ($i_src_height / $a_thumb_size[1]))
    {
        $i_thumb_width = min($i_src_width, $a_thumb_size[0]);
        $i_thumb_height = (int) ($i_thumb_width / $i_ratio);
    }
    else
    {
        $i_thumb_height = min($i_src_height, $a_thumb_size[1]);
        $i_thumb_width = (int) ($i_thumb_height * $i_ratio);
    }

    $r_thumb_img = imagecreatetruecolor($i_thumb_width, $i_thumb_height);

    if ($a_src_info[2] === IMAGETYPE_PNG || $a_src_info[2] === IMAGETYPE_GIF)
    {
        imagealphablending($r_thumb_img, false);
        imagesavealpha($r_thumb_img, true);

        imagefill($r_thumb_img, 0, 0, imagecolorallocatealpha($r_thumb_img, 255, 255, 255, 127));
    }

    imagecopyresampled($r_thumb_img, $r_src_img, 0, 0, 0, 0, $i_thumb_width, $i_thumb_height, $i_src_width, $i_src_height);
    $s_save_function($r_thumb_img, $s_thumb_path);
    imagedestroy($r_thumb_img);

    $i_mode = octdec(get_configuration('server/filemode'));
    $i_current_umask = umask();
    umask(0000);
    @chmod($s_thumb_path, $i_mode);
    @umask($i_current_umask);

    return true;
}
