<?php
/*
 * Copyright 2012 Sébastien Raud
 *
 * This file is part of beCms.
 *
 * beCms is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * beCms is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with beCms.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * Dispatch module.
 *
 * @param  array  $a_page_path           Page path.
 * @param  array  $a_modules_installed   Installed modules.
 * @param  array  $a_modules_availables  Availables modules.
 * @return void
 */
function modules_dispatch($a_page_path, $a_modules_installed, $a_modules_availables)
{
    $b_is_success = true;
    $b_is_ajax = 1 == get_request_parameter('ajax');

    $s_module_key = array_shift($a_page_path);

    if (!array_key_exists($s_module_key, $a_modules_installed)) set_message('error', __('ErrModuleNotInstalled', $s_module_key));
    else
    {
        $s_file = null;
        $s_function = null;

        $s_module_key_code = str_replace('-', '_', $s_module_key);
        $i_dispatchers = intval($a_modules_installed[$s_module_key]['dispatchers']);

        // dispatchers exists ?
        if (0 === $i_dispatchers) { $b_is_success = false; set_message('error', __('ErrPageNotFound')); }
        if (false !== ($b_is_admin_page = !empty($a_page_path) && 'admin' === $a_page_path[0]))
        {
            if (USER_ADMIN != get_user_property('rights')) { $b_is_success = false; set_message('error', __('ErrUserAccess')); }
            if (!(1 === (1 & $i_dispatchers))) { $b_is_success = false; set_message('error', __('ErrPageNotFound')); }
        }
        else if (!(2 === (2 & $i_dispatchers))) { $b_is_success = false; set_message('error', __('ErrPageNotFound')); }

        if (!$b_is_success) flash_messages_and_redirect('modules');

        set_configuration('module/key', $s_module_key);
        set_page_var(array(
            'module_key' => $s_module_key,
            'module_is_admin' => $b_is_admin_page,
            'module_base_uri' => get_configuration('app/page') . '/' . implode('/', get_configuration('app/page-path'))));

        module_load_language($s_module_key, get_user_property('lang', 'fr'));

        if (!is_file($s_file = get_module_configuration($s_module_key, 'module/path/back').'core.php')) $b_is_success = false;
        else {
            include $s_file;
            if (!function_exists($s_function = $s_module_key_code.'_back_'.($b_is_admin_page ? 'admin_' : '') .'dispatcher')) $b_is_success = false;
            else $s_function($a_page_path);
        }
        if (!$b_is_success) set_message('error', __('ErrPageNotFound'));
    }

    if ($b_is_ajax) {
        return true;
    }
}

/**
 * Returns a list of availables modules.
 *
 * @param  array  $a_installed_modules  List of installed modules, can be use to returns only non-installed modules.
 * @return array  [key] => array('path', 'config')
 */
function modules_get_availables_modules($a_installed_modules = null)
{
    $a_modules = array();

    if (!$a_installed_modules) $a_installed_modules = modules_get_installed_modules();

    foreach (bfglob(get_configuration('app/path/modules'), '*', 1) as $s_path)
    {
        $s_module_key = basename($s_path);
        if (preg_match('~^[a-zA-Z0-9_\-]+$~', $s_module_key) && !array_key_exists($s_module_key, $a_installed_modules))
            $a_modules[$s_module_key] = array('path' => $s_path, 'info' => modules_get_info($s_module_key));
    }

    return $a_modules;
}

/**
 * Returns a list of installed modules.
 *
 * @return array
 */
function modules_get_installed_modules()
{
    return get_data('modules', array());
}

/**
 * Returns informations about a module.
 *
 * @param  string  $s_module_key  The module key.
 * @return array
 */
function modules_get_info($s_module_key)
{
    return array_deep_merge(
                array('version' => '', 'title' => '', 'description' => ''),
                get_module_data($s_module_key, 'info', array()),
                get_module_data($s_module_key, 'info-'.get_user_property('lang', 'fr'), array())
            );
}

/**
 * Installs a module.
 *
 * @param  string  $s_module_key  The module key.
 * @return boolean
 */
function module_install($s_module_key)
{
    // installation :
    //  - copies des fichiers (module/web) vers les bons répertoires ;
    //  - ajout de la conf du module dans get_configuration('app/path/data').'modules.php' ;
    //  - lectures et enregistrement des callbacks.

    if (USER_ADMIN != get_user_property('rights'))
    {
        set_message('error', __('ErrUserRightsAction'));
        return false;
    }

    $a_errors = array();
    $a_module_conf = array('dispatchers' => 0);

    // module key must contains only letters, numbers, _ and -
    if (!preg_match('~^[a-zA-Z0-9_\-]+$~', $s_module_key))
        $a_errors[] = __('MsgCheckModuleKey', $s_module_key);

    if (!is_dir(get_configuration('app/path/modules').$s_module_key))
        $a_errors[] = __('MsgCheckModuleDir', get_configuration('app/path/modules').$s_module_key, $s_module_key);

    if (count($a_errors))
    {
        set_message('error', $a_errors);
        return false;
    }

    if (false === module_copy_ressources($s_module_key))
        return false;

    // search dispatchers
    $s_module_key_code = str_replace('-', '_', $s_module_key);
    if (is_file($s_file = get_module_configuration($s_module_key, 'module/path/back').'core.php'))
    {
        $a_dispatchers_functions = array();
        $s_content = file_get_contents($s_file);
        if (preg_match_all('~^\s*function\s*(' . $s_module_key_code.'_back_(admin_)?dispatcher)\s*\(([\$a-z0-9_\s,=\(\)"\'&]+)?\)\s*\{$~im', $s_content, $a_dispatchers_functions))
        {
            if (in_array($s_module_key_code.'_back_admin_dispatcher', $a_dispatchers_functions[1])) $a_module_conf['dispatchers'] += 1;
            if (in_array($s_module_key_code.'_back_dispatcher', $a_dispatchers_functions[1])) $a_module_conf['dispatchers'] += 2;
        }
    }
    if (is_file($s_file = get_module_configuration($s_module_key, 'app/path/modules').'core.php'))
    {
        $a_dispatchers_functions = array();
        $s_content = file_get_contents($s_file);
        if (preg_match_all('~^\s*function\s*(' . $s_module_key_code.'_front_dispatcher)\s*\(([\$a-z0-9_\s,=\(\)"\'&]+)?\)\s*\{$~im', $s_content, $a_dispatchers_functions))
        {
            if (in_array($s_module_key_code.'_front_dispatcher', $a_dispatchers_functions[1])) $a_module_conf['dispatchers'] += 4;
        }
    }

    // add module to modules conf
    $a_modules_conf = array_merge(modules_get_installed_modules(), array($s_module_key => $a_module_conf));

    write_data(array_merge(get_data('modules', array()), $a_modules_conf), 'modules');

    // chainables callbacks
    $a_chains_config = array('not-install' => array());
    if (is_file($s_file = get_module_configuration($s_module_key, 'module/path/install') . 'chain.php')) $a_chains_config = array_merge($a_chains_config, include $s_file);

    $a_chains = array('back' => array(), 'front' => array());

    foreach (array('back', 'front') as $s_app)
    {
        if (is_file($s_file = get_module_configuration($s_module_key, 'module/path/'.$s_app).'chain.php'))
        {
            $a_chains_functions = array();
            $s_content = file_get_contents($s_file);
            if (0 != ($i_count = preg_match_all('~^\s*function\s*(' . $s_module_key_code.'_chain_([a-z0-9\-_]+))\s*\(([\$a-z0-9_\s,=\(\)"\'&]+)?\)\s*\{$~im', $s_content, $a_chains_functions)))
            {
                for ($i = 0; $i < $i_count; $i++)
                {
                    if (!in_array($s_app.'/'.$a_chains_functions[1][$i], $a_chains_config['not-install']))
                        $a_chains[$s_app][] = array('base_function' => $a_chains_functions[2][$i], 'callback_function' => $a_chains_functions[1][$i]);
                }
            }
        }
    }

    if (count($a_chains))
    {
        // @todo : voir comment se passera la réorganisation
        write_module_data($s_module_key, $a_chains, 'chains');

        foreach ($a_chains as $s_app => $a_chains_functions)
        {
            foreach ($a_chains_functions as $i_index => $a_function)
            {
                $s_file = 'modules'.DIRECTORY_SEPARATOR.$s_app.DIRECTORY_SEPARATOR.'chains'.DIRECTORY_SEPARATOR.$a_function['base_function'];
                $a_data = get_data($s_file, array());
                $a_data[$s_module_key] = $a_function['callback_function'];

                if (!is_dir($s_dir = dirname(get_configuration('/app/path/data') . $s_file))) make_dir($s_dir);
                if (!write_data($a_data, $s_file))
                    echo 'erreur fichier ' . $s_file;
            }
        }
    }

    if (is_file($s_file = get_module_configuration($s_module_key, 'module/path/install') . 'install.php'))
    {
        include_once $s_file;
        if (function_exists($s_function = $s_module_key_code . '_install'))
            $s_function();
    }

    return true;
}

/**
 * Adds a module function to a chain list.
 *
 * @param  string  $s_function_name  Module function.
 * @param  string  $s_app            Application (APP_BACK / APP_FRONT).
 * @param  string  $s_module_key     Module key.
 * @return boolean
 */
function module_add_chain_function($s_function_name, $s_app = null, $s_module_key = null)
{
    $s_module_key = $s_module_key ? $s_module_key : get_configuration('module/key');

    $a_installed_modules = modules_get_installed_modules();

    if (!array_key_exists($s_module_key, $a_installed_modules))
    {
        set_message('error', __('ErrModuleNotInstalled', $s_module_key));
        return false;
    }

    $s_app = $s_app ? $s_app : get_configuration('app/name');

    $s_module_key_code = str_replace('-', '_', $s_module_key);

    //$a_chains = array_merge(array('back' => array(), 'front' => array()), get_module_data($s_module_key, 'chains', array('back' => array(), 'front' => array())));
    $a_chains = array('back' => array(), 'front' => array());

    if ($s_module_key_code.'_chain_' !== substr($s_function_name, 0, strlen($s_module_key_code) + 7))
        $a_chains = array('base_function' => $s_function_name, 'callback_function' => $s_module_key_code.'_chain_'.$s_function_name);
    else
        $a_chains = array('base_function' => substr($s_function_name, strlen($s_module_key_code) + 7), 'callback_function' => $s_function_name);

    // adds to module data
    $a_data_chains = get_module_data($s_module_key, 'chains', array('back' => array(), 'front' => array()));
    $a_data_chains[$s_app][] = $a_chains;
    write_module_data($s_module_key, $a_data_chains, 'chains');

    // adds to global chaining data
    $s_file = 'modules'.DIRECTORY_SEPARATOR.$s_app.DIRECTORY_SEPARATOR.'chains'.DIRECTORY_SEPARATOR.$a_chains['base_function'];
    $a_data = get_data($s_file, array());
    $a_data[$s_module_key] = $a_chains['callback_function'];
    array_reorder($a_data, array_keys($a_installed_modules));
    write_data($a_data, $s_file);
    return true;
}

/**
 * Removes a module function from a chain list.
 *
 * @param  string  $s_function_name  Module function.
 * @param  string  $s_app            Application (APP_BACK / APP_FRONT).
 * @param  string  $s_module_key     Module key.
 * @return boolean
 */
function module_remove_chain_function($s_function_name, $s_app = null, $s_module_key = null)
{
    $s_module_key = $s_module_key ? $s_module_key : get_configuration('module/key');

    $a_installed_modules = modules_get_installed_modules();

    if (!array_key_exists($s_module_key, $a_installed_modules))
    {
        set_message('error', __('ErrModuleNotInstalled', $s_module_key));
        return false;
    }

    $s_app = $s_app ? $s_app : get_configuration('app/name');

    $s_module_key_code = str_replace('-', '_', $s_module_key);

    $a_chains = array('back' => array(), 'front' => array());

    if ($s_module_key_code.'_chain_' !== substr($s_function_name, 0, strlen($s_module_key_code) + 7))
        $a_chains = array('base_function' => $s_function_name, 'callback_function' => $s_module_key_code.'_chain_'.$s_function_name);
    else
        $a_chains = array('base_function' => substr($s_function_name, strlen($s_module_key_code) + 7), 'callback_function' => $s_function_name);

    // removes to module data
    $a_data_chains = get_module_data($s_module_key, 'chains', array('back' => array(), 'front' => array()));
    foreach ($a_data_chains[$s_app] as $i_index => $a_function)
    {
        if ($a_function['base_function'] === $a_chains['base_function'])
        {
            unset($a_data_chains[$s_app][$i_index]);
            $a_data_chains[$s_app] = array_values($a_data_chains[$s_app]);
            break;
        }
    }
    write_module_data($s_module_key, $a_data_chains, 'chains');

    // removes to global chaining data
    $s_file = 'modules'.DIRECTORY_SEPARATOR.$s_app.DIRECTORY_SEPARATOR.'chains'.DIRECTORY_SEPARATOR.$a_chains['base_function'];
    $a_data = get_data($s_file, array());
    if (array_key_exists($s_module_key, $a_data)) unset($a_data[$s_module_key]);
    array_reorder($a_data, array_keys($a_installed_modules));
    write_data($a_data, $s_file);
    if (!count($a_data)) @unlink(get_configuration('app/path/data') . $s_file . '.php');
    return true;
}

/*
 * code from  ananax at yelostudio dot com (array_slice comment on php.net)
 */
function array_reorder(array & $a_reorder, array $a_with){
    $a_temp = array();

    foreach ($a_with as $i) {
        if (isset($a_reorder[$i])) {
            $m_tmp_value = array_slice($a_reorder, array_search($i,array_keys($a_reorder)), 1, true);
            $a_temp[$i] = array_shift($m_tmp_value);
            unset($a_reorder[$i]);
        }
    }
    $a_reorder = array_merge($a_temp, $a_reorder);
}

/**
 * Uninstalls a module.
 *
 * @param  string  $s_module_key  The module key.
 * @return boolean
 */
function module_uninstall($s_module_key)
{
    if (USER_ADMIN != get_user_property('rights'))
    {
        set_message('error', __('ErrUserRightsAction'));
        return false;
    }

    $a_errors = array();

    // removes chains
    if ($a_chains = get_module_data($s_module_key, 'chains'))
    {
        foreach ($a_chains as $s_app => $a_chains_functions)
        {
            foreach ($a_chains_functions as $i_index => $a_function)
            {
                $s_file = 'modules'.DIRECTORY_SEPARATOR.$s_app.DIRECTORY_SEPARATOR.'chains'.DIRECTORY_SEPARATOR.$a_function['base_function'];

                if ($a_data = get_data($s_file))
                {
                    if (array_key_exists($s_module_key, $a_data)) unset($a_data[$s_module_key]);
                    write_data($a_data, $s_file);
                    if (!count($a_data)) @unlink(get_configuration('app/path/data') . $s_file . '.php');
                }
            }
        }
    }

    // removes module
    if ($a_data = get_data('modules'))
    {
        if (array_key_exists($s_module_key, $a_data)) unset($a_data[$s_module_key]);
        write_data($a_data, 'modules');
    }

    if ($a_copied_files = get_module_data($s_module_key, 'copied_files'))
    {
        foreach ($a_copied_files['files'] as $s_base_file => $s_copied_file)
        {
            if (is_file($s_file = get_configuration('app/path/base') . $s_copied_file))
                @unlink($s_file);
        }
        foreach ($a_copied_files['dirs'] as $s_mk_dir)
        {
            if (is_dir($s_dir = get_configuration('app/path/base') . $s_mk_dir))
            {
                @rmdir($s_dir);
            }
        }
    }

    if (is_file($s_file = get_module_configuration($s_module_key, 'module/path/install') . 'uninstall.php'))
    {
        $s_module_key_code = str_replace('-', '_', $s_module_key);

        include_once $s_file;
        if (function_exists($s_function = $s_module_key_code . '_uninstall'))
            $s_function();
    }

    // removes data files
    foreach (array('chains.php', 'copied_files.php') as $s_data_file)
    {
        if (is_file($s_file = get_module_configuration($s_module_key, 'module/path/data') . $s_data_file))
            @unlink($s_file);
    }

    return true;
}

/**
 * Copys ressources files during a module installation.
 *
 * @param  string  $s_module_key  The module key.
 * @return boolean
 */
function module_copy_ressources($s_module_key)
{
    $a_copied_files = array('files' => array(), 'dirs' => array());

    // copier les fichiers dans dest/module_key/
    foreach (bfglob(get_module_configuration($s_module_key, 'module/path/web'), '*', 2, -1) as $s_file)
    {
        $a_copy_to = pathinfo(str_replace(get_module_configuration($s_module_key, 'module/path/base'), get_configuration('app/path/base'), $s_file));

        // [front|back]/dir1/dir.../dirn/file => [front|back]/module/dir1/dir.../dirn
        $a_dirs = explode(DIRECTORY_SEPARATOR, $a_copy_to['dirname']);
        if (false === ($i_pos = array_search(APP_FRONT, $a_dirs)))
            $i_pos = array_search(APP_BACK, $a_dirs);

        if (false !== $i_pos) {
            $a_copy_to['dirname'] = implode(DIRECTORY_SEPARATOR, array_slice($a_dirs, 0, $i_pos + 1));
            $a_copy_to['basename'] = implode(DIRECTORY_SEPARATOR, array_slice($a_dirs, $i_pos + 1)) . DIRECTORY_SEPARATOR . $a_copy_to['basename'];
        }

        $s_copy_to = $a_copy_to['dirname'].DIRECTORY_SEPARATOR.'modules'.DIRECTORY_SEPARATOR.$s_module_key.DIRECTORY_SEPARATOR.$a_copy_to['basename'];

        if (!is_dir($s_dir = dirname($s_copy_to)))
        {
            if (false === make_dir($s_dir))
            {
                set_message('error', __('ErrMakeDir', $s_dir));
                return false;
            }
            if (!in_array($s_dir, $a_copied_files['dirs']))
                $a_copied_files['dirs'][] = str_replace(get_configuration('app/path/base'), '', $s_dir);
        }
        if (false === copy_file($s_file, $s_copy_to))
        {
            set_message('error', __('ErrCopyFile', $s_file, $s_copy_to));
            return false; // @todo revert
        }

        $a_copied_files['files'][str_replace(get_module_configuration($s_module_key, 'module/path/base'), '', $s_file)] = str_replace(get_configuration('app/path/base'), '', $s_copy_to);
    }

    return (count($a_copied_files['files']) || count($a_copied_files['dirs'])) ? write_module_data($s_module_key, $a_copied_files, 'copied_files') : true;
}

/**
 * Up's a module.
 *
 * @param  string  $s_module_key  Module key.
 * @return boolean
 */
function module_up($s_module_key)
{
    return _module_up_down($s_module_key, true);
}

/**
 * Down's a module.
 *
 * @param  string  $s_module_key  Module key.
 * @return boolean
 */
function module_down($s_module_key)
{
    return _module_up_down($s_module_key, false);
}

/**
 * Up's or down's a module.
 *
 * @param  string  $s_module_key  Module key.
 * @return boolean
 */
function _module_up_down($s_module_key, $b_up)
{
    if (USER_ADMIN != get_user_property('rights'))
    {
        set_message('error', __('ErrUserRightsAction'));
        return false;
    }

    $a_installed_modules = modules_get_installed_modules();

    if (!array_key_exists($s_module_key, $a_installed_modules))
    {
        set_message('error', __('ErrModuleNotInstalled', $s_module_key));
        return false;
    }

    $a_modules_keys = array_keys($a_installed_modules);
    $i_module_pos = array_search($s_module_key, $a_modules_keys);

    if ($b_up)
    {
        if ($i_module_pos > 0 && count($a_modules_keys) > 1)
        {
            unset($a_modules_keys[$i_module_pos]);
            $a_modules_keys = array_values($a_modules_keys);
            array_splice($a_modules_keys, $i_module_pos - 1, 0, $s_module_key);
        }
        else
            return true;
    }
    else
    {
        if ($i_module_pos < count($a_modules_keys) -1 && count($a_modules_keys) > 1)
        {
            unset($a_modules_keys[$i_module_pos]);
            $a_modules_keys = array_values($a_modules_keys);
            array_splice($a_modules_keys, $i_module_pos + 1, 0, $s_module_key);
        }
        else
            return true;
    }

    // reorganize chains
    if ($a_chains = get_module_data($s_module_key, 'chains'))
    {
        foreach ($a_chains as $s_app => $a_chains_functions)
        {
            foreach ($a_chains_functions as $i_index => $a_function)
            {
                $s_file = 'modules'.DIRECTORY_SEPARATOR.$s_app.DIRECTORY_SEPARATOR.'chains'.DIRECTORY_SEPARATOR.$a_function['base_function'];

                if ($a_data = get_data($s_file))
                {
                    array_reorder($a_data, $a_modules_keys);
                    write_data($a_data, $s_file);
                }
            }
        }
    }

    // save modules
    array_reorder($a_installed_modules, $a_modules_keys);
    return write_data($a_installed_modules, 'modules');
}

/**
 * Adds a module view to display it in the back-office.
 *
 * @param  mixed  $m_view_name string/array. The view name or an array of view names.
 * @return void
 */
function set_module_view($m_view_name)
{
    foreach ((array)$m_view_name as $s_view_name)
        add_configuration('module/views', $s_view_name);
}

/**
 * Includes all views.
 *
 * @param  array   $a_vars        optional, variables for view.
 * @return void
 */
function include_module_views($a_vars = array(), $s_module_key = null)
{
    $s_module_key = get_configuration('module/key');

    $a_views = get_configuration('module/views', array());
    if (!count($a_views) || !$s_module_key) return ;

    if (null !== ($a_user_vars = get_application_parameter('_page/_template/_vars'))) extract($a_user_vars);
    if (count($a_vars)) extract($a_vars);
    $s_view_path = get_module_configuration($s_module_key, 'module/path/contents').'back'.DIRECTORY_SEPARATOR;

    foreach ($a_views as $s_view)
    {
        if (file_exists($s_file = $s_view_path.$s_view.'.php'))
            include $s_file;
    }
}

/**
 * Includes a content view.
 *
 * @param  string  $s_view        View name.
 * @param  array   $a_vars        optional, variables for view.
 * @return string
 */
function module_include_view($s_view_name, $a_vars = array())
{
    module_get_view($s_view_name, $a_vars, true);
}

/**
 * Returns a content view.
 *
 * @param  string   $s_view     View name.
 * @param  array    $a_vars     optional, variables for view.
 * @param  boolean  $b_include  optional, true if view must be included.
 * @return string
 */
function module_get_view($s_view_name, $a_vars = array(), $b_include = false)
{
    //$a_module = explode('/', $s_view_name);
    $s_module_key = /*(2 === count($a_module)) ? $a_module[0] :*/ get_configuration('module/key');
    /*$s_view_name = (2 === count($a_module)) ? $a_module[1] : $s_view_name;*/
    if (!$s_module_key) return '';

    if (null !== ($a_user_vars = get_application_parameter('_page/_template/_vars'))) extract($a_user_vars);
    if (count($a_vars)) extract($a_vars);
    $s_view_path = get_module_configuration($s_module_key, 'module/path/contents').'back'.DIRECTORY_SEPARATOR;

    if (file_exists($s_file = $s_view_path.$s_view_name.'.php'))
    {
        if ($b_include) {
            include $s_file;
            return ;
        }
        ob_start();
        include $s_file;
        return ob_get_clean();
    }

    return '';
}

/**
 * Returns all view contents.
 *
 * @param  string  $s_module_key  optional, the module key. If not defined, uses the current module key.
 * @param  array   $a_vars        optional, variables for view.
 * @return array  Associative array : array('view_name_1' => view_content_1, 'view_name_2' => view_content_2, ...)
 */
function module_get_views($a_vars = array())
{
    $s_module_key = /*$s_module_key ? $s_module_key :*/ get_configuration('module/key');

    $a_views = get_configuration('module/views', array());
    if (!count($a_views) || !$s_module_key) return array();

    if (null !== ($a_user_vars = get_application_parameter('_page/_template/_vars'))) extract($a_user_vars);
    if (count($a_vars)) extract($a_vars);
    $s_view_path = get_module_configuration($s_module_key, 'module/path/contents').'back'.DIRECTORY_SEPARATOR;

    $a_return = array();
    foreach ($a_views as $s_view)
    {
        if (file_exists($s_file = $s_view_path.$s_view.'.php'))
        {
            ob_start();
            include $s_file;
            $a_return[$s_view] = ob_get_clean();
        }
        else
            $a_return[$s_view] = '';
    }

    return $a_return;
}

/**
 * Loads a language file for a module.
 *
 * @param  string  $s_module_key  The module key.
 * @param  string  $s_lang        Language code.
 * @return void
 */
function module_load_language($s_module_key, $s_lang)
{
    load_language($s_lang, $s_module_key);
}