<?php
/*
 * Copyright 2012 Sébastien Raud
 *
 * This file is part of beCms.
 *
 * beCms is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * beCms is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with beCms.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * Creates a new page.
 *
 * @param  array   $a_properties  Page properties.
 * @param  string  $s_content     Page content.
 * @param  string  $s_parent_id   Parent id, root if null.
 * @return string  boolean
 */
function new_page(& $a_properties, $s_content = '', $s_parent_id = null)
{
    include_once get_configuration('app/path/back').'structure.php';

    if (!$s_parent_id) $s_parent_id = key(get_structure());
    elseif (!page_is_id($s_parent_id)) return false;

    $a_properties['parent_id'] = $s_parent_id;
    $a_dump_properties = $a_properties;
    if (!structure_add_child($s_parent_id, $a_properties)) return false;

    if (!page_write_content($a_properties['id'], $s_content) || !page_write_properties($a_properties['id'], array_merge(page_get_defaults(), $a_dump_properties)))
    {
        structure_delete_element($a_properties['id']);
        return false;
    }

    return true;
}

/**
 * Deletes a page.
 *
 * @param  string  $s_page_id  Page id.
 * @return boolean
 */
function delete_page($s_page_id)
{
    include_once get_configuration('app/path/back').'structure.php';
    return structure_delete_element($s_page_id);
}

/**
 * Returns an array with page informations.
 *
 * @param  string  $s_page_id  Page id.
 * @param  string  $b_draft    Indicates if is draft page.
 * @return array
 */
function get_editable_page($s_page_id, $b_draft = false)
{
    if (!page_is_id($s_page_id))
    {
        set_message('error', __('ErrPageNotExists', $s_page_id));
        return false;
    }

    if (!file_exists($s_properties_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.($b_draft ? '.draft' : '').'.properties.php'))
        if (!page_write_default_properties($s_page_id, $b_draft))
            return false;

    $a_structure_element = page_get_structure_element($s_page_id);
    $a_properties = array_merge(
                    include $s_properties_file,
                    array('redirection' => $a_structure_element['redirection'], 'type' => $a_structure_element['type'], 'slug' => $a_structure_element['slug'], 'route' => page_get_route($s_page_id), 'draft' => $b_draft));

    if (!file_exists($s_content_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.($b_draft ? '.draft' : '').'.php'))
        if (!page_write_content($s_page_id, '', $b_draft))
            return false;

    $s_content = file_get_contents($s_content_file);
    $s_content = str_replace(
                    array('$PROTOCOL$', '$BASE_URI_RESSOURCES_PATH$'),
                    array(get_configuration('request/protocol'), get_configuration('server/base_uri_ressources_path')),
                    $s_content);

    if (!isset($a_properties['locked'])) $a_properties['locked'] = false;

    return array_merge($a_properties, array('content' => $s_content));
}

/**
 * Returns an array with draft page informations.
 *
 * @param  string  $s_page_id  Page id.
 * @return array
 */
function get_editable_page_draft($s_page_id)
{
    if (!page_is_id($s_page_id))
    {
        set_message('error/message', __('ErrPageNotExists', $s_page_id));
        return false;
    }

    if (!file_exists($s_content_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.draft.php'))
        if (!page_copy_draft_content($s_page_id))
            return false;

    if (!file_exists($s_properties_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.draft.properties.php'))
        if (!page_copy_draft_properties($s_page_id))
            return false;

    return get_editable_page($s_page_id, true);
}

/**
 * Returns a slug for a page.
 *
 * @param  string  $s_page_id  Page id.
 * @return string
 */
function page_get_slug($s_page_id)
{
    if (!count($a_routes = page_get_route($s_page_id)))
        return null;

    $a_routes = explode('/', $a_routes['route']);
    $s_slug = array_pop($a_routes);
    return $s_slug;
}

/**
 * Deletes cache page if exists.
 *
 * @param  string  $s_page_id  Page id.
  * @return void
  */
function page_delete_cache($s_page_id)
{
    $s_cache_file = get_configuration('app/path/cache').str_replace('/', DIRECTORY_SEPARATOR, preg_replace('~(https?)://~i', '\1/', trim(get_front_uri($s_page_id), '/'))).'.cache.php';
    if (file_exists($s_cache_file)) unlink($s_cache_file);
}

/**
 * Returns a property value for a page.
 *
 * @param  string   $s_page_id   Page id.
 * @param  boolean  $b_is_draft  Indicates if the property is for the draft page.
 * @retunr mixed
 */
function page_get_properties($s_page_id, $b_is_draft = false)
{
    if (file_exists($s_properties_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.($b_is_draft ? '.draft' : '').'.properties.php'))
    {
        $a = include $s_properties_file;
        if (!isset($a['locked'])) $a['locked'] = false;
        return $a;
    }
    return null;
}

/**
 * Returns a route for a page.
 *
 * @param  string  $s_page_id  Page id.
 * @return string
 */
function page_get_route($s_page_id)
{
    if (!page_is_id($s_page_id))
        return array('route' => null, 'extension' => null);

    $s_routes_file = get_configuration('app/path/data').'front_routes_flip.php';

    if (!file_exists($s_routes_file))
        return array();

    $a_routes = include $s_routes_file;

    if (!array_key_exists($s_page_id, $a_routes))
        return array('route' => null, 'extension' => null);

    $a_routes_values = array_values($a_routes[$s_page_id]); $a_extension_values = array_keys($a_routes[$s_page_id]);
    return array('route' => array_shift($a_routes_values), 'extension' => array_shift($a_extension_values));
}

/**
 * Returns element from structure for a page.
 *
 * @param  string  $s_page_id    Page id.
 * @return array
 */
function page_get_structure_element($s_page_id)
{
    include_once get_configuration('app/path/back').'structure.php';
    return structure_get_element($s_page_id);
}

/**
 * Indicates if an id correspond to a page.
 *
 * @param  string  $s_page_id  Page id.
 * @return boolean
 */
function page_is_id($s_page_id)
{
    return null !== page_get_structure_element($s_page_id);
}

/**
 * Indicates if page is root page.
 *
 * @param  string  $s_page_id  Page id.
 * @return boolean
 */
function page_is_root($s_page_id)
{
    $a_structure = get_structure();
    list($s_key, $a_properties) = each($a_structure);
// ? page_id = '';
    return intval($s_key === $s_page_id);
}

/**
 * Writes defaults properties file for a page.
 *
 * @param  string  $s_page_id  Page id.
 * @param  string  $b_draft    Indicates if is draft page.
 * @return boolean
 */
function page_write_default_properties($s_page_id, $b_draft = false)
{
    $s_page_id .= ($b_draft ? '.draft' : '');
    $s_properties_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.properties.php';
    return write_file($s_properties_file, '<?php'."\n".'return '.var_export(page_get_defaults(), true).';');
}

/**
 * Writes properties file for a page.
 *
 * @param  string  $s_page_id     Page id.
 * @param  array   $a_properties  Properties.
 * @param  string  $b_draft       Indicates if is draft page.
 * @return array
 */
function page_write_properties($s_page_id, $a_properties, $b_draft = false)
{
    if (!page_is_id($s_page_id))
        return false;

    if (is_array($a_properties['locked']))
    {
        if ($a_properties['locked']['id'] !== get_user_id())
        {
            set_message('error', __('ErrPageLocked'));
            return false;
        }
    }

    $s_page_id .= ($b_draft ? '.draft' : '');

    $a_real_properties = array();
    foreach (page_get_defaults() as $s_key => $m_default)
        $a_real_properties[$s_key] = array_key_exists($s_key, $a_properties) ? $a_properties[$s_key] : $m_default;

    $s_properties_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.properties.php';
    $a_real_properties['dates']['modification'] = strval(time());

    if (page_is_root($s_page_id)) $a_real_properties['slug'] = '';

    if (write_file($s_properties_file, '<?php'."\n".'return '.var_export($a_real_properties, true).';'))
    {
        $a = array('id' => str_replace('.draft', '', $s_page_id), 'properties' => $a_real_properties, 'is_draft' => $b_draft);
        module_chain(__FUNCTION__, $a);
        return true;
    }
    return false;
}

/**
 * Copy real properties page to draft properties page.
 *
 * @param  string  $s_page_id  Page id.
 * @retur boolean.
 */
function page_copy_draft_properties($s_page_id)
{
    if (!page_is_id($s_page_id))
        return false;

    $s_original_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.properties.php';
    $s_draft_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.draft.properties.php';

    if (!file_exists($s_original_file))
        return false;

    return copy_file($s_original_file, $s_draft_file);
}

/**
 * Writes contents file for a page.
 *
 * @param  string  $s_page_id  Page id.
 * @param  string  $s_content  Content.
 * @param  string  $b_draft    Indicates if is draft page.
 * @retur boolean.
 */
function page_write_content($s_page_id, $s_content, $b_draft = false)
{
    if (!page_is_id($s_page_id))
        return false;

    if (($a_properties = page_get_properties($s_page_id, $b_draft)) && is_array($a_properties['locked']))
    {
        if ($a_properties['locked']['id'] !== get_user_id())
        {
            set_message('error', __('ErrPageLocked'));
            return false;
        }
    }

    $s_page_id .= ($b_draft ? '.draft' : '');
    $s_content_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.php';

    $s_content = str_ireplace(
                    array('http://'.get_configuration('server/base_uri_ressources_path'), 'https://'.get_configuration('server/base_uri_ressources_path'), get_configuration('server/base_uri_ressources_path')),
                    array('$PROTOCOL$://$BASE_URI_RESSOURCES_PATH$', '$PROTOCOL$://$BASE_URI_RESSOURCES_PATH$', '$BASE_URI_RESSOURCES_PATH$'),
                    $s_content);

    if (write_file($s_content_file, $s_content))
    {
        $a = array('id' => str_replace('.draft', '', $s_page_id), 'content' => $s_content, 'is_draft' => $b_draft);
        module_chain(__FUNCTION__, $a);
        return true;
    }
    return false;
}

/**
 * Copy real page to draft page.
 *
 * @param  string  $s_page_id  Page id.
 * @param  string  $b_draft    Indicates if is draft page.
 * @retur boolean.
 */
function page_copy_draft_content($s_page_id)
{
    if (!page_is_id($s_page_id))
        return false;

    $s_original_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.php';
    $s_draft_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.draft.php';

    if (!file_exists($s_original_file))
        return false;

    return copy_file($s_original_file, $s_draft_file);
}

/**
 * Updates structure with a page properties.
 *
 * @param  string  $s_page_id  Page id.
 * @param
 * @return boolean
 */
function page_update_structure($s_page_id, & $a_page_properties, $b_draft = false)
{
    if ($b_draft) return true;

    if (!count($a_page_properties))
    {
        $s_properties_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_page_id.'.properties.php';
        if (!file_exists($s_properties_file))
            return false;

        $a_page_properties = include $s_properties_file;
    }

    if (!($a_element = page_get_structure_element($s_page_id)))
        return false;

    // dump page to structure
    foreach (array_keys($a_page_properties) as $s_key) $a_element[$s_key] = $a_page_properties[$s_key];
    if (!structure_modify_element($a_element['id'], $a_element)) return false;

    return true;
}

/**
 * Indicates if a page exists as a cache file.
 *
 * @param  string  $s_id  Page id.
 * @return boolean
 */
function page_is_cached($s_id)
{
    $a_routes = get_data('front_routes_flip');
    if (!array_key_exists($s_id, $a_routes)) return false;

    return is_file(get_configuration('app/path/cache').str_replace('/', DIRECTORY_SEPARATOR, ($a_routes[$s_id]['is_secure'] ? 'https' : 'http') .DIRECTORY_SEPARATOR.get_configuration('server/base_uri_path').DIRECTORY_SEPARATOR.$a_routes[$s_id]['route']).'.cache.php');
}