<?php
/*
 * Copyright 2012 Sébastien Raud
 *
 * This file is part of beCms.
 *
 * beCms is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * beCms is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with beCms.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * Adds a new element to the site's structure.
 *
 * @param  string  $s_parent_id  Parent's id.
 * @param  array   $a_element    Element values.
 * @return boolean
 */
function structure_add_child($s_parent_id, & $a_element)
{
    if (!structure_control_values($a_element))
        return false;

    $a_structure = get_structure();

    if (!isset($a_structure[$s_parent_id]))
    {
        set_message('error', __('ErrPageNotExists', $s_parent_id));
        return false;
    }

    // add to structure
    $a_element['parent_id'] = $s_parent_id;
    $a_structure[$a_element['id']] = $a_element;
    $a_structure[$s_parent_id]['children'][] = $a_element['id'];

    if (!structure_write_properties($a_element['id'], $a_element)) return false;
    if (!structure_write_content($a_element['id'])) return false;

    if (!structure_save($a_structure, $a_element['id'])) // update routes, cache and menu for element
        return false;

    module_chain(__FUNCTION__, $a_element);
    return true;
}

/**
 * Modifys an element.
 *
 * @param  string  $s_element_id  Element id.
 * @param  array   $a_element     Element values.
 * @return boolean
 */
function structure_modify_element($s_element_id, & $a_element)
{
    if (!structure_control_values($a_element))
        return false;

    $a_structure = get_structure();

    if (!isset($a_structure[$s_element_id]))
    {
        set_message('error', __('ErrPageNotExists', $s_parent_id));
        return false;
    }

    if ($s_element_id === key($a_structure)) // root
    {
        $a_element['slug'] = '';
        $a_element['show_slug'] = false;
        $a_element['extension'] = '';
    }

    $a_element['children'] = $a_structure[$s_element_id]['children'];
    $a_structure[$a_element['id']] = $a_element;

    if (!structure_write_properties($a_element['id'], $a_element, true)) return false;

    if (!intval($a_element['cache']) || STRUCTURE_STATE_VISIBLE != $a_element['state'])
    {
        // deletes cache
        if (file_exists($s_cache_file = get_configuration('app/path/cache').str_replace('/', DIRECTORY_SEPARATOR, preg_replace('~(https?)://~i', '\1/', trim(get_front_uri($a_element['id']), '/'))).'.cache.php'))
            unlink($s_cache_file);
    }

    if (!structure_save($a_structure, $a_element['id'])) // update routes, cache and menu for element
        return false;

    module_chain(__FUNCTION__, $a_element);
    return true;
}

/**
 * Up's an element.
 *
 * @param  string  $s_element_id  Element id.
 * @return boolean
 */
function structure_up_element($s_element_id)
{
    $a_structure = get_structure();

    if (!isset($a_structure[$s_element_id]))
    {
        set_message('error', __('ErrPageNotExists', $s_parent_id));
        return false;
    }

    if ($s_element_id === key($a_structure)) // root
    {
        set_message('error', __('ErrStructureRootUp'));
        return false;
    }

    $s_parent_id = $a_structure[$s_element_id]['parent_id'];
    array_up($a_structure[$s_parent_id]['children'], $s_element_id);

    if (!structure_save($a_structure, $s_parent_id, 4)) // update menu parent
        return false;

    module_chain(__FUNCTION__, $s_element_id);
    return true;
}

/**
 * Down's an element.
 *
 * @param  string  $s_element_id  Element id.
 * @return boolean
 */
function structure_down_element($s_element_id)
{
    $a_structure = get_structure();

    if (!isset($a_structure[$s_element_id]))
    {
        set_message('error', __('ErrPageNotExists', $s_parent_id));
        return false;
    }

    if ($s_element_id === key($a_structure)) // root
    {
        set_message('error', __('ErrStructureRootUp'));
        return false;
    }

    $s_parent_id = $a_structure[$s_element_id]['parent_id'];
    array_down($a_structure[$s_parent_id]['children'], $s_element_id);

    if (!structure_save($a_structure, $s_parent_id, 4)) // update menu parent
        return false;

    module_chain(__FUNCTION__, $s_element_id);
    return true;
}

/**
 * Deletes an element.
 *
 * @param  string  $s_element_id  Element id.
 * @return boolean
 */
function structure_delete_element($s_element_id)
{
    $a_structure = get_structure();

    if (!isset($a_structure[$s_element_id]))
    {
        set_message('error', __('ErrPageNotExists', $s_element_id));
        return false;
    }

    if ($s_element_id === key($a_structure)) // root
    {
        set_message('error', __('ErrStructureRootRemove'));
        return false;
    }

    $s_parent_id = $a_structure[$s_element_id]['parent_id'];

    // move element children ids as children of parent element
    if (!empty($a_structure[$s_element_id]['children']))
    {
        // change parent_id
        foreach ($a_structure[$s_element_id]['children'] as $s_id)
            $a_structure[$s_id]['parent_id'] = $s_parent_id;

        if (false !== ($i_pos = array_search($s_element_id, $a_structure[$s_parent_id]['children'])))
        {
            $a_structure[$s_parent_id]['children'] = array_merge(array_slice($a_structure[$s_parent_id]['children'], 0, $i_pos),
                                                                 $a_structure[$s_element_id]['children'],
                                                                 array_slice($a_structure[$s_parent_id]['children'], $i_pos + 1, count($a_structure[$s_parent_id]['children']) - $i_pos));
        }
    }
    else
        array_remove($a_structure[$s_parent_id]['children'], $s_element_id);

    // remove element from structure
    unset($a_structure[$s_element_id]);

    if (!structure_save($a_structure, array($s_parent_id, $s_element_id), 8)) return false; // update parent and delete element

    // deletes files
    foreach (array(get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_element_id.'.properties.php',
                   get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_element_id.'.properties.draft.php',
                   get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_element_id.'.php',
                   get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_element_id.'.draft.php') as $s_rm_file)
    {

        if (file_exists($s_rm_file))
            unlink($s_rm_file);
    }

    module_chain(__FUNCTION__, $s_element_id);
    return true;
}

/**
 * Gets an element from structure.
 *
 * @param  string  $s_element_id  Element id.
 * @return void
 */
function structure_get_element($s_element_id)
{
    $a_structure = get_structure();
    return isset($a_structure[$s_element_id]) ? $a_structure[$s_element_id] : null;
}

/**
 * Saves structure.
 *
 * @param  array    $a_structure   Site's structure.
 * @param  string   $m_element_id  Element id, to update routes, cache and menu. Array for a deleted element (parent_id, element_id).
 * @param  integer  $i_update      Indexed files to update (1 routes ; 2 cache ; 4 : menu, 8 : delete).
 * @return boolean
 */
function structure_save($a_structure, $m_element_id = null, $i_update = 7)
{
    if (!write_data($a_structure, 'structure'))
        return false;

    return structure_mk_indexed_files($a_structure, $i_update, $m_element_id);
}

function structure_mk_indexed_files(& $a_structure, $i_update = 7, $m_element_id = null)
{
    if (!$m_element_id) $m_element_id = key($a_structure);

    if (8 === ($i_update & 8) && !is_array($m_element_id)) return false;

    $a_params = array(
        'routes' => get_data('front_routes'),
        'routes_flip' => get_data('front_routes_flip'),
        'cache' => ((8 === ($i_update & 8)) || (2 === ($i_update & 2))) ? get_data('front_cache') : array(),
        'menu' => array(),
        'admin_route' => strtolower(get_configuration('administration/route'))
    );

    if (8 === ($i_update & 8))
    {
        structure_mk_indexed_files_delete($a_structure, $a_params, $m_element_id);
        $m_element_id = $m_element_id[0];
    }
    if ((8 === ($i_update & 8)) || (1 === ($i_update & 1))) structure_mk_indexed_files_routes($a_structure, $a_params, $m_element_id);
    if ((8 === ($i_update & 8)) || (2 === ($i_update & 2))) structure_mk_indexed_files_cache($a_structure, $a_params, $m_element_id);
    if ((8 === ($i_update & 8)) || (4 === ($i_update & 4)))
    {
        $a_params['structure'] = & $a_structure;
        structure_walk('structure_mk_menu', null, $a_params);
    }

    $b_is_success = true;

    if ((8 === ($i_update & 8)) || (1 === ($i_update & 1)))
        $b_is_success = ($b_is_success && write_data($a_params['routes'], 'front_routes') && write_data($a_params['routes_flip'], 'front_routes_flip'));
    if ((8 === ($i_update & 8)) || (2 === ($i_update & 2)))
        $b_is_success = ($b_is_success && write_data($a_params['cache'], 'front_cache'));
    if ((8 === ($i_update & 8)) || (4 === ($i_update & 4)))
        $b_is_success = ($b_is_success && write_data($a_params['menu'], 'front_menu'));

    return $b_is_success;
}

function structure_mk_indexed_files_delete(& $a_structure, & $a_params, $a_elements_id)
{
    $s_parent_id = $a_elements_id[0];
    $s_element_id = $a_elements_id[1];

    if (!isset($a_params['routes_flip'][$s_element_id])) return ;

    $s_route = $a_params['routes_flip'][$s_element_id]['route'];
    unset($a_params['routes_flip'][$s_element_id]);

    if (isset($a_params['routes'][$s_route])) unset($a_params['routes'][$s_route]);
    if (isset($a_params['cache'][$s_route]))  unset($a_params['cache'][$s_route]);
}

function structure_mk_indexed_files_routes(& $a_structure, & $a_params, $s_element_id)
{
    // on prend le parent et sa route (depuis routes_flip),
    // on applique l'algo de routes sur l'élément et ses enfants
    // pour chaque route dans routes on vire l'ancienne route
    // on ajoute la nouvelle
    $a_element = $a_structure[$s_element_id];

    if (isset($a_params['routes_flip'][$s_element_id]))
    {
        $s_route = $a_params['routes_flip'][$s_element_id]['route'];
        unset($a_params['routes'][$s_route]);
    }

    $s_parent_id = $a_element['parent_id'];
    $s_parent_route = empty($s_parent_id) ? '' : $a_params['routes_flip'][$s_parent_id]['route']; // empty if $s_element_id === root

    $s_current_route = $s_parent_route;

    if (intval($a_element['show_slug']))
        $s_current_route = ($s_parent_route ? $s_parent_route.'/' : '').$a_element['slug'];

    $s_route_path = strtolower($s_current_route.('' !== $a_element['extension'] && strlen($a_element['extension'] && intval($a_element['show_slug'])) ? '.'.$a_element['extension'] : ''));

    if (array_key_exists($s_route_path, $a_params['routes']))
        set_message('warning', __('MsgRouteCollision', $s_route_path, $a_element['title'], $s_element_id));

    if ($s_route_path === $a_params['admin_route'])
        set_message('error', __('ErrRouteNotAvailable', $s_route_path, $a_element['title'], $s_element_id));
    else
    {
        $a_params['routes'][$s_route_path] =  array('id' => $a_element['id'], 'title' => $a_element['title'], 'redirection' => $a_element['redirection'], 'redirection_code' => $a_element['redirection_code'], 'is_secure' => $a_element['is_secure']);
        $a_params['routes_flip'][$s_element_id] = array('route' => $s_route_path, 'title' => $a_element['title'], 'is_secure' => $a_element['is_secure']);
    }

    foreach ($a_element['children'] as $s_element_id)
        structure_mk_indexed_files_routes($a_structure, $a_params, $s_element_id);
}

function structure_mk_indexed_files_cache(& $a_structure, & $a_params, $s_element_id)
{
    $s_route = $a_params['routes_flip'][$s_element_id]['route'];
    $i_cache = empty($a_structure[$s_element_id]['cache']) || !intval($a_structure[$s_element_id]['state']) ? 0 : intval($a_structure[$s_element_id]['cache']);

    if ($i_cache)
        $a_params['cache'][$s_route] = $i_cache;

    elseif (!$i_cache && isset($a_params['cache'][$s_route]))
        unset($a_params['cache'][$s_route]);
}

function structure_mk_indexed_files_menu(& $a_structure, & $a_params, $s_element_id)
{
    $a_params['structure'] = $a_structure;
    structure_walk('structure_mk_menu', null, $a_params);
    unset($a_params['structure']);
}

function structure_mk_menu($a_element, $i_level, & $a_params)
{
    /*if (intval($a_element['index_menu']) && intval($a_element['state']) && array_key_exists($a_element['id'], $a_params['routes_flip']))
    {
        $a_params['menu'][$a_element['id']] = array_merge(
            array(
                'id' => $a_element['id'],
                'parent_id' => '',
                'children' => array()),
            $a_params['routes_flip'][$a_element['id']]
        );

        structure_mk_indexed_files_menu_link_to_parent($a_params, $a_params['menu'][$a_element['id']]);
    }*/
    if (array_key_exists($a_element['id'], $a_params['routes_flip']))
    {
        $a_params['menu'][$a_element['id']] = array_merge(
            array(
                'id' => $a_element['id'],
                'parent_id' => '',
                'children' => array()),
            $a_params['routes_flip'][$a_element['id']]
        );

        structure_mk_indexed_files_menu_link_to_parent($a_params, $a_params['menu'][$a_element['id']]);

        if (!intval($a_element['index_menu']) || !intval($a_element['state']))
            unset($a_params['menu'][$a_element['id']]);
    }
}

function structure_mk_indexed_files_menu_link_to_parent(& $a_params, & $a_element)
{
    $s_element_id = $a_element['id'];
    $s_current_id = $s_element_id;

    while (true)
    {
        if (empty($s_current_id)) return false;

        $s_parent_id = $a_params['structure'][$s_current_id]['parent_id'];

        if (empty($s_parent_id)) return false;

        $a_parent = & $a_params['structure'][$s_parent_id];

        if (!intval($a_parent['index_menu']) || !intval($a_parent['state']))
            $s_current_id = $a_parent['id'];
        else
        {
            $a_params['menu'][$s_parent_id]['children'][] = $s_element_id;
            $a_element['parent_id'] = $s_parent_id;
            return ;
        }
    }
}

/**
 * Checks values for an element.
 *
 * @param  array    $a_element  Element values.
 * @return boolean
 */
function structure_control_values(& $a_element)
{
    $b_error = false;

    $a_defaults = structure_get_defaults();
    foreach ($a_defaults as $s_key => $m_value)
    {
        $a_element[$s_key] = array_key_exists($s_key, $a_element) ? $a_element[$s_key] : $m_value;
    }

    foreach ($a_element as $s_key => $m_value)
    {
        if (!array_key_exists($s_key, $a_defaults))
            unset($a_element[$s_key]);
    }

    foreach (array('id', 'title', 'slug', 'cache') as $s_key)
        $a_element[$s_key] = trim($a_element[$s_key]);

    if (empty($a_element['id']))
        $a_element['id'] = generate_id();

    if (empty($a_element['title']))
    {
        set_message('error', __('MsgRequiredTitle'));
        $b_error = true;
    }

    include_once get_configuration('app/path/back').'urlize.php';

    if (empty($a_element['slug']))
        $a_element['slug'] = urlize($a_element['title']);
    else
        $a_element['slug'] = urlize($a_element['slug']);

    if (!preg_match('/^[\d]+$/', $a_element['cache']))
        $a_element['cache'] = 0;

    if (!empty($a_element['redirection']) && empty($a_element['redirection_code'])) $a_element['redirection_code'] = 'xxx';
    if (empty($a_element['redirection'])) $a_element['redirection_code'] = '';

    return !$b_error;
}

/**
 * Writes properties file for a page.
 *
 * @param  string   $s_page_id     Page id.
 * @param  array    $a_properties  Properties.
 * @param  boolean  $b_merge       Indicates if values must be merged with actuals proprties.
 * @return boolean
 */
function structure_write_properties($s_element_id, $a_properties, $b_merge = false)
{
    $s_properties_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_element_id.'.properties.php';

    if ($b_merge && is_file($s_properties_file))
        $a_properties = array_merge(include $s_properties_file, $a_properties);

    // clean
    $a_page_defaults = page_get_defaults();
    foreach ($a_page_defaults as $s_key => $m_value)
    {
        if (!array_key_exists($s_key, $a_properties))
            $a_properties[$s_key] = $m_value;
    }

    foreach ($a_properties as $s_key => $m_value)
    {
        if (!array_key_exists($s_key, $a_page_defaults))
            unset($a_properties[$s_key]);
    }

    return write_file($s_properties_file, '<?php'."\n".'return '.var_export($a_properties, true).';');
}

/**
 * Writes content file for a page.
 *
 * @param  string  $s_page_id  Page id.
 * @return boolean
 */
function structure_write_content($s_element_id)
{
    $s_contents_file = get_configuration('app/path/contents').APP_FRONT.DIRECTORY_SEPARATOR.$s_element_id.'.php';
    return write_file($s_contents_file, '');
}

/**
 * Returns element id from id / path id.
 *
 * @param  string  $s_id  Id or path id.
 * @return string
 */
function structure_get_id($s_id)
{
    return (false !== ($i_index = strrpos($s_id, '/'))) ? substr($s_id, $i_index + 1) : $s_id;
}

function structure_walk($s_function, $s_id = null, & $a_params = array())
{
    $a_structure = get_structure();
    if (!$s_id) $s_id = key($a_structure);

    _structure_walk($s_function, $a_structure, $s_id, $a_params);
}

function _structure_walk($s_function, & $a_structure, $s_id, & $a_params = array(), $i_level = 0)
{
    $a_properties = $a_structure[$s_id];

    $s_function($a_properties, $i_level, $a_params);

    foreach ($a_properties['children'] as $s_index)
        _structure_walk($s_function, $a_structure, $s_index, $a_params, $i_level + 1);
}