<?php
/*
 * Copyright 2012 Sébastien Raud
 *
 * This file is part of beCms.
 *
 * beCms is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * beCms is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with beCms.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * Checks values for a user.
 *
 * @param  array    $a_user    User values.
 * @param  boolean  $b_is_new  Indicates if user is a new user.
 * @return boolean
 */
function user_control_values(& $a_user, $b_is_new = false)
{
    $a_errors = array();

    /* required fields */
    foreach (array('login', 'firstname', 'lastname', 'rights', 'password') as $s_field)
    {
        if (!array_key_exists($s_field, $a_user))
        {
            $a_errors[] = __('MsgRequiredField', $s_field);
            return false;
        }
    }

    if (!preg_match('~^[a-zA-Z0-9_\-]+$~', $a_user['login']))
        $a_errors[] = __('MsgCheckLogin', $a_user['login']);

    if (strlen($a_user['login']) < 4)
        $a_errors[] = __('MsgLengthLogin', $a_user['login']);

    if ($b_is_new)
    {
        $a_users = get_users();
        if (array_key_exists($a_user['login'], $a_users))
            $a_errors[] = __('ErrLoginExists', $a_user['login']);
    }

    if (!$a_user['firstname'] || !strlen($a_user['firstname']))
        $a_errors[] = __('MsgRequiredField', __('firstname'));

    if (!$a_user['lastname'] || !strlen($a_user['lastname']))
        $a_errors[] = __('MsgRequiredField', __('lastname'));

    if (!preg_match('~^[a-zA-Z0-9_\-]+$~', $a_user['password']))
        $a_errors[] = __('MsgCheckPassword');

    if (!$a_user['password'] || strlen($a_user['password']) < 6)
        $a_errors[] = __('MsgLengthPassword');

    $a_user['lang'] = 'fr';

    $a_user['rights'] = intval($a_user['rights']);

    if (count($a_errors))
    {
        set_message('error', $a_errors);
        return false;
    }

    return true;
}

/**
 * Save a new user.
 *
 * @param  array  $a_user  User values.
 * @return boolean
 */
function user_new($a_user)
{
    if (!user_control_values($a_user, true))
        return false;

    $a_user['password'] = user_password_encode($a_user['password']);
    $a_users = include get_configuration('app/path/data').'users.php';
    $a_users[$a_user['login']] = $a_user;
    unset($a_users[$a_user['login']]['login']);

    return user_save_file($a_users);
}

/**
 * Save a modified user.
 *
 * @param  array  $a_user  User values.
 * @return boolean
 */
function user_modify($a_user)
{
    if (!user_control_values($a_user))
        return false;

    $a_user['password'] = user_password_encode($a_user['password']);
    $a_users = get_users();
    $a_users[$a_user['login']] = $a_user;
    unset($a_users[$a_user['login']]['login']);

    return user_save_file($a_users);
}

/**
 * Deletes an user
 *
 * @param  array  $a_user  User values.
 * @return boolean
 */
function user_delete($a_user)
{
    $a_users = get_users();

    if (array_key_exists($a_user['login'], $a_users))
    {
        unset($a_users[$a_user['login']]);

        // one admin user is required
        $b_has_admin = false;
        foreach ($a_users as $a_properties)
            if (($b_has_admin = ($a_properties['rights'] === USER_ADMIN)))
                break;

        if (!$b_has_admin)
        {
            set_message('error', __('ErrNoAdminUser'));
            return false;
        }

        return user_save_file($a_users);
    }

    return false;
}

/**
 * Returns the users list.
 *
 * @param  boolean  $b_sort  Indicates if the list must be sort.
 * @return array
 */
function users_get_list($b_sort = true)
{
    $a_users = get_users();
    if ($b_sort) ksort($a_users);
    foreach ($a_users as $s_login => $a_properties) $a_users[$s_login]['login'] = $s_login;
    return $a_users;
}

/**
 * Saves users in users file.
 *
 * @param  array  $a_users  All users to save.
 * @return boolean
 */
function user_save_file($a_users)
{
    foreach ($a_users as $s_login => & $a_user)
        if (array_key_exists('login', $a_user)) unset($a_user['login']);

    return write_data($a_users, 'users');
}

/**
 * Encodes password.
 *
 * @param  string  $s_password  Non encoding password.
 * @return string
 */
function user_password_encode($s_password)
{
    return base64_encode($s_password);
}

/**
 * Decodes password.
 *
 * @param  string  $s_password  Encoding password.
 * @return string
 */
function user_password_decode($s_password)
{
    return base64_decode($s_password);
}

/**
 * Returns the default configuration for one user.
 *
 * @return array
 */
function user_get_default()
{
    return array(
        'firstname' => '',
        'lastname' => '',
        'rights' => USER_USER,
        'password' => '',
        'lang' => 'fr',
    );
}