<?php
/*
 * Copyright 2012 Sébastien Raud
 *
 * This file is part of beCms.
 *
 * beCms is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * beCms is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with beCms.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * Initializes application.
 *
 * @return void
 */
function initialize()
{
    // constants
    define('APPLICATION_NAME',      'beCms');
    define('APPLICATION_VERSION',   '1.0');
    define('APPLICATION_DATE',      '2014-01-12');  // yyyy-mm-dd
    define('beCms',                 true);

    define('ACTION_GET',        1);
    define('ACTION_SET',        2);
    define('ACTION_DELETE',     3);
    define('ACTION_LOAD',       4);

    define('USER_ADMIN',    1);
    define('USER_USER',     2);

    define('APP_BACK',      'back');
    define('APP_FRONT',     'front');

    define('STRUCTURE_STATE_VISIBLE',   1);
    define('STRUCTURE_STATE_HIDDEN',    0);

    fix_magic_quotes_gpc();

    // init bases paths to directories
    $s_app_path_base = realpath(dirname(__FILE__).DIRECTORY_SEPARATOR.'..'.DIRECTORY_SEPARATOR.'..'.DIRECTORY_SEPARATOR.'..').DIRECTORY_SEPARATOR;
    $s_cms_path_base = realpath(dirname(__FILE__).DIRECTORY_SEPARATOR.'..'.DIRECTORY_SEPARATOR.'..').DIRECTORY_SEPARATOR;

    set_configuration('app/path/base',      $s_app_path_base);
    set_configuration('app/path/cms',       $s_cms_path_base);
    set_configuration('app/path/libs',      $s_cms_path_base.'libs'.DIRECTORY_SEPARATOR);
    set_configuration('app/path/core',      $s_cms_path_base.'libs'.DIRECTORY_SEPARATOR.'core'.DIRECTORY_SEPARATOR);
    set_configuration('app/path/data',      $s_cms_path_base.'data'.DIRECTORY_SEPARATOR.'config'.DIRECTORY_SEPARATOR);
    set_configuration('app/path/back',      $s_cms_path_base.'libs'.DIRECTORY_SEPARATOR.'apps'.DIRECTORY_SEPARATOR.APP_BACK.DIRECTORY_SEPARATOR);
    set_configuration('app/path/front',     $s_cms_path_base.'libs'.DIRECTORY_SEPARATOR.'apps'.DIRECTORY_SEPARATOR.APP_FRONT.DIRECTORY_SEPARATOR);
    set_configuration('app/path/contents',  $s_cms_path_base.'data'.DIRECTORY_SEPARATOR.'contents'.DIRECTORY_SEPARATOR);
    set_configuration('app/path/helpers',   $s_cms_path_base.'data'.DIRECTORY_SEPARATOR.'helpers'.DIRECTORY_SEPARATOR);
    set_configuration('app/path/lang',      $s_cms_path_base.'data'.DIRECTORY_SEPARATOR.'lang'.DIRECTORY_SEPARATOR);
    set_configuration('app/path/layouts',   $s_cms_path_base.'layouts'.DIRECTORY_SEPARATOR);
    set_configuration('app/path/modules',   $s_cms_path_base.'modules'.DIRECTORY_SEPARATOR);
    set_configuration('app/path/web',       $s_app_path_base.'web'.DIRECTORY_SEPARATOR);
    set_configuration('app/path/cache',     $s_app_path_base.'web'.DIRECTORY_SEPARATOR.'cache'.DIRECTORY_SEPARATOR);

    // load configuration file
    load_configuration();

    $s_base_uri_path = trim(preg_replace('#(/+)#', '/', get_configuration('server/base_url')), '/');
    set_configuration('server/base_uri_path',               $s_base_uri_path);
    set_configuration('server/base_uri_ressources_path',    trim(preg_replace('#(/+)#', '/', $s_base_uri_path . '/' . ('/' === get_configuration('server/base_dir') ? '/web/' : '/')), '/'));
    set_configuration('server/script_name',                 defined('INDEX_FILE_NAME') ? INDEX_FILE_NAME : basename($_SERVER['SCRIPT_NAME']));

    // modules
    $s_module_path_base = get_configuration('app/path/modules') . '$MODULE_KEY$' . DIRECTORY_SEPARATOR;

    set_configuration('module/path/base',       $s_module_path_base);
    set_configuration('module/path/data',       $s_module_path_base.'data'.DIRECTORY_SEPARATOR.'config'.DIRECTORY_SEPARATOR);
    set_configuration('module/path/libs',       $s_module_path_base.'libs'.DIRECTORY_SEPARATOR);
    set_configuration('module/path/install',    $s_module_path_base.'libs'.DIRECTORY_SEPARATOR.'install'.DIRECTORY_SEPARATOR);
    set_configuration('module/path/back',       $s_module_path_base.'libs'.DIRECTORY_SEPARATOR.'apps'.DIRECTORY_SEPARATOR.APP_BACK.DIRECTORY_SEPARATOR);
    set_configuration('module/path/front',      $s_module_path_base.'libs'.DIRECTORY_SEPARATOR.'apps'.DIRECTORY_SEPARATOR.APP_FRONT.DIRECTORY_SEPARATOR);
    set_configuration('module/path/contents',   $s_module_path_base.'data'.DIRECTORY_SEPARATOR.'contents'.DIRECTORY_SEPARATOR);
    set_configuration('module/path/helpers',    $s_module_path_base.'data'.DIRECTORY_SEPARATOR.'helpers'.DIRECTORY_SEPARATOR);
    set_configuration('module/path/lang',       $s_module_path_base.'data'.DIRECTORY_SEPARATOR.'lang'.DIRECTORY_SEPARATOR);
    set_configuration('module/path/web',        $s_module_path_base.'web'.DIRECTORY_SEPARATOR);
}

/**
 * Dispatchs to pages.
 *
 * @return void
 */
function dispatch()
{
    if (cache_include())
        return true;

    $a_parameters = get_uri_parameters();
    $a_page = explode('/', $a_parameters['page']);

    /* are we in back or front area ? */
    $s_application = ($a_page[0] === get_configuration('administration/route')) ? APP_BACK: APP_FRONT;

    set_configuration('app/name', $s_application);
    include get_configuration('app/path/'.$s_application).$s_application.'_core.php';

    return call_user_func($s_application.'_dispatch');

}

/*
 * Fixes the problem of too many slashes when magic_quotes_gpc is ON.
 */
function fix_magic_quotes_gpc()
{
    if (get_magic_quotes_gpc())
    {
        $process = array(&$_GET, &$_POST, &$_COOKIE, &$_REQUEST);
        while (list($key, $val) = each($process))
        {
            foreach ($val as $k => $v)
            {
                unset($process[$key][$k]);
                if (is_array($v))
                {
                    $process[$key][stripslashes($k)] = $v;
                    $process[] = &$process[$key][stripslashes($k)];
                }
                else
                {
                    $process[$key][stripslashes($k)] = stripslashes($v);
                }
            }
        }
        unset($process);
    }
}

/**
 * Includes cache file if exists.
 *
 * @return boolean
 */
function cache_include()
{
    $a_parameters = get_uri_parameters();
    set_configuration('app/name', APP_FRONT);
    $s_cache_file = get_configuration('app/path/cache').str_replace('/', DIRECTORY_SEPARATOR, get_configuration('request/protocol').DIRECTORY_SEPARATOR.get_configuration('server/base_uri_path').(strlen($a_parameters['page']) ? DIRECTORY_SEPARATOR.$a_parameters['page'] : '')).'.cache.php';

    if (file_exists($s_cache_file))
    {
        $r = null; module_chain(__FUNCTION__, $r);
        include $s_cache_file;
        return true;
    }

    return false;
}

/**
 * Returns an element from configuration, or all configuration array (special key 'all').
 * @see config/config.php
 *
 * @param  string  $s_key      Key for configuration.
 * @param  mixed   $m_default  Default value if key is not found.
 * @return mixed
 */
function & get_configuration($s_key = null, $m_default = null)
{
    return get_application_parameter('_app/_configuration/'.$s_key, $m_default);
}

/**
 * Adds an element in configuration.
 *
 * @param  string  $s_key    Element key.
 * @param  mixed   $m_value  Value.
 * @return void
 */
function add_configuration($s_key, $m_value)
{
    $a_conf = get_configuration($s_key, array());
    $a_conf[] = $m_value;
    return set_configuration($s_key, $a_conf);
}

/**
 * Sets an element in configuration.
 *
 * @param  string  $s_key    Element key.
 * @param  mixed   $m_value  Value.
 * @return void
 */
function set_configuration($s_key, $m_value)
{
    return set_application_parameter('_app/_configuration/'.$s_key, $m_value);
}

/**
 * Returns an element from a module configuration..
 *
 * @param  string  $s_module_key  The module key.
 * @param  string  $s_key         Key for configuration.
 * @param  mixed   $m_default     Default value if key is not found.
 * @return mixed
 */
function & get_module_configuration($s_module_key, $s_key = null, $m_default = null)
{
    $m_return = get_configuration($s_key, $m_default);

    if (is_string($m_return)) $m_return = str_replace('$MODULE_KEY$', $s_module_key, $m_return);

    return $m_return;
}

/**
 * Loads configuration from config/config.php.
 *
 * @return void
 */
function load_configuration()
{
    $a_configuration = include get_configuration('app/path/data').'config.php';
    $a_current_configuration = get_application_parameter('_app/_configuration');

    return set_application_parameter('_app/_configuration', array_merge($a_current_configuration, $a_configuration)); //array_merge($a_configuration, $a_current_configuration));
}

/**
 * Indicates if a var exists in templates file.
 *
 * @param  string  $s_var_name  Var name (without '$').
 * @return boolean
 */
function has_page_var($s_var_name)
{
    return has_application_parameter('_page/_template/_vars/'.$s_var_name);
}

/**
 * Returns a template var.
 *
 * @param  string  $s_var_name  Var name (without '$').
 * @param  mixed   $m_default   Default value if var not exists.
 * @return mixed.
 */
function get_page_var($s_var_name, $m_default = null)
{
    return get_application_parameter('_page/_template/_vars/'.$s_var_name, $m_default);
}

/**
 * Returns all template vars.
 *
 * @return array
 */
function get_page_vars()
{
    return get_application_parameter('_page/_template/_vars', array());
}

/**
 * Sets a template var.
 *
 * @param  mixed  $m_var_name  string/array. Var name (without '$') or array of var names with values.
 * @param  mixed  $m_value     Value.
 * @return void
 */
function set_page_var($m_var_name, $m_value = null)
{
    if (is_array($m_var_name))
    {
        foreach ($m_var_name as $s_var_name => $m_value)
            set_application_parameter('_page/_template/_vars/'.$s_var_name, $m_value);
    }
    else
        set_application_parameter('_page/_template/_vars/'.$m_var_name, $m_value);
}

/**
 * Deletes a template var.
 *
 * @param  mixed  $m_var_name  string/array. Var name (without '$') or array of var names.
 * @return void
 */
function delete_page_var($m_var_name)
{
    if (is_array($m_var_name))
        foreach ($m_var_name as $s_var_name)
            delete_application_parameter('_page/_template/_vars/'.$s_var_name);
    else
        delete_application_parameter('_page/_template/_vars/'.$m_var_name);
}

/**
 * Deletes all template vars.
 *
 * @return void
 */
function delete_page_vars()
{
    delete_application_parameter('_page/_template/_vars');
}

/**
 * Indicates if a key is defined.
 *
 * @param  string   $s_key        Key.
 * @param  boolean  $b_not_empty  If true, return false if value is empty.
 * @return boolean
 */
function has_application_parameter($s_key, $b_not_empty = true)
{
    $m_value = get_application_parameter($s_key);
    return $b_not_empty ? null !== $m_value && !empty($m_value) : null !== $m_value;
}

/**
 * Returns an application parameter.
 *
 * @param  string  $s_key      Element key.
 * @param  mixed   $m_default  Returns $m_default if key not exists.
 * @return mixed
 */
function & get_application_parameter($s_key, $m_default = null)
{
    return _get_set_application_parameter(ACTION_GET, $s_key, $m_default);
}

/**
 * Sets an application parameter.
 *
 * @param  string  $s_key     Element key.
 * @param  mixed   $m_value   Value.
 * @return mixed
 */
function set_application_parameter($s_key, $m_value)
{
    return _get_set_application_parameter(ACTION_SET, $s_key, $m_value);
}

/**
 * Removes an application parameter.
 *
 * @param  string  $s_key     Element key.
 */
function delete_application_parameter($s_key)
{
    return _get_set_application_parameter(ACTION_DELETE, $s_key);
}

/**
 * Internal function to manage get and set parameter.
 *
 * @param  integer $i_action  Action (ACTION_GET, ACTION_SET, ACTION_ALL).
 * @param  string  $s_key     Element key.
 * @param  mixed   $m_value   Value.
 * @return mixed
 */
function & _get_set_application_parameter($i_action, $s_key = null, $m_value = null)
{
    static $a_parameters = array();

    $s_key = $s_key ? preg_replace('~/+~', '/', $s_key) : null;

    switch($i_action)
    {
        case ACTION_GET:
            if ('all' === $s_key)
                return $a_parameters;

            return array_get_from_key($s_key, $a_parameters, $m_value);

        case ACTION_SET:
            if ($s_key)
            {
                $a_keys = explode('/', $s_key);
                $i_limit = count($a_keys) - 1;
                $a = array();
                $b = & $a;
                foreach ($a_keys as $i_index => $s_sub_key)
                {
                    $b[$s_sub_key] = ($i_index < $i_limit) ? array() : $m_value;
                    if ($i_index < $i_limit)
                        $b = & $b[$s_sub_key];
                }
                $a_parameters = array_deep_merge($a_parameters, $a);
            }
            break;

        case ACTION_DELETE:
            if ($s_key)
            {
                $a_keys = explode('/', $s_key);
                $i_limit = count($a_keys) - 1;
                $a_temp_conf = & $a_parameters;

                foreach ($a_keys as $i_index => $s_sub_key)
                {
                    if (!array_key_exists($s_sub_key, $a_temp_conf))
                        break;

                    if ($i_index === $i_limit) {
                        unset($a_temp_conf[$s_sub_key]);
                        break;
                    }

                    $a_temp_conf = & $a_temp_conf[$s_sub_key];
                }
            }
            break;
    }

    $m_return = null; // because return by reference
    return $m_return;
}

/**
 * Returns a value from array from a deep key.
 *
 * @param  string  $s_key     Element key.
 * @param  mixed   $m_value   Value.
 * @return mixed
 */
function & array_get_from_key($s_key, & $a_array, $m_default = null)
{
    $m_return = null;

    if ($s_key)
    {
        // manage key with "/" separator for levels in array.
        //   ie : array('key1' => array('key2' => 'value')), get_configuration('key1/key2') returns 'value'
        $a_keys = explode('/', rtrim($s_key, '/'));
        $i_limit = count($a_keys) - 1;
        $a_temp_conf = & $a_array;

        foreach ($a_keys as $i_index => $s_sub_key)
        {
            if (!array_key_exists($s_sub_key, $a_temp_conf))
                break;

            if ($i_index === $i_limit)
            {
                $m_return = null !== $a_temp_conf[$s_sub_key] ? $a_temp_conf[$s_sub_key] : $m_default;
                return $m_return; // because return by reference
            }

            $a_temp_conf = & $a_temp_conf[$s_sub_key];
        }
    }

    return $m_default;
}

/*
 * code from php at moechofe dot com (array_merge comment on php.net)
 * array_deep_merge
 *
 * array array_deep_merge ( array array1 [, array array2 [, array ...]] )
 *
 * Like array_merge
 *
 *   array_deep_merge() merges the elements of one or more arrays together so
 * that the values of one are appended to the end of the previous one. It
 * returns the resulting array.
 *   If the input arrays have the same string keys, then the later value for
 * that key will overwrite the previous one. If, however, the arrays contain
 * numeric keys, the later value will not overwrite the original value, but
 * will be appended.
 *   If only one array is given and the array is numerically indexed, the keys
 * get reindexed in a continuous way.
 *
 * Different from array_merge
 *   If string keys have arrays for values, these arrays will merge recursively.
 */
function array_deep_merge()
{
    switch (func_num_args())
    {
        case 0:
            return false;

        case 1:
            return func_get_arg(0);

        case 2:
            $a_args = func_get_args();
            $a_args[2] = array();

            if (is_array($a_args[0]) && is_array($a_args[1]))
            {
                foreach (array_unique(array_merge(array_keys($a_args[0]), array_keys($a_args[1]))) as $s_key)
                {
                    $b_is_key_0 = array_key_exists($s_key, $a_args[0]);
                    $b_is_key_1 = array_key_exists($s_key, $a_args[1]);

                    if ($b_is_key_0 && $b_is_key_1 && is_array($a_args[0][$s_key]) && is_array($a_args[1][$s_key]))
                        $a_args[2][$s_key] = array_deep_merge($a_args[0][$s_key], $a_args[1][$s_key]);
                    else if ($b_is_key_0 && $b_is_key_1)
                        $a_args[2][$s_key] = $a_args[1][$s_key];
                    else if (!$b_is_key_1)
                        $a_args[2][$s_key] = $a_args[0][$s_key];
                    else if (!$b_is_key_0)
                        $a_args[2][$s_key] = $a_args[1][$s_key];
                }
                return $a_args[2];
            }
            else
                return $a_args[1];
        default :
            $a_args = func_get_args();
            $a_args[1] = array_deep_merge($a_args[0], $a_args[1]);
            array_shift($a_args);
            return call_user_func_array('array_deep_merge', $a_args);
            break;
    }
}

/**
 * Ups a value in an array.
 *
 * @param  array   $a      The array.
 * @param  string  $s_val  The value;
 * @return void
 */
function array_up(& $a, $s_val)
{
    if (false !== ($i_index = array_search($s_val, $a)) && $i_index > 0)
    {
        $a_before = array_slice($a, 0, $i_index); // part defore value
        $a_after = array_slice($a, $i_index + 1, count($a) - $i_index); // part after values

        $a_permute = array($a[$i_index], array_pop($a_before)); // permuted values

        $a = array_merge($a_before, $a_permute, $a_after);
    }
}
function array_kup(& $a, $s_key)
{
    $a_1 = array_keys($a);
    $a_2 = array_values($a);

    if (false !== ($i_index = array_search($s_key, $a_1)) && $i_index > 0)
    {
        $a_before = array_slice($a_1, 0, $i_index); // part defore value
        $a_after = array_slice($a_1, $i_index + 1, count($a_1) - $i_index); // part after values
        $a_permute = array($a_1[$i_index], array_pop($a_before)); // permuted values

        $a_1 = array_merge($a_before, $a_permute, $a_after);

        $a_before = array_slice($a_2, 0, $i_index); // part defore value
        $a_after = array_slice($a_2, $i_index + 1, count($a_2) - $i_index); // part after values
        $a_permute = array($a_2[$i_index], array_pop($a_before)); // permuted values

        $a_2 = array_merge($a_before, $a_permute, $a_after);

        $a = array_combine($a_1, $a_2);
    }
}

/**
 * Downs a value in an array.
 *
 * @param  array   $a      The array.
 * @param  string  $s_val  The value;
 * @return void
 */
function array_down(& $a, $s_val)
{
    if (false !== ($i_index = array_search($s_val, $a)) && $i_index < count($a) - 1)
    {
        $a_before = array_slice($a, 0, $i_index); // part defore value
        $a_after = array_slice($a, $i_index + 1, count($a) - $i_index); // part after values

        $a_permute = array(array_shift($a_after), $a[$i_index]); // permutted values

        $a = array_merge($a_before, $a_permute, $a_after);
    }
}
function array_kdown(& $a, $s_key)
{
    $a_1 = array_keys($a);
    $a_2 = array_values($a);

    if (false !== ($i_index = array_search($s_key, $a_1)) && $i_index < count($a_1) - 1)
    {
        $a_before = array_slice($a_1, 0, $i_index); // part defore value
        $a_after = array_slice($a_1, $i_index + 1, count($a_1) - $i_index); // part after values
        $a_permute = array(array_shift($a_after), $a_1[$i_index]); // permutted values

        $a_1 = array_merge($a_before, $a_permute, $a_after);

        $a_before = array_slice($a_2, 0, $i_index); // part defore value
        $a_after = array_slice($a_2, $i_index + 1, count($a_2) - $i_index); // part after values
        $a_permute = array(array_shift($a_after), $a_2[$i_index]); // permutted values

        $a_2 = array_merge($a_before, $a_permute, $a_after);

        $a = array_combine($a_1, $a_2);
    }
}

/**
 * Removes value in an array.
 *
 * @param  array   $a      The array.
 * @param  string  $s_val  The value;
 * @return void
 */
function array_remove(& $a, $s_val)
{
    if (false !== ($i_index = array_search($s_val, $a)))
        $a = array_merge(array_slice($a, 0, $i_index), array_slice($a, $i_index + 1, count($a) - $i_index));
}

/*
 * code from arvin at sudocode dot net (glob comment on php.net)
 */
function bfglob($s_path, $s_pattern = '*', $i_flags = 0, $i_depth = 0) {
    $a_matches = array();
    $a_folders = array(rtrim($s_path, DIRECTORY_SEPARATOR));

    while ($s_folder = array_shift($a_folders))
    {
        $a_matches = array_merge($a_matches, x_glob($s_folder.DIRECTORY_SEPARATOR, $s_pattern, $i_flags));
        if ($i_depth != 0)
        {
            $a_more_folders = x_glob($s_folder.DIRECTORY_SEPARATOR, '*', 1);
            $i_depth   = ($i_depth <= -1) ? -1 : $i_depth + count($a_more_folders) - 2;
            $a_folders = array_merge($a_folders, $a_more_folders);
        }
    }

    return $a_matches;
}

/*
 * flags :
 *  - 1 only dir
 *  - 2 only files
 */
function x_glob($s_path, $s_pattern = '*', $i_flags = 0)
{
    $s_pattern = str_replace(array('.', '*'), array('\.', '.*'), $s_pattern);
    if (!$i_flags) $i_flags = 3;

    $a_files = array();
    if (is_dir($s_path))
    {
        if ($r_dh = opendir($s_path))
        {
            while (false !== ($s_file = readdir($r_dh)))
            {
                if ('.' !== $s_file && '..' !== $s_file && preg_match('/' . $s_pattern . '/', $s_file))
                {
                    if ((1 === ($i_flags & 1)) && (is_dir($s_path . $s_file))) $a_files[] = $s_path . $s_file;
                    if ((2 === ($i_flags & 2)) && (is_file($s_path . $s_file))) $a_files[] = $s_path . $s_file;
                }
            }
        }
    }
    //sort($a_files);
    natcasesort($a_files);
    return array_unique($a_files);
}

/**
 * Loads a language file.
 *
 * @param  string  $s_lang        Language code.
 * @param  string  $s_module_key  Module key.
 * @return void
 */
function load_language($s_lang = null, $s_module_key = null)
{
    $s_file = $s_module_key ? get_module_configuration($s_module_key, 'module/path/lang').$s_lang.'.php' : get_configuration('app/path/lang').DIRECTORY_SEPARATOR.$s_lang.'.php';

    set_configuration('app/lang', (file_exists($s_file) ? array_merge(get_configuration('app/lang', array()), include $s_file) : array()));
}

/**
 * Returns a string translation for a key.
 * First param must be the key. Others params can be variables.
 *
 * @return string
 */
function __()
{
    $a_args = func_get_args();

    switch(func_num_args())
    {
        case 0:
            return ''; break;

        case 1:
            $a_lang = & get_configuration('app/lang', array());
            return array_key_exists($a_args[0], $a_lang) ? $a_lang[$a_args[0]] : $a_args[0];
            break;

        default:
            $a_lang = & get_configuration('app/lang', array());
            $s_key = array_shift($a_args);
            return array_key_exists($s_key, $a_lang) ? vsprintf($a_lang[$s_key], $a_args) : vsprintf($s_key, $a_args);
            break;
    }
}

/**
 * Returns the current uri.
 *
 * @param  boolean  $b_lowercase  Indicates if uri must be lowercased.
 * @return string
 */
function get_current_uri($b_lowercase = false)
{
    $s_request_uri = (!isset($_SERVER['REQUEST_URI'])) ? $_SERVER['PHP_SELF'] : $_SERVER['REQUEST_URI'];
    $s_protocol = get_configuration('request/protocol');
    return !$b_lowercase ? $s_protocol.'://'.$_SERVER['SERVER_NAME'].$s_request_uri : strtolower($s_protocol.'://'.$_SERVER['SERVER_NAME'].$s_request_uri);
}

/**
 * Returns parameters from current url.
 * http://www.mydomain.com/page_name?var1=val1&var2=val2
 * array('page' => 'page_name',
 *       'parameters' => array(
 *          'var1' => 'val1',
 *          'var2' => 'val2'
 *      )
 * )
 *
 * @return array
 */
function & get_uri_parameters()
{
    static $a_parameters = array('page' => null, 'is_secure' => false, 'parameters' => array());
    static $b_loaded = false;

    if (!$b_loaded)
    {
        $a_parameters['is_secure'] = (array_key_exists('HTTPS', $_SERVER) && 'on' === $_SERVER['HTTPS']) ||
                                     (array_key_exists('SCRIPT_URI', $_SERVER) && 'https' === strtolower(substr($_SERVER['SCRIPT_URI'], 0, 5))) ||
                                     (array_key_exists('SERVER_PROTOCOL', $_SERVER) && 'https' === strtolower(substr($_SERVER['SERVER_PROTOCOL'], 0, 5)));
        set_configuration('request/protocol', $a_parameters['is_secure'] ? 'https' : 'http');

        // parse query string
        $a_parameters['parameters'] = $_GET;
        $s_page_uri = str_replace(get_configuration('server/base_uri_path'), '', strtolower($_SERVER['HTTP_HOST'].$_SERVER['REQUEST_URI']));
        if (false !== ($i_pos = strpos($s_page_uri, '?')))
            $s_page_uri = substr($s_page_uri, 0, $i_pos);

        // parse page
        if (get_configuration('server/url_rewriting', false))
            $a_parameters['page'] = trim($s_page_uri, '/');
        else
            $a_parameters['page'] = (array_key_exists($s_page_var_name = get_configuration('server/page_var_name', 'p'), $a_parameters['parameters'])) ? $a_parameters['parameters'][$s_page_var_name] : '';

        $b_loaded = true;
    }

    return $a_parameters;
}

/**
 * Removes a key from parameters.
 *
 * @param  string   $s_key      Element key.
 * @return void
 */
function remove_request_parameter($s_key)
{
    get_request_parameter($s_key, null, true);
}

/**
 * Returns a value from request parameters.
 *
 * @param  string   $s_key      Element key.
 * @param  mixed    $m_default  Returns $m_default if key not exists.
 * @param  boolean  $b_remove   Indicates if key must be removed from parameters.
 * @return mixed
 */
function get_request_parameter($s_key = null, $m_default = null, $b_remove = false)
{
    static $a_parameters = array();
    static $b_loaded = false;

    if (!$b_loaded)
    {
        if ('GET' === get_request_method())
            $a_parameters = $_GET;
        else
            $a_parameters = array_merge($_GET, $_POST);
        $b_loaded = true;
    }
    if ($b_remove && array_key_exists($s_key, $a_parameters))
    {
        unset($a_parameters[$s_key]);
    }

    return !$s_key ? $a_parameters : array_get_from_key($s_key, $a_parameters, $m_default);
}

/**
 * Returns request method ('POST', 'GET').
 *
 * @return string
 */
function get_request_method()
{
    return $_SERVER['REQUEST_METHOD'];
}

/**
 * Returns an internal uri for a page.
 *
 * @param  string  $s_page         Page key.
 * @param  array   $a_parameters   Uri parameters.
 * @param  string  $s_application  Application (APP_FRONT, APP_BACK).
 * @return string
 */
function get_uri($s_page = null, $a_parameters = array(), $s_application = null, $s_var_separator = '&amp;')
{
    $s_page = $s_page ? $s_page : get_configuration('app/page');
    $s_application = $s_application ? $s_application : get_configuration('app/name');

    return call_user_func('get_'.$s_application.'_uri', $s_page, $a_parameters, null, $s_var_separator);
}

/**
 * Returns an internal uri for a back office page.
 *
 * @param  string  $s_page        Page key.
 * @param  array   $a_parameters  Uri parameters.
 * @return string
 */
function get_back_uri($s_page, $a_parameters = array(), $s_inner_ref = null, $s_var_separator = '&amp;')
{
    $b_is_module = 'modules/' === substr($s_page, 0, 8);
    $a_routes = get_data('back_routes', array());

    if (!$s_page || (!array_key_exists($s_page, $a_routes) && !$b_is_module))
    {
        echo __('ErrRouteNotExists', $s_page);
        exit;
    }

    $b_has_rewriting_mode = get_configuration('server/url_rewriting', false);

    if ($a_parameters && array_key_exists('action', $a_parameters) && isset($_COOKIE['session_token'])) $a_parameters['session_token'] = $_COOKIE['session_token'];

    $s_url = sprintf('%s/%s',
                get_configuration('server/base_uri_path'),
                (!$b_has_rewriting_mode ? get_configuration('server/script_name') . '?' .get_configuration('server/page_var_name', 'p').'=' : '').
                get_configuration('administration/route').'/'.
                $s_page);

    if (count($a_parameters))
        $s_url .= (!$b_has_rewriting_mode ? $s_var_separator : '?').http_build_query($a_parameters, '', $s_var_separator);

    return 'http'.(get_configuration('administration/is_secure') ? 's' : '').'://'.$s_url . ($s_inner_ref ? '#' . $s_inner_ref : '');
}

/**
 * Returns an internal uri for a front office page.
 *
 * @param  string  $s_page_id     Page key.
 * @param  array   $a_parameters  Uri parameters.
 * @return string
 */
function get_front_uri($s_page_id, $a_parameters = array(), $s_inner_ref = null, $s_var_separator = '&amp;')
{
    $a_routes = get_data('front_routes_flip', array());

    if (!$s_page_id || !array_key_exists($s_page_id, $a_routes))
        return false;

    $s_page = $a_routes[$s_page_id]['route'];

    $b_has_rewriting_mode = get_configuration('server/url_rewriting', false);

    $s_url = sprintf('http%s://%s/%s',
                ($a_routes[$s_page_id]['is_secure'] ? 's' : ''),
                get_configuration('server/base_uri_path'),
                (!empty($s_page) && !$b_has_rewriting_mode ? get_configuration('server/script_name') . '?' .get_configuration('server/page_var_name', 'p').'=' : '').
                $s_page);

    if (count($a_parameters))
        $s_url .= (!$b_has_rewriting_mode ? $s_var_separator : '?').http_build_query($a_parameters, '', $s_var_separator);

    if ($s_inner_ref)
        $s_url .= '#' . $s_inner_ref;

    return $s_url;
}

/**
 * Indicates if current uri is securised (used https).
 *
 * @return boolean
 */
function current_uri_is_securised()
{
    $a_parameters = get_uri_parameters();
    return $a_parameters['is_secure'];
}

/**
 * Redirection to another page.
 *
 * @param  string   $s_page         Internal page name.
 * @param  integer  $i_status_code  HTTP status code.
 * @param  array    $a_parameters   Uri parameters.
 * @param  string   $s_application  Application name (APP_BACK, APP_FRONT).
 * @return void
 */
function redirect($s_page, $i_status_code = 302, $a_parameters = array(), $s_application = null)
{
    $s_application = $s_application ? $s_application : get_configuration('app/name');
    $s_page = preg_match('~^https?://~', $s_page) ? $s_page : call_user_func('get_'.$s_application.'_uri', $s_page, $a_parameters, null, '&');
    header('Status: '.$i_status_code, 0, $i_status_code);
    header('Location: '.$s_page);
    exit;
}

/**
 * Displays the page 404.
 *
 * @return void
 */
function error_404()
{
    $r = null;module_chain(__FUNCTION__, $r);
    header('HTTP/1.0 404 Not Found');
    $a_routes = get_data('front_routes');

    if (array_key_exists('404', $a_routes))
    {
        set_configuration('app/page', $a_routes['404']['id']);
        if (STRUCTURE_STATE_VISIBLE === intval(get_page_property('state', STRUCTURE_STATE_HIDDEN))) { display_page(); exit; }
    }
    echo __('Err404');
    exit;
}

/**
 * Load helpers.
 *
 * @param  mixed  $m_helper_path  string/array  Helper name or list of helpers names.
 * @return void
 */
function load_helpers($m_helper_path)
{
    /* [application]/[module]/<name> */
    $s_app = get_configuration('app/name');
    foreach ((array)$m_helper_path as $s_helper_path)
    {
        $a_path_temp = explode('/', $s_helper_path);
        $a_path = array('application' => $s_app, 'module' => null, 'name' => null);

        switch (count($a_path_temp))
        {
            case 1:
                $a_path['module'] = get_configuration('app/path/helpers');
                $a_path['name'] = $a_path_temp[0];
                break;

            case 2:
                $a_path['application'] = 'data' . DIRECTORY_SEPARATOR . 'helpers' . DIRECTORY_SEPARATOR . $a_path['application'];
                $a_path['module'] = get_configuration('app/path/modules') . DIRECTORY_SEPARATOR . $a_path_temp[0];
                $a_path['name'] = $a_path_temp[1];
                break;

            case 3:
                $a_path['application'] = 'data' . DIRECTORY_SEPARATOR . 'helpers' . DIRECTORY_SEPARATOR . $a_path_temp[0];
                $a_path['module'] = get_configuration('app/path/modules') . DIRECTORY_SEPARATOR . $a_path_temp[1];
                $a_path['name'] = $a_path_temp[2];
                break;

            default:
                echo __('ErrHelperNotFound', $s_helper_path);
                exit;
        }

        $s_path = $a_path['module'] . DIRECTORY_SEPARATOR . $a_path['application'] . DIRECTORY_SEPARATOR . $a_path['name'] . '.php';

        if (!is_file($s_path))
        {
            echo __('ErrHelperNotFound', $s_path);
            exit;
        }

        include_once $s_path;
    }
}

/**
 * Displays page.
 *
 * @return void
 */
function display_page()
{
    if (STRUCTURE_STATE_HIDDEN === intval(get_page_property('state', STRUCTURE_STATE_HIDDEN)))
        error_404();

    $s_page_type = get_page_property('type');
    $s_page_type_mime = get_page_property('type_mime');
    if ('xml' === $s_page_type || 'xml' === $s_page_type_mime) header('content-type: text/xml; charset=UTF-8');
    else if ('html' ===  $s_page_type || 'xhtml' === $s_page_type_mime) header('content-type: text/html; charset=UTF-8');

    $r = null; module_chain(__FUNCTION__, $r);
    if (null !== ($a_vars = get_application_parameter('_page/_template/_vars'))) extract($a_vars);
    $s_layout_file = get_configuration('app/path/layouts').get_configuration('app/name').DIRECTORY_SEPARATOR.get_page_property('layout', (APP_BACK === get_configuration('app/name') ? 'layout' : null)).'.php';

    if (is_file($s_layout_file)) {
        ob_start();
        include $s_layout_file;
        echo fix_css_js(ob_get_clean());
    }
    else echo get_page_content();
}

/**
 * Returns page content.
 *
 * @return string
 */
function get_page_content()
{
    $s_content = null;
    $s_content_file = get_configuration('app/path/contents').get_configuration('app/name').DIRECTORY_SEPARATOR.get_configuration('app/page').'.php';

    if (!is_file($s_content_file))
    {
        echo __('ErrFileNotRead', $s_content_file);
        exit;
    }

    if (null !== ($a_vars = get_application_parameter('_page/_template/_vars'))) extract($a_vars);

    if (APP_BACK === get_configuration('app/name'))
    {
        ob_start();
        include $s_content_file;
        $s_content = ob_get_clean();
    }
    else
    {
        // change internal-page, internal-image, internal-document with real path
        $s_content = fix_internals_uri(file_get_contents($s_content_file));

        if (in_array(get_page_property('type'), array('html', 'xml'))) // @todo : place it in module
        {
            $s_blocks = 'noscript|noframes|div|p|h1|h2|h3|h4|h5|h6|ul|ol|menu|dir|li|dl|dt|dd|address|hr|pre|blockquote|center|map|area|form|fieldset|legend|isindex|table|caption|thead|tfoot|tbody|colgroup|col|tr|th|td';
            $s_content = preg_replace('/('.$s_blocks.')>\s+<('.$s_blocks.')/smi', '$1><$2', $s_content); // removes blanks between blocks for display: inline-block.
        }
        else
        {
            ob_start();
            eval('?>'.$s_content);
            $s_content = ob_get_clean();
        }
    }

    $s_content = fix_cms_var($s_content);
    module_chain(__FUNCTION__, $s_content);
    return $s_content;
}

function fix_css_js($s_content)
{
    return str_replace(
        array('<!-- beCms:css -->', '<!-- beCms:js -->'),
        array(_get_page_css(), _get_page_js()),
        $s_content
    );
}

/**
 * Replaces specifics inners uri (internal-page:...) with real uri in a string.
 *
 * @param  string  $s_content  Content to change.
 * @return string
 */
function fix_internals_uri($s_content)
{
    if (!strlen($s_content)) return $s_content;

    return preg_replace_callback(
        '/<(a|form)\s*(.*?)\s*(href|action)=[\'"]+?(internal-page\:([\w\d\.]+)(#([\w\d\.\-\_]+))?)[\'"]+?\s*(.*?)\s*>\s*/im',
        create_function(
            '$matches',
            'return str_replace($matches[4], get_front_uri($matches[5], array(), (!empty($matches[6]) ? substr($matches[6], 1) : null)), $matches[0]);'
        ),
        $s_content
    );
}

/**
 * Replaces specifics cms vars in a string.
 *
 * @param  string  $s_content  Content to change.
 * @return string
 */
function fix_cms_var($s_content)
{
    return str_replace(
            array('$PROTOCOL$', '$BASE_URI_RESSOURCES_PATH$', '$BASE_URI_PATH$'),
            array(get_configuration('request/protocol'), get_configuration('server/base_uri_ressources_path'), get_configuration('server/base_uri_path')),
            $s_content);
}


/**
 * Returns page doctype.
 *
 * @return string
 */
function get_page_doctype()
{
    return 'xhtml-1.0-strict' === get_page_property('doctype') ?
            '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">' :
            '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">';
}

/**
 * Returns page title.
 *
 * @param  boolean  $b_html_encode  Encoding specials chars in html if true. Default true.
 * @return string
 */
function get_page_title($b_html_encode = true)
{
    return $b_html_encode ? _htmlentities(get_page_property('title')) : get_page_property('title');
}

/**
 * Returns site title.
 *
 * @param  boolean  $b_html_encode  Encoding specials chars in html if true. Default true.
 * @return string
 */
function get_site_title($b_html_encode = true)
{
    return $b_html_encode ? _htmlentities(get_configuration('misc/site/title')) : get_configuration('misc/site/title');
}

/**
 * Returns formated title.
 *
 * @param  boolean  $b_html_encode  Encoding specials chars in html if true. Default true.
 * @return string
 */
function get_title($b_html_encode = true)
{
    if ($s_title_format = get_configuration('misc/site/title_format'))
    {
        return $b_html_encode ? _htmlentities(str_replace(array('%site-title%', '%page-title%'), array(get_site_title(false), get_page_title(false)), $s_title_format)) :
                                str_replace(array('%site-title%', '%page-title%'), array(get_site_title(false), get_page_title(false)), $s_title_format);
    }
    return null;
}

/**
 * Includes menu.
 *
 * @return void
 */
function get_page_menu()
{
    load_helpers('defaults');
    return call_user_func('get_'.get_configuration('app/name').'_menu');
}

/**
 * Return lines for including css.
 * If css file path contains /(all|braille|embossed|handheld|print|projection|screen|speech|tty|tv)/ in the path, the value is used for the "media" attribute, otherwise "all" is used.
 *
 * return string
 */
function get_page_css()
{
    return '<!-- beCms:css -->';
}

/*
 * real get_page_css().
 */
function _get_page_css()
{
    $a_stylesheets = array();
    $s_protocol = get_configuration('request/protocol');

    $a_css = array_unique(array_merge(get_page_property('css_files', array()), get_configuration('ressources/css', array())));
    module_chain('get_page_css', $a_css);

    if (count($a_css))
    {
        foreach ($a_css as $s_css_file)
        {
            if ($s_css_file && strlen($s_css_file))
            {
                $s_media = 'all';
                $a_media = array();
                if (preg_match('~/(all|braille|embossed|handheld|print|projection|screen|speech|tty|tv)/~', $s_css_file, $a_media)) $s_media = $a_media[1];
                $a_stylesheets[] = sprintf('<link rel="stylesheet" type="text/css" href="%s" media="'.$s_media.'" />', $s_protocol.'://'.get_configuration('server/base_uri_ressources_path').$s_css_file. ('php' === strtolower(substr($s_css_file, -3)) ? '?uri='.$s_protocol.'://'.get_configuration('server/base_uri_ressources_path') : ''));
            }
        }

        return implode("\n", $a_stylesheets)."\n";
    }

    return '';
}

/**
 * Adds one or many css files.
 *
 * @param  mixed   $m_css_file     string / array, css files or list of css files.
 * @parap  string  $s_module_name  Module name.
 * @return void
 */
function add_css($m_css_file, $s_module_name = null)
{
    $a_css_file = (array)$m_css_file;
    foreach ($a_css_file as $i_index => $s_css_file) {
        if ($s_module_name) $s_css_file = 'modules/' . $s_module_name . '/' . $s_css_file;
        if ('/css/' !== substr($s_css_file, 0, 5)) $s_css_file = '/css/' . get_configuration('app/name') . '/' . $s_css_file;
        $a_css_file[$i_index] = $s_css_file;
    }
    set_configuration('ressources/css', array_unique(array_merge(get_configuration('ressources/css', array()), $a_css_file)));
}

/**
 * Return lines for including javascripts.
 *
 * @return string
 */
function get_page_js()
{
    return '<!-- beCms:js -->';
}

/*
 * real get_page_js().
 */
function _get_page_js()
{
    $a_scripts = array();
    $s_protocol = get_configuration('request/protocol');

    $a_js = array_unique(array_merge(get_page_property('js_files', array()), get_configuration('ressources/js', array())));
    module_chain('get_page_js', $a_js);

    if (count($a_js))
    {
        foreach ($a_js as $s_js_file)
        {
            if ($s_js_file && strlen($s_js_file))
                $a_scripts[] = sprintf('<script type="text/javascript" src="%s"></script>', $s_protocol.'://'.get_configuration('server/base_uri_ressources_path').$s_js_file. ('php' === strtolower(substr($s_js_file, -3)) ? '?uri='.$s_protocol.'://'.get_configuration('server/base_uri_ressources_path') : ''));
        }

        return implode("\n", $a_scripts)."\n";
    }

    return '';
}

/**
 * Adds one or many js files.
 *
 * @param  mixed   $m_js_file      string / array, js files or list of js files.
 * @param  string  $s_module_name  Module name.
 * @return void
 */
function add_js($m_js_file, $s_module_name = null)
{
    $a_js_file = (array)$m_js_file;
    foreach ($a_js_file as $i_index => $s_js_file) {
        if ($s_module_name) $s_js_file = 'modules/' . $s_module_name . '/' . $s_js_file;
        if ('/js/' !== substr($s_js_file, 0, 4)) $s_js_file = '/js/' . get_configuration('app/name') . '/' . $s_js_file;
        $a_js_file[$i_index] = $s_js_file;
    }
    set_configuration('ressources/js', array_unique(array_merge(get_configuration('ressources/js', array()), $a_js_file)));
}

/**
 * Return lines for including meta informations.
 *
 * @return string
 */
function get_page_meta()
{
    $a_metadata = array();

    if ($a_meta = get_page_property('metadata'))
    {
        foreach ($a_meta as $s_key => $s_meta)
        {
            if ($s_meta && strlen($s_meta))
                $a_metadata[] = sprintf('<meta name="%s" content="%s" />', $s_key, _htmlentities($s_meta));
        }

        return implode("\n", $a_metadata)."\n";
    }

    return '';
}

/**
 * Returns the body tag.
 *
 * @param  array  $a_attributes  Body tag attributes. An associative array ('id' => 'body-id', 'class' => 'body-class1 body-class2', ...).
 * @return string
 */
function get_page_body_tag($a_attributes = array())
{
    if (('' !== ($s_id = get_page_property('body_id')) && $s_id) && !array_key_exists('id', $a_attributes)) $a_attributes['id'] = $s_id;
    if ('' !== ($s_classes = get_page_property('body_classes')) && $s_classes)
    {
        if (!array_key_exists('class', $a_attributes)) $a_attributes['class'] = $s_classes;
        else
        {
            $a_attr_classes = preg_split('/\s+/', $a_attributes['class']);
            $a_declared_classes = preg_split('/\s+/', $s_classes);
            $a_attributes['class'] = implode(' ', array_merge($a_attr_classes, $a_declared_classes));
        }
    }

    if (array_key_exists('class', $a_attributes)) $a_attributes['class'] = implode(' ', array_unique(preg_split('/\s+/', $a_attributes['class'])));

    $a_body = array();
    foreach ($a_attributes as $s_attr => $s_value) $a_body[] = $s_attr . '="' . $s_value . '"';
    return '<body'.(count($a_body) ? ' ' . implode(' ', $a_body) : '').'>';
}

/**
 * Returns a property for current page.
 *
 * @param  string  $s_key      Property key.
 * @param  mixed   $m_default  Default value if property key not found.
 * @return mixed
 */
function get_page_property($s_key, $m_default = null)
{
    static $s_page = null;

    if (!$s_page || $s_page !== get_configuration('app/page'))
    {
        $s_properties_file = get_configuration('app/path/contents').get_configuration('app/name').DIRECTORY_SEPARATOR.get_configuration('app/page').'.properties.php';

        if (!is_file($s_properties_file))
        {
            echo __('ErrFileNotRead', $s_properties_file);
            exit;
        }

        set_configuration('app/page-properties', include $s_properties_file);
        $s_page = get_configuration('app/page');
    }

    return get_configuration('app/page-properties/'.$s_key, $m_default);
}


/**
 * Returns site's structure.
 *
 * @return array
 */
function get_structure()
{
    return get_data('structure');
}

/**
 * Convert string with htmlentities.
 *
 * @param  string  $s_text.
 * @return string
 */
function _htmlentities($s_text)
{
    return htmlspecialchars($s_text, ENT_COMPAT | ENT_QUOTES, 'UTF-8');
}

function module_chain($s_function, & $m_data)
{
    if ($a_data = get_data('modules'.DIRECTORY_SEPARATOR.get_configuration('app/name').DIRECTORY_SEPARATOR.'chains'.DIRECTORY_SEPARATOR.$s_function))
    {
        foreach ($a_data as $s_module_key => $s_function)
        {
            if (file_exists($s_file = get_module_configuration($s_module_key, 'module/path/'.get_configuration('app/name')) . 'chain.php'))
            {
                if (!in_array($s_file, get_included_files()))
                {
                    include $s_file;
                    $a_loaded_files[] = $s_file;
                }
                $s_function($m_data);
            }
        }
    }
    return $m_data;
}

/**
 * Returns an unique id.
 *
 * @param  boolean  $b_rand   Adds characters at end of id.
 * @param  boolean  $b_sleep  Wait 1 microsecond after id generation.
 * @return string
 */
function generate_id($b_rand = true, $b_sleep = false)
{
    return uniqid('', true);
}

/**
 * Returns data file values.
 *
 * @param  string  $s_data  Data file basename.
 * @return mixed   array / false.
 */
function get_data($s_data, $m_default = false)
{
    $s_data_file = get_configuration('app/path/data').$s_data.'.php';

    if (file_exists($s_data_file))
        return include $s_data_file;

    return $m_default;
}

/**
 * Writes a data file.
 *
 * @param  array   $a_data  Data.
 * @param  string  $s_data  Data file basename.
 * @return boolean
 */
function write_data($a_data, $s_data)
{
    return write_file(get_configuration('app/path/data').$s_data.'.php', '<?php'."\n".'return '.var_export($a_data, true).';');
}

/**
 * Returns module data file values.
 *
 * @param  string  $s_module_key  The module key.
 * @param  string  $s_data        Data file basename.
 * @return mixed   array / false.
 */
function get_module_data($s_module_key, $s_data, $m_default = false)
{
    $s_data_file = get_module_configuration($s_module_key, 'module/path/data').$s_data.'.php';

    if (file_exists($s_data_file))
        return include $s_data_file;

    return $m_default;
}

/**
 * Writes a module data file.
 *
 * @param  string  $s_module_key  The module key.
 * @param  array   $a_data        Data.
 * @param  string  $s_data        Data file basename.
 * @return boolean
 */
function write_module_data($s_module_key, $a_data, $s_data)
{
    return write_file(get_module_configuration($s_module_key, 'module/path/data').$s_data.'.php', '<?php'."\n".'return '.var_export($a_data, true).';');
}

/**
 * Writes a file and fix rights.
 *
 * @param  string  $s_file     File path (with name).
 * @param  string  $s_content  File content.
 * @param  string  $s_access   Access mode to write file. Default is 'w'.
 * @param  number  $i_mode     Mode number.
 * @return boolean
 */
function write_file($s_file, $s_content, $s_access = 'w', $i_mode = null)
{
    if (!$i_mode) $i_mode = octdec(get_configuration('server/filemode'));

    $i_current_umask = umask();
    umask(0000);
    if (false === ($r_fp = @fopen($s_file, $s_access))) {
        if (function_exists('set_message')) set_message('error', __('ErrFileNotWrite', $s_file));
        return false;
    }
    fputs($r_fp, $s_content);
    fclose($r_fp);
    @chmod($s_file, $i_mode);
    @umask($i_current_umask);
    return true;
}

/**
 * Renames a file and fix rights.
 *
 * @param  string  $s_from     File to rename.
 * @param  string  $s_to       New name.
 * @param  number  $i_mode     Mode number.
 * @return boolean
 */
function rename_file($s_from, $s_to, $i_mode = null)
{
    if (!$i_mode) $i_mode = octdec(get_configuration('server/filemode'));

    $i_current_umask = umask();
    umask(0000);
    if (false === @rename($s_from, $s_to)) {
        if (function_exists('set_message')) set_message('error', __('ErrFileRename', $s_from, $s_to));
        return false;
    }
    @chmod($s_to, $i_mode);
    @umask($i_current_umask);
    return true;
}

/**
 * Copy a file and fix rights.
 *
 * @param  string  $s_from     File to copy.
 * @param  string  $s_to       Destination.
 * @param  number  $i_mode     Mode number.
 * @return boolean
 */
function copy_file($s_from, $s_to, $i_mode = null)
{
    if (!$i_mode) $i_mode = octdec(get_configuration('server/filemode'));

    $i_current_umask = umask();
    umask(0000);
    if (false === @copy($s_from, $s_to)) {
        if (function_exists('set_message')) set_message('error', __('ErrFileCopy', $s_from, $s_to));
        return false;
    }
    @chmod($s_to, $i_mode);
    @umask($i_current_umask);
    return true;
}

/**
 * Make a directory (recursively) and fix rights.
 *
 * @param  string  $s_dir  Directory path and name.
 * @param  number  $i_mode     Mode number.
 * @return boolean
 */
function make_dir($s_dir, $i_mode = null)
{
    if (!$i_mode) $i_mode = octdec(get_configuration('server/dirmode'));

    $i_current_umask = umask();
    umask(0000);
    if (false === @mkdir($s_dir, $i_mode, true)) {
        if (function_exists('set_message')) set_message('error', __('ErrMakeDir', $s_dir));
        return false;
    }
    @chmod($s_dir, $i_mode);
    @umask($i_current_umask);
    return true;
}

/**
 * Returns defaults configuration array.
 *
 * @return array
 */
function configuration_get_defaults()
{
    return array(
        'server' => array(
            'base_url' => '',
            'url_rewriting' => 0,
            'page_var_name' => 'p',
            'base_dir' => '/',
            'filesize' => 3000000,
        ),
        'pages' => array(
            'default_extension' => 'html',
            'default_layout' => '',
            'default_doctype' => 'xhtml-1.0-strict',
            'default_css' => '',
            'default_js' => '',
        ),
        'administration' => array(
            'route' => 'admin',
            'is_secure' => false,
        ),
        'misc' => array(
            'site' => array(
                'title' => '',
                'title_format' => '%site-title% - %page-title%',
            ),
            'images' => array(
                'thumb_width' => 100,
                'thumb_height' => 100,
                'name_format' => '%title%-%date%',
            ),
            'documents' => array(
                'name_format' => '%title%-%date%',
            ),
        ),
    );
}

/**
 * Returns defaults page array.
 *
 * @return array
 */
function page_get_defaults()
{
    return array(
        'title' => '',
        'cache' => 0,
        'slug' => '',
        'state' => STRUCTURE_STATE_HIDDEN,
        'layout' => get_configuration('pages/default_layout', ''),
        'type' => '',
        'type_mime' => '',
        'doctype' => get_configuration('pages/default_doctype', 'xhtml-1.0-transitional'),
        'js_files' => ('' !== ($s_js_files = get_configuration('pages/default_js')) && $s_js_files ? explode(';', $s_js_files) : array()),
        'css_files' => ('' !== ($s_css_files = get_configuration('pages/default_css')) && $s_css_files ? explode(';', $s_css_files) : array()),
        'body_id' => '',
        'body_classes' => '',
        'parent_id' => '',
        'locked' => false,
        'metadata' => array(
            'description' => '',
            'keywords' => '',
        ),
        'dates' => array(
            'creation' => strval(time()),
            'modification' => strval(time()),
        ),
    );
}

/**
 * Returns defaults structure array.
 *
 * @return array
 */
function structure_get_defaults()
{
    return array(
        'id' => '',
        'title' => '',
        'slug' => '',
        'parent_id' => '',
        'state' => STRUCTURE_STATE_HIDDEN,
        'show_slug' => 1,
        'redirection' => '',
        'redirection_code' => '',
        'cache' => 0,
        'type' => 'html',
        'extension' => get_configuration('pages/default_extension', ''),
        'index_menu' => 0,
        'is_secure' => 0,
        'children' => array());
}
