/*
 * Copyright 2012 Sébastien Raud
 *
 * This file is part of beCms.
 *
 * beCms is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * beCms is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with beCms.  If not, see <http://www.gnu.org/licenses/>.
 */

// @todo : check separator when item is not visible
if (undefined === weEdContextMenu) {
    var weEdContextMenu = function(options, editor) {
        this.editor = editor;
        this.plugin_name = 'context_menu';

        this.$items = {};
        this.items_callback = {};
        this.separators = 0;
        this.$menu = jQuery.createElement('ul', { "class": "weed-context-menu" });
        this.show_menus = [];

        var self = this;

        /**
         * Adds an item to the context menu.
        *
        * @param  string    item_key           Item key.
        * @param  string    item_label         Item label.
        * @param  callback  callback_function  Callback function, when item is clicked.
        * @param  string    parent_item_key    Parent item group key.
        * @param  boolean   enabled            Indicates if item is enabled.
        * @return boolean
        */
        this.editor.addContextMenuItem = function(item_key, item_label, callback_function, parent_item_key, enabled) {
            if ('separator' == item_key) return self.editor.addContextMenuSeparator((parent_item_key || null));
            enabled = undefined == enabled ? true : enabled;

            var $li = jQuery.createElement('li', { "class": "weed-context-menu-item " + (!enabled ? " weed-context-menu-item-desabled" : "") })
                        .append(jQuery.createElement('span', { "text": item_label, "class": "weed-context-menu-label-key" }));

            //$li.data('weed-context-menu-key', item_key);
            $li.attr('id', item_key);

            if (callback_function) self.items_callback[item_key] = callback_function;//$li.data('weed-context-menu-callback', callback_function);

            self.bindEnabledItem($li, callback_function, enabled);

            self.$items[item_key] = $li;

            if (parent_item_key) {
                var $li_parent = (self.$items[parent_item_key] ? self.$items[parent_item_key] : null);
                if ($li_parent) {
                    var $li_parent_children = $li_parent.children('ul');
                    if (!$li_parent_children[0]) {
                        $li_parent_children = jQuery.createElement('ul', { "class": "editor-context-sub-menu" });
                        $li_parent.append($li_parent_children);
                        $li_parent.children('span:first')
                            .append(jQuery.createElement('span', { "class": "weed-context-menu-group", "style": "float: right" }).html('&#8227'));
                    }
                    $li_parent_children.append($li);
                    //return true;
                    return $li;
                }
                return false;
            }

            self.$menu.append($li);

            //return true;
            return $li;
        };

        /**
         * Adds a separator to the context menu.
         *
         * @param  string  parent_item_key  Parent item group key.
         * @return boolean
         */
        this.editor.addContextMenuSeparator = function(parent_item_key) {
            var $li = jQuery.createElement('li', { "class": "weed-context-menu-separator" });
            if (parent_item_key) {
                var $li_parent = (self.$items[parent_item_key] ? self.$items[parent_item_key] : null);
                if ($li_parent) {
                    var $li_parent_children = $li_parent.children('ul');
                    if ($li_parent_children[0]) {
                        $li_parent_children.append($li);
                        return true;
                    }
                }
                return false
            }

            self.$menu.append($li);
            return true;
        };

        /**
         * Sets an item enabled / disabled.
         *
         * @param  string   item_key  Item key.
         * @param  boolean  enabled   Indicates if item is enabled. true by default.
         * @return boolean
         */
        this.editor.setContextMenuItemEnabled = function(item_key, enabled) {
            enabled = undefined == enabled ? true : enabled;

            var $li = (self.$items[item_key] ? self.$items[item_key] : null);
            if ($li) {
                self.bindEnabledItem($li, self.items_callback[$li.attr('id')]/*$li.data('weed-context-menu-callback')*/, enabled);
                return true;
            }
            return false;
        };

        /**
         * Sets an item visible / hidden.
         *
         * @param  string   item_key  Item key.
         * @param  boolean  visible   Indicates if item is visible. true by default.
         * @return boolean
         */
        this.editor.setContextMenuItemVisible = function(item_key, visible) {
            visible = undefined == visible ? true : visible;
            var $li = (self.$items[item_key] ? self.$items[item_key] : null);
            if ($li) {
                if (visible) $li.show();
                else $li.hide();
                return true;
            }
            return false;
        },

        /**
         * Removes all items.
         *
         * @return void
         */
        this.editor.clearContextMenuItems = function() {
            self.$menu.children().remove();
            self.$items = {};
        };

        /**
         * Removes an item.
         *
         * @param  string   item_key  Item key.
         * @return boolean
         */
        this.editor.removeContextMenuItem = function(item_key) {
            if ('string' == typeof item_key) {
                var $li = (self.$items[item_key] ? self.$items[item_key] : null);
                if (!$li) return false;
                var $ul = $li.closest('ul');
                var $li_parent = $ul.closest('li');

                $li.remove();
                if (!$ul.children().length) {
                    $ul.remove();
                    if ($li_parent[0]) $li_parent.find('span.weed-context-menu-group:first').remove();
                }

                var $new_items = {};
                for (var i in self.$items)
                    if (i != item_key)
                        $new_items[i] = self.$items[i];

                self.$items = $new_items;
                return true;
            }
            return false;
        };

        this.editor.removeContextMenuItemByExpr = function(expr) {
            function remove($li) {
                if (!$li[0]) return false;
                var $ul = $li.closest('ul');
                var $li_parent = $ul.closest('li');

                //delete self.$items[$li.data('weed-context-menu-key')];
                delete self.$items[$li.attr('id')];
                if (self.items_callback[$li.attr('id')]) delete self.items_callback[$li.attr('id')];

                $li.remove();
                if (!$ul.children().length) {
                    $ul.remove();
                    if ($li_parent[0]) $li_parent.find('span.weed-context-menu-group:first').remove();
                }
                return true;
            };

            for (var i in self.$items) {
                var $li = self.$items[i];
                //if (expr.test($li.data('weed-context-menu-key')))
                if (expr.test($li.attr('id')))
                    remove($li);
            }
        };

        this.editor.getContextMenu = function() {
            return self.$menu;
        };

        this.editor.hideContextMenu = function(e) {
            this.trigger('hideContextMenu', { "event": e });
        };

        this.editor._hideContextMenu = function(data) {
            if (self.$menu.is(':visible')) {
                self.$menu.hide();
                self.$menu.find('.weed-context-menu-hover').removeClass('weed-context-menu-hover').css("background-color", "#eee");
                this.$toolbar.unbind('click.context-menu');
                this.$statusbar.unbind('click.context-menu');
                self.editor.focus();
            }
        };

        this.editor.showContextMenu = function(e) {
            this.trigger('showContextMenu', { "event": e });
        };

        this.editor._showContextMenu = function(data) {
            if (data.event) {
                data.event.preventDefault();
                data.event.stopPropagation();
            }

            self.getContextMenu();

            if (self.$menu.children('li.weed-context-menu-item:visible').length) {
                var position = self.editor.$iframe.offset();
                var $contents = self.editor.$iframe.contents();

                /* .css({ "top": (data.event.pageY + position.top + 5 - $contents.scrollTop()) + "px", "left": (data.event.pageX + position.left + 5 - $contents.scrollLeft()) + "px" }) */
                self.$menu
                    .css({ "top": (data.event.clientY - 5 + position.top) + "px", "left": (data.event.clientX + 15 + position.left) + "px" })
                    .show();

                function resize_items($ul) {
                    var $ul_children = $ul.children('li:visible').children('ul');
                    $ul_children.each(function() { resize_items(jQuery(this)); });
                    var $ul_li = $ul.children('li:visible');
                    var max_width = 0;
                    var elem_width = 0;

                    $ul.find('span.weed-context-menu-group').css('float', 'none'); // bug on opera
                    var $menu_group = $ul.find('span.weed-context-menu-group:first');
                    var add_width = 10;

                    $ul_li.children('span.weed-context-menu-label-key').each(function() {
                        elem_width = parseInt(jQuery(this).outerWidth(true));
                        if (elem_width > max_width) {
                            max_width = elem_width;
                        }
                    });

                    max_width += add_width;

                    $ul_li.css('width', max_width);

                    max_width = (parseInt($ul.children('li.weed-context-menu-item:visible:first').outerWidth(true)) || max_width);

                    $ul.width(max_width);
                    $ul_children.css('left', max_width);

                    $ul.children('li.weed-context-menu-separator').width(max_width);
                    $ul.find('span.weed-context-menu-group').css('float', 'right');
                }

                resize_items(self.$menu);
                self.$menu.find('li.weed-context-menu-separator').each(function() {
                    var $this = jQuery(this);
                    if (!$this.prevAll('li.weed-context-menu-item:visible').length || !$this.nextAll('li.weed-context-menu-item:visible').length) $this.hide();
                    else $this.show();
                });
                self.$menu.find('li ul').hide();

                var offset = self.$menu.offset();
                if (offset && (parseInt(offset.left) + parseInt(self.$menu.width()) >= parseInt($(window).width()))) {
                    self.$menu.css('left', parseInt(self.$menu.css('left')) - (parseInt(self.$menu.outerWidth(true))) - 15);
                }
                if (offset && (parseInt(offset.top) + parseInt(self.$menu.height()) >= parseInt($(window).height()))) {
                    self.$menu.css('top', parseInt(self.$menu.css('top')) - (parseInt(self.$menu.outerHeight(true))) - 5);
                }
            }
            else {
                self.$menu.hide();
            }
        };
    };

        //this.has_undo_plugin = false;
    weEdContextMenu.prototype.init = function() {
        var self = this;

        this.$menu
            .bind('contextmenu', function(e) {
                e.preventDefault();
                e.stopPropagation();
                return false;
            })
            .hide();

        //this.$menu.hide();
        this.editor.$container.append(this.$menu);

        this.editor.$doc
            .bind('click.context-menu', function(e) {
                if (e.button != 2) {
                    self.editor.hideContextMenu(e);
                }
            })
            .bind('contextmenu', function(e) {
                e.preventDefault();
                e.stopPropagation();
                self.editor.showContextMenu(e);
                return false;
            });

        this.editor.bind('showContextMenu.__weed', { "event": null }, self.editor._showContextMenu);
        this.editor.bind('hideContextMenu.__weed', { "event": null }, self.editor._hideContextMenu);
    };

    weEdContextMenu.prototype.bindEnabledItem = function($li, callback_function, enabled) {
        var self = this;
        $li .unbind('click.context-menu mouseover.context-menu')
            .removeClass('weed-context-menu-item-desabled weed-context-menu-hover')
            .css({ "background-color": "#eee", "opacity": 1 });

        if (enabled) {
            if (callback_function && (jQuery.isFunction(callback_function) || jQuery.isArray(callback_function))) {
                $li.bind('click.context-menu', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    self.editor.hideContextMenu(e);
                    //if (self.editor.beforeUpdateHtml) self.editor.beforeUpdateHtml();
                    if (jQuery.isFunction(callback_function)) callback_function();
                    else callback_function[0].apply(callback_function[1]);
                    //if (self.editor.afterUpdateHtml) self.editor.afterUpdateHtml();
                });
            }
            $li
            .css('background-color', '#eee')
            .bind('mouseover.context-menu', function(e) {
                    var $this = jQuery(this);
                    e.stopPropagation();

                    if ($this.hasClass('weed-context-menu-hover')) return ;

                    // clear all hover
                    self.$menu.find('.weed-context-menu-hover').removeClass('weed-context-menu-hover').css("background-color", "#eee");

                    // add hover to current
                    $this.parents('li').andSelf().addClass('weed-context-menu-hover');
                    if (!$this.hasClass('weed-context-menu-item-desabled')) {
                        var position = $this.position();
                        var $ul = $this.find('ul:first');
                        $ul.css("top", (position.top - 1) + 'px').show();

                        var offset = $ul.offset();
                        if (offset && (parseInt(offset.left) + parseInt($ul.width()) >= parseInt($(window).width()))) {
                            $ul.css('left', - (parseInt($ul.width()) + 2));
                        }
                        if (offset && (parseInt(offset.top) + parseInt($ul.height()) >= parseInt($(window).height()))) {
                            $ul.css('top', -(parseInt($ul.outerHeight(true)) - (position.top + parseInt($this.outerHeight(true)))));
                        }

                        $this.parents('li').andSelf().css('background-color', '#3399ff');
                    }

                    $this.prevAll('li.weed-context-menu-item').css("background-color", "#eee").find('ul:visible').hide();
                    $this.nextAll('li.weed-context-menu-item').css("background-color", "#eee").find('ul:visible').hide();
                });
        }
        else if (!$li.hasClass('weed-context-menu-item-desabled')) {
            $li.addClass('weed-context-menu-item-desabled').css({ "background-color": "#eee", "opacity": 0.3 })
                .bind('mouseover.context-item', function(e) {
                    var $this = jQuery(this);
                    e.stopPropagation();

                    // clear all hover
                    self.$menu.find('.weed-context-menu-hover').removeClass('weed-context-menu-hover').css("background-color", "#eee");

                    // add hover to current
                    $this.parents('li').addClass('weed-context-menu-hover');
                    $this.parents('li').css('background-color', '#3399ff');
                });
        }
    };

    weEdContextMenu.prototype.getContextMenu = function() {
        // removes separators neighbors, first and last separators
        this.$menu.find('li.weed-context-menu-separator').next('li.weed-context-menu-separator').remove();
        this.$menu.find('li.weed-context-menu-separator:first-child').remove();
        this.$menu.find('li.weed-context-menu-separator:last-child').remove();

        var css = {
            "background-color": "#eee",
            "border":   "1px solid #ccc",
            "display":  "block",
            "margin":   0,
            "padding":  0,
            "position": "absolute",
            "width":    "100%",
            "cursor":  "pointer",
            "z-index":  1000
        };
        this.$menu.css(css);
        this.$menu.find('ul').css(css);

        this.$menu.find('li.weed-context-menu-item').css({
            "font-size":    "11px",
            "list-style":   "none",
            "padding":      "5px"
        });
        this.$menu.find('li.weed-context-menu-separator').css({
            "background-color": "#ccc",
            "height":       "1px",
            "list-style":   "none",
            "padding":      0
        });
        this.$menu.find('li span').css({
            "margin": 0,
            "padding": 0
        });

        return this.$menu;
    };
}