/*
 * Copyright 2012 Sébastien Raud
 *
 * This file is part of beCms.
 *
 * beCms is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * beCms is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with beCms.  If not, see <http://www.gnu.org/licenses/>.
 */

if (undefined === weEdContextMenuFormatBlocksTools) {
    var weEdContextMenuFormatBlocksTools  = function(options, editor) {
        this.editor = editor,
        this.plugin_name = 'context_menu_format_blocks_tools',

        this.$selected_node = null,
        this.convert_blocks_items = {},
        this.add_block_items = {},
        this.duplicate_block_items = {},
        this.delete_block_items = {};

        this.editor.loadLanguage(this.plugin_name);
    };

    weEdContextMenuFormatBlocksTools.prototype.init  = function() {
        var self = this;

        if (!this.editor.hasPlugin('context_menu')) return ;

        var blocks = { "p": "paragraph", "div": "div", "h1": "heading1", "h2": "heading2", "h3": "heading3", "h4": "heading4", "h5": "heading5", "h6": "heading6" };

        this.editor.addContextMenuSeparator();
        this.editor.addContextMenuItem('blocks_tools', self._('Blocks elements') + '...');
        this.editor.addContextMenuItem('convert_blocks', self._('Convert') + '...', null, 'blocks_tools');
        this.editor.addContextMenuSeparator();
        this.editor.addContextMenuItem('add_block', self._('Add paragraph') + '...', null, 'blocks_tools');
        this.editor.addContextMenuSeparator();
        this.editor.addContextMenuItem('duplicate_block', self._('Duplicate block') + '...', null, 'blocks_tools');
        this.editor.addContextMenuSeparator();
        this.editor.addContextMenuItem('delete_block', self._('Delete block') + '...', null, 'blocks_tools');

        this.editor.bind('showContextMenu.pre.context_menu_format_blocks_tools', function() {
            var _self = this; // editor
            var index_convert = 0, index_add = 0, index_duplicate = 0, index_delete = 0;

            function clear() {
                convert_clear();
                add_clear();
                duplicate_clear();
                delete_clear();
            }

            function convert_clear() {
                for (var i in self.convert_blocks_items) {
                    _self.removeContextMenuItem(i);
                }

                self.convert_blocks_items = {};
            }

            function add_clear() {
                for (var i in self.add_block_items) {
                    _self.removeContextMenuItem(i + '_before');
                    _self.removeContextMenuItem(i + '_after');
                }

                self.add_block_items = {};
            }

            function duplicate_clear() {
                for (var i in self.duplicate_block_items) {
                    _self.removeContextMenuItem(i);
                }

                self.duplicate_block_items = {};
            }

            function delete_clear() {
                for (var i in self.delete_blocks_items) {
                    _self.removeContextMenuItem(i);
                }

                self.delete_blocks_items = {};
            }

            function convert_add_item($node) {
                var blocks = { "p": "paragraph", "div": "div", "h1": "heading1", "h2": "heading2", "h3": "heading3", "h4": "heading4", "h5": "heading5", "h6": "heading6" };
                var node_name = $node[0].nodeName.toLowerCase();

                if (undefined == blocks[node_name]) return;

                self.convert_blocks_items['convert_blocks_' + node_name + '_' + index_convert] = [];
                self.convert_blocks_items['convert_blocks_' + node_name + '_' + index_convert].push(_self.addContextMenuItem('convert_blocks_' + node_name + '_' + index_convert, self._(blocks[node_name]) + '...', null, 'convert_blocks'));

                for (var j in blocks) {
                    if (j == node_name) continue;
                    var fn = self.makeConvertFunction(node_name, j);
                    self.convert_blocks_items['convert_blocks_' + node_name + '_' + index_convert].push(_self.addContextMenuItem('convert_blocks_' + node_name + '_' + j, self._(blocks[j]), [self[fn], self], 'convert_blocks_' + node_name + '_' + index_convert));
                }

                for (var i = 0; i < self.convert_blocks_items['convert_blocks_' + node_name + '_' + index_convert].length; i++) {
                    self.convert_blocks_items['convert_blocks_' + node_name + '_' + index_convert][i]
                        .bind('mouseover.context_menu_format_blocks_tools', function() {
                            _self.emphasisNode($node, true);
                        })
                        .bind('mouseout.context_menu_format_blocks_tools', function() {
                            _self.emphasisNode($node, false);
                        });
                }

                index_convert++;
            }

            function add_add_item($node) {
                // add p before / after :
                // div, p, hX, table, form, ul, ol, dl, hr, pre, blockquote, address, fieldset, legend
                var for_elem = { "p": "paragraph", "div": "div", "h1": "heading1", "h2": "heading2", "h3": "heading3", "h4": "heading4", "h5": "heading5", "h6": "heading6", "table": "table",
                                "form": "form", "ul": "ul", "ol": "ol", "dl": "dl", "hr": "hr", "pre": "pre", "blockquote": "blockquote", "address": "address", "fieldset": "fieldset", "legend": "legend" };

                var node_name = $node[0].nodeName.toLowerCase();

                if (undefined == for_elem[node_name]) return;

                self.add_block_items['add_block_items_' + node_name + '_' + index_add] = [];
                var fn = self.makeAddFunction('before', 'p', $node);
                self.add_block_items['add_block_items_' + node_name + '_' + index_add].push(_self.addContextMenuItem('add_block_items_' + node_name + '_' + index_add + '_before', self._('Add before') + ' "' + self._(for_elem[node_name]) + '"', [fn, self], 'add_block'));
                var fn = self.makeAddFunction('after', 'p', $node);
                self.add_block_items['add_block_items_' + node_name + '_' + index_add].push(_self.addContextMenuItem('add_block_items_' + node_name + '_' + index_add + '_after', self._('Add after') + ' "' + self._(for_elem[node_name]) + '"', [fn, self], 'add_block'));

                for (var i = 0; i < self.add_block_items['add_block_items_' + node_name + '_' + index_add].length; i++) {
                    self.add_block_items['add_block_items_' + node_name + '_' + index_add][i]
                        .bind('mouseover.context_menu_format_blocks_tools', function() {
                            _self.emphasisNode($node, true);
                        })
                        .bind('mouseout.context_menu_format_blocks_tools', function() {
                            _self.emphasisNode($node, false);
                        });
                }

                index_add++;
            }

            function duplicate_add_item($node) {
                var blocks = { "p": "paragraph", "div": "div", "h1": "heading1", "h2": "heading2", "h3": "heading3", "h4": "heading4", "h5": "heading5", "h6": "heading6",
                               "ul": "ul", "ol": "ol", "dl": "dl", "li": "li", "pre": "pre", "blockquote": "blockquote", "address": "address" };

                var node_name = $node[0].nodeName.toLowerCase();

                if (undefined == blocks[node_name]) return;

                var fn = self.makeDuplicateFunction($node);
                self.duplicate_block_items['duplicate_block_' + node_name + '_' + index_duplicate] = [_self.addContextMenuItem('duplicate_block_' + node_name + '_' + index_duplicate, self._('Duplicate') + ' "' + self._(blocks[node_name]) + '"', [fn, self], 'duplicate_block')];

                for (var i = 0; i < self.duplicate_block_items['duplicate_block_' + node_name + '_' + index_duplicate].length; i++) {
                    self.duplicate_block_items['duplicate_block_' + node_name + '_' + index_duplicate][i]
                        .bind('mouseover.context_menu_format_blocks_tools', function() {
                            _self.emphasisNode($node, true);
                        })
                        .bind('mouseout.context_menu_format_blocks_tools', function() {
                            _self.emphasisNode($node, false);
                        });
                }

                index_duplicate++;
            }

            function delete_add_item($node) {
                //var blocks = { "p": "paragraph", "div": "div", "h1": "heading1", "h2": "heading2", "h3": "heading3", "h4": "heading4", "h5": "heading5", "h6": "heading6" };
                var blocks = { "p": "paragraph", "div": "div", "h1": "heading1", "h2": "heading2", "h3": "heading3", "h4": "heading4", "h5": "heading5", "h6": "heading6",
                               "ul": "ul", "ol": "ol", "dl": "dl", "li": "li", "pre": "pre", "blockquote": "blockquote", "address": "address" };
                var node_name = $node[0].nodeName.toLowerCase();

                if (undefined == blocks[node_name]) return;

                var fn = self.makeDeleteFunction($node);
                self.delete_blocks_items['delete_blocks_' + node_name + '_' + index_delete] = [_self.addContextMenuItem('delete_blocks_' + node_name + '_' + index_delete, self._('Delete') + ' "' + self._(blocks[node_name]) + '"', [fn, self], 'delete_block')];

                for (var i = 0; i < self.delete_blocks_items['delete_blocks_' + node_name + '_' + index_delete].length; i++) {
                    self.delete_blocks_items['delete_blocks_' + node_name + '_' + index_delete][i]
                        .bind('mouseover.context_menu_format_blocks_tools', function() {
                            _self.emphasisNode($node, true);
                        })
                        .bind('mouseout.context_menu_format_blocks_tools', function() {
                            _self.emphasisNode($node, false);
                        });
                }

                index_delete++;
            }

            function add_item($node) {
                convert_add_item($node);
                add_add_item($node);
                duplicate_add_item($node);
                delete_add_item($node);
            }

            function exec() {
                clear();

                var $selected_node = jQuery(_self.getSelectedNode());
                add_item($selected_node);

                $selected_node.parents().each(function() {
                    add_item(jQuery(this));
                });

                _self.setContextMenuItemEnabled('convert_blocks', (index_convert > 0));
                _self.setContextMenuItemEnabled('add_block', (index_add > 0));
                _self.setContextMenuItemEnabled('duplicate_block', (index_duplicate > 0));
                _self.setContextMenuItemEnabled('delete_block', (index_delete > 0));
            }

            exec();
        });
    };

    weEdContextMenuFormatBlocksTools.prototype.convert = function(from, to) {
        this.editor.emphasisNode(jQuery(this.editor.getSelectedNode()), false);

        this.editor.addCommand('convertFormatBlock', [function() {
            var $selected_node = jQuery(this.editor.getSelectedNode());
            if ($selected_node[0].nodeName.toLowerCase() != from) {
                $selected_node = $selected_node.parents(from + ':first');
            }
            if ($selected_node[0].nodeName.toLowerCase() == from) {
                var $new = jQuery.createElement(to);
                $new.append($selected_node.contents());
                if ('' !== $selected_node.attr('class')) {
                    $new.addClass($selected_node.attr('class'));
                    if ($new.hasClass('weed-css-class-selected')) {
                        $new.removeClass('weed-css-class-selected');
                        if ('' === $new.attr('class')) $new.removeAttr('class');
                    }
                }
                if ('' !== $selected_node.attr('id')) $new.attr('id', $selected_node.attr('id'));
                $selected_node.after($new);
                $selected_node.remove();
            }
        }, this]);
        this.editor.execCommand('convertFormatBlock');
    };

    weEdContextMenuFormatBlocksTools.prototype.makeConvertFunction = function(from, to) {
        var fn = 'convert' + from.toUpperCase() + 'to' + to.toUpperCase();
        if (undefined === weEdContextMenuFormatBlocksTools.prototype[fn])
            weEdContextMenuFormatBlocksTools.prototype[fn] = function() { this.convert(from, to); };
        return fn;
    };

    weEdContextMenuFormatBlocksTools.prototype.makeAddFunction = function(position, type, $node) {
        return function() {
            this.editor.emphasisNode($node, false);

            this.editor.addCommand('addElement', [function() {
                var $new_node = jQuery.createElement(type, { "html": "<br />" });

                if ('before' === position) $node.before($new_node);
                else $node.after($new_node);

                this.editor.focusToNode($new_node[0]);
            }, this]);

            this.editor.execCommand('addElement');
        };
    };

    weEdContextMenuFormatBlocksTools.prototype.makeDuplicateFunction = function($node) {
        return function() {
            this.editor.emphasisNode($node, false);

            this.editor.addCommand('duplicateBlockElement', [function() {
                var $new = $node.clone();
                $node.after($new);
                function removeId($node) {
                    $node.removeAttr('id');
                    removeId($node.children());
                }
                removeId($new);

            }, this]);
            this.editor.execCommand('duplicateBlockElement');
        };
    };

    weEdContextMenuFormatBlocksTools.prototype.makeDeleteFunction = function($node) {
        return function() {
            this.editor.emphasisNode($node, false);

            this.editor.addCommand('deleteBlockElement', [function() {
                $node.remove();
            }, this]);
            this.editor.execCommand('deleteBlockElement');
        };
    };

    weEdContextMenuFormatBlocksTools.prototype._ = function(key) {
        return this.editor._(key, this.plugin_name);
    };
};
