/*
 * Copyright 2012 Sébastien Raud
 *
 * This file is part of beCms.
 *
 * beCms is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * beCms is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with beCms.  If not, see <http://www.gnu.org/licenses/>.
 */

if (undefined === weEdContextMenuTableTools) {
    var weEdContextMenuTableTools  = function(options, editor) {
        this.editor = editor;
        this.plugin_name = 'context_menu_table_tools';
        this.$selected_node = null;

        this.editor.loadLanguage(this.plugin_name);
    };

    weEdContextMenuTableTools.prototype.init  = function() {
        var self = this;

        if (!this.editor.hasPlugin('context_menu')) return ;

        this.editor.addContextMenuItem('table-tools-table', self._('Table'));
        this.editor.addContextMenuSeparator();
        this.editor.addContextMenuItem('table-tools-head', self._('Headers'), null, 'table-tools-table');
        this.editor.addContextMenuItem('table-tools-head-thead-to-tbody', self._('Move header to body'), [self.moveTheadToBody, self], 'table-tools-head');
        this.editor.addContextMenuItem('table-tools-head-th-to-td', self._('Change cells as "td"'), [self.moveThToTd, self], 'table-tools-head');

        this.editor.addContextMenuSeparator('table-tools-table');
        this.editor.addContextMenuItem('table-tools-cell', self._('Cell'), null, 'table-tools-table');
        this.editor.addContextMenuItem('table-tools-cell-add-before', self._('Insert cell before'), [self.addCellBefore, self], 'table-tools-cell');
        this.editor.addContextMenuItem('table-tools-cell-add-after', self._('Insert cell after'), [self.addCellAfter, self], 'table-tools-cell');
        this.editor.addContextMenuItem('table-tools-cell-remove', self._('Delete cell'), [self.removeCell, self], 'table-tools-cell');
        this.editor.addContextMenuItem('table-tools-cell-merge-right', self._('Merge right'), [self.mergeCellWithLeft, self], 'table-tools-cell');
        this.editor.addContextMenuItem('table-tools-cell-merge-down', self._('Merge down'), [self.mergeCellWithDown, self], 'table-tools-cell');
        this.editor.addContextMenuItem('table-tools-cell-split-horizontally', self._('Split Cell Horizontally'), [self.splitCellHorizontally, self], 'table-tools-cell');
        this.editor.addContextMenuItem('table-tools-cell-split-verticaly', self._('Split Cell Vertically'), [self.splitCellVertically, self], 'table-tools-cell');

        this.editor.addContextMenuSeparator('table-tools-table');
        this.editor.addContextMenuItem('table-tools-line', self._('Row'), null, 'table-tools-table');
        this.editor.addContextMenuItem('table-tools-line-add-before', self._('Insert row before'), [self.addLineBefore, self], 'table-tools-line');
        this.editor.addContextMenuItem('table-tools-line-add-after', self._('Insert row after'), [self.addLineAfter, self], 'table-tools-line');
        this.editor.addContextMenuItem('table-tools-line-delete', self._('Delete row'), [self.removeLine, self], 'table-tools-line');

        this.editor.addContextMenuSeparator('table-tools-table');
        this.editor.addContextMenuItem('table-tools-column', self._('Column'), null, 'table-tools-table');
        this.editor.addContextMenuItem('table-tools-column-insert-before', self._('Insert column before'), [self.addColumnBefore, self], 'table-tools-column');
        this.editor.addContextMenuItem('table-tools-column-insert-after', self._('Insert column after'), [self.addColumnAfter, self], 'table-tools-column');
        this.editor.addContextMenuItem('table-tools-column-delete', self._('Delete column'), [self.removeColumn, self], 'table-tools-column');

        this.editor.addContextMenuSeparator('table-tools-table');
        this.editor.addContextMenuItem('table-tools-delete', self._('Delete table'), [self.removeTable, self], 'table-tools-table');

        //this.editor.chainMethod('showContextMenu', 'pre', function(vars) {
        this.editor.bind('showContextMenu.pre.context_menu_table_tools', function() {
            var $selected_node = jQuery(this.getSelectedNode());
            var node_name = $selected_node[0].nodeName.toLowerCase();

            if ('tr' == node_name) {
                $selected_node = $selected_node.find('td:first, th:first');
                node_name = $selected_node[0].nodeName.toLowerCase();
            }

            if ('td' == node_name || 'th' == node_name) {
                self.$selected_node = $selected_node;
                self.showMenu();
            }
            else {
                var $parent = $selected_node.closest('td, th');
                if ($parent[0]) {
                    self.$selected_node = $parent;
                    self.showMenu();
                }
                else
                    self.hideMenu();
            }
        });
    };

    weEdContextMenuTableTools.prototype.showMenu = function() {
        this.editor.setContextMenuItemVisible('table-tools-table',  (this.$selected_node.parents('table').length > 0));
        this.editor.setContextMenuItemVisible('table-tools-head',   true);
        this.editor.setContextMenuItemVisible('table-tools-cell',   true);
        this.editor.setContextMenuItemVisible('table-tools-line',   true);
        this.editor.setContextMenuItemVisible('table-tools-column', true);
        this.editor.setContextMenuItemVisible('table-tools-delete', true);

        var has_thead = this.$selected_node.closest('thead').length;
        var is_th = ('th' == this.$selected_node[0].nodeName.toLowerCase());
        this.editor.setContextMenuItemEnabled('table-tools-head', (has_thead || is_th));
        this.editor.setContextMenuItemEnabled('table-tools-head-thead-to-tbody', (has_thead));
        this.editor.setContextMenuItemEnabled('table-tools-head-th-to-td', (is_th));

        var mergeable_targets = weEdContextMenuTableTools.mergeableCells(this.$selected_node);
        this.editor.setContextMenuItemEnabled('table-tools-cell-merge-right', (null != mergeable_targets['right']));
        this.editor.setContextMenuItemEnabled('table-tools-cell-merge-down', (null != mergeable_targets['down']));
    };

    weEdContextMenuTableTools.prototype.hideMenu = function() {
        this.editor.setContextMenuItemVisible('table-tools-table',  false);
        this.editor.setContextMenuItemVisible('table-tools-head',   false);
        this.editor.setContextMenuItemVisible('table-tools-cell',   false);
        this.editor.setContextMenuItemVisible('table-tools-line',   false);
        this.editor.setContextMenuItemVisible('table-tools-column', false);
        this.editor.setContextMenuItemVisible('table-tools-delete', false);
    };

    weEdContextMenuTableTools.prototype.moveTheadToBody = function($selected_node) {
        this.editor.addCommand('moveTheadToBody', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;
            var $table = $selected_node.parents('table:first');
            var $thead = $table.children('thead');
            var $tbody = $table.children('tbody:first');
            if (!$tbody.length) {
                $tbody = jQuery.createElement('tbody');
                $thead.children('tr').each(function() {
                    $tbody.append(jQuery(this));
                });
                $thead.after($tbody);
            }
            else {
                if ($tbody.children('tr').length) {
                    $thead.children('tr').each(function(index) {
                        if (!index) $tbody.children('tr:first').before(jQuery(this));
                        else $tbody.children('tr:first').after(jQuery(this));
                    });
                }
                else {
                    $thead.children('tr').each(function() {
                        $tbody.append(jQuery(this));
                    });
                }
            }
            $thead.remove();
            var $cell = $table.find('th:first, td:first');
            this.editor.focusToNode($cell[0]);
            this.$selected_node = null; }, this]);
        this.editor.execCommand('moveTheadToBody');
    };

    weEdContextMenuTableTools.prototype.moveThToTd = function($selected_node) {
        this.editor.addCommand('moveThToTd', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;
            var $table = $selected_node.parents('table:first');
            var $tr =  $selected_node.parents('tr:first');
            var $target = $tr.clone();
            $target.children().remove();

            $tr.children().each(function() {
                var $tmp = jQuery.createElement('div').append(jQuery(this).clone(true)).html().replace(/<(\/)?th/gi,'<\$1td');
                $target.append($tmp);
            });

            $tr.after($target);
            $tr.remove();

            var $cell = $table.find('th:first, td:first');
            this.editor.focusToNode($cell[0]);
            this.$selected_node = null; }, this]);
        this.editor.execCommand('moveThToTd');
    };

    weEdContextMenuTableTools.prototype.addCellBefore = function($selected_node) {
        this.editor.addCommand('addCellBefore', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;
            var $cell = $selected_node.clone().html('');
            $cell.insertBefore($selected_node);
            $cell.html('<br />');
            this.editor.focusToNode($cell[0]);
            this.$selected_node = null;}, this]);
        this.editor.execCommand('addCellBefore');
    };

    weEdContextMenuTableTools.prototype.addCellAfter = function($selected_node) {
        this.editor.addCommand('addCellAfter', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;
            var $cell = $selected_node.clone().html('');
            $selected_node.after($cell);
            $cell.html('<br />');
            this.editor.focusToNode($cell[0]);
            this.$selected_node = null;}, this]);
        this.editor.execCommand('addCellAfter');
    };

    weEdContextMenuTableTools.prototype.removeCell = function($selected_node) {
        this.editor.addCommand('removeCell', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;
            var $cell = ($selected_node.next()[0] ? $selected_node.next() : ($selected_node.prev()[0] ? $selected_node.prev() : null));
            var $tr = $selected_node.parents('tr:first');
            var $thb = $tr.parents('tbody:first, thead:first, tfoot:first');
            var $table = $tr.parents('table:first');

            $selected_node.remove();

            if (!$tr.children().length) $tr.remove();
            if ($thb && !$thb.children().length) $thb.remove();
            if (!$table.children().length) $table.remove();

            if ($cell && $cell[0]) this.editor.focusToNode($cell[0]);
            else if ($table && $table[0]) {
                var cell = $table.find('th:first, td:first')[0];
                this.editor.focusToNode(cell);
            }
            this.$selected_node = null; }, this]);
        this.editor.execCommand('removeCell');
    };

    weEdContextMenuTableTools.prototype.mergeCellWithLeft = function($selected_node) {
        this.editor.addCommand('mergeCellWithLeft', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;
            var mergeable_targets = weEdContextMenuTableTools.mergeableCells($selected_node);
            var $target = mergeable_targets['right'];
            var colspan = ($selected_node.attr('colspan') || 1) + ($target.attr('colspan') || 1);
            var $last = $selected_node.children().last();

            if ($last.is('br')) $last.remove();
            $last = $target.children().last();
            if ($last.is('br')) $last.remove();

            if ($selected_node.text().length && $target.text().length) $selected_node.append('<br />');

            $selected_node.append($target.html());
            $target.remove();
            $selected_node.attr('colspan', colspan);
            this.editor.focusToNode($selected_node[0]);
            this.$selected_node = null; }, this]);
        this.editor.execCommand('mergeCellWithLeft');
    };

    weEdContextMenuTableTools.prototype.mergeCellWithDown = function($selected_node) {
        this.editor.addCommand('mergeCellWithDown', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;
            var mergeable_targets = weEdContextMenuTableTools.mergeableCells($selected_node);
            var $target = mergeable_targets['down'];
            var rowspan = parseInt($selected_node.attr('rowspan') || 1) + parseInt($target.attr('rowspan') || 1);
            var $last = $selected_node.children().last();

            if ($last.is('br')) $last.remove();
            $last = $target.children().last();
            if ($last.is('br')) $last.remove();

            if ($selected_node.text().length && $target.text().length) $selected_node.append('<br />');

            $selected_node.append($target.html());
            $target.remove();
            $selected_node.attr('rowspan', rowspan);
            this.editor.focusToNode($selected_node[0]);
            this.$selected_node = null; }, this]);
        this.editor.execCommand('mergeCellWithDown');
    };

    weEdContextMenuTableTools.prototype.splitCellHorizontally = function($selected_node) {
        this.editor.addCommand('splitCellHorizontally', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;
            var colspan = $selected_node.attr('colspan') || 1;
            var map = weEdContextMenuTableTools.tableToArray($selected_node);
            var newcolspan = 1, newcellcolspan = 1;

            if (colspan > 1) {
                newcolspan = newcellcolspan = Math.ceil(colspan / 2);
            }
            else {
                var cells_in_col = weEdContextMenuTableTools.cellInCol(map, weEdContextMenuTableTools.cellIndex(map, $selected_node));

                for ( var i = 0; i < cells_in_col.length; i++ ) {
                    if (cells_in_col[i] && cells_in_col[i][0] != $selected_node[0])
                        cells_in_col[i].attr('colspan', parseInt(cells_in_col[i].attr('colspan') || 1) + 1);
                }
            }

            var $newcell = $selected_node.clone();
            $newcell.html('');
            $selected_node.after($newcell);
            $newcell.html('<br />');
            $selected_node.attr('colspan', newcolspan);
            $newcell.attr('colspan', newcellcolspan);

            if (1 == $selected_node.attr('colspan')) $selected_node.removeAttr('colspan');
            if (1 == $newcell.attr('colspan')) $newcell.removeAttr('colspan');
            this.editor.focusToNode($newcell[0]);
            this.$selected_node = null; }, this]);
        this.editor.execCommand('splitCellHorizontally');
    };

    weEdContextMenuTableTools.prototype.splitCellVertically = function($selected_node) {
        this.editor.addCommand('splitCellVertically', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;
            var map = weEdContextMenuTableTools.tableToArray($selected_node);
            var $tr = $selected_node.parent('tr:first');
            var rowindex = $tr[0].rowIndex;
            var colindex = weEdContextMenuTableTools.cellIndex(map, $selected_node);
            var rowspan = $selected_node.attr('rowspan') || 1;
            var newrowspan = 1, newcellrowspan = 1, newrowindex = 0;
            var $newcelltr = null;
            var $newcell = $selected_node.clone();
            $newcell.html('');

            if (rowspan > 1) {
                newrowspan = newcellrowspan = Math.ceil(rowspan / 2);
                newrowindex = rowindex + newrowspan;

                $newcelltr = map[newrowindex][0].parents('tr:first');
                var newcellrow = map[newrowindex];
                var $candidatecell = null;

                for (var c = 0; c < newcellrow.length; c++) {
                    $candidatecell = newcellrow[c];

                    if ($candidatecell.parents('tr:first')[0] == $newcelltr[0] && c > colindex) {
                        $candidatecell.before($newcell);
                        break;
                    }
                    else
                        $candidatecell = null;
                }

                if (!$candidatecell)
                    $newcelltr.append($newcell);
            }
            else {
                $newcell = $selected_node.clone();
                newcellrowspan = newrowspan = 1;

                $newcelltr = $tr.clone();
                $tr.after($newcelltr);
                $newcelltr.html('').append($newcell);

                var cells_in_same_row = map[rowindex];

                for (var i = 0; i < cells_in_same_row.length; i++ )
                    cells_in_same_row[i].attr('rowspan', (cells_in_same_row[i].attr('rowspan') || 1) + 1);
            }

            $selected_node.attr('rowspan', newrowspan);
            $newcell.attr('rowspan', newcellrowspan).html('<br />');
            if (newrowspan == 1) $selected_node.removeAttr('rowspan');
            if (newcellrowspan == 1) $newcell.removeAttr('rowspan');
            this.editor.focusToNode($newcell[0]);
            this.$selected_node = null; }, this]);
        this.editor.execCommand('splitCellVertically');
    };

    weEdContextMenuTableTools.prototype.addLineBefore = function($selected_node) {var self = this;
        this.editor.addCommand('addLineBefore', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;
            var $tr = $selected_node.parent('tr:first');
            var $target = $tr.clone();

            $target.children().each(function() {
                jQuery(this).html('');
                jQuery(this).removeAttr('rowspan');
            });

            $tr.before($target);

            $target.children().each(function() { jQuery(this).html('<br />'); });
            var $cell = $target.children(':first');
            if ($cell && $cell[0]) this.editor.focusToNode($cell[0]);
            this.$selected_node = null; }, this]);
        this.editor.execCommand('addLineBefore');
    };

    weEdContextMenuTableTools.prototype.addLineAfter = function($selected_node) {
        this.editor.addCommand('addLineAfter', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;
            var $tr = $selected_node.parent('tr:first');
            var $target = $tr.clone();

            $target.children().each(function() {
                jQuery(this).html('');
                jQuery(this).removeAttr('rowspan');
            });

            $tr.after($target);

            $target.children().each(function() { jQuery(this).html('<br />'); });
            var $cell = $target.children(':first');
            if ($cell && $cell[0]) this.editor.focusToNode($cell[0]);
            this.$selected_node = null; }, this]);
        this.editor.execCommand('addLineAfter');
    };

    weEdContextMenuTableTools.prototype.removeLine = function($selected_node) {
        this.editor.addCommand('removeLine', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;

            var $tr = $selected_node.parent('tr:first'),
                $table = $tr.parents('table:first'),
                map = weEdContextMenuTableTools.tableToArray($selected_node),
                startRowIndex = $tr[0].rowIndex,
                endRowIndex = startRowIndex + ($selected_node.attr('rowspan') || 1) - 1,
                rowsToDelete = [];

            var $tr_sel = ($tr.next()[0] ? $tr.next() : ($tr.prev()[0] ? $tr.prev() : null));

            for ( var i = startRowIndex; i <= endRowIndex; i++ ) {
                var mapRow = map[ i ],
                    row = $table[0].rows[i];

                for ( var j = 0; j < mapRow.length; j++ ) {
                    var cell = mapRow[j],
                        cellRowIndex = cell.parents('tr:first')[0].rowIndex,
                        cellRowSpan = (cell.attr('rowspan') || 1);

                    if ( cellRowSpan == 1 ) cell.remove();
                    else {
                        cell.attr('rowspan', cellRowSpan - 1);

                        if ( cellRowIndex == i ) {
                            var nextMapRow = map[ i + 1 ];

                            if (nextMapRow[ j - 1 ]) {
                                nextMapRow[ j - 1 ].after(cell);
                            }
                            else {
                                jQuery($table[0].rows[ i + 1 ] ).append(cell);
                            }
                        }
                    }

                    j += (cell.attr('colspan') || 1) - 1;
                }

                rowsToDelete.push( row );
            }

            var thb = [];
            for ( i = rowsToDelete.length ; i >= 0 ; i-- ) {
                if (rowsToDelete[i]) {
                    var $thb = $tr.parents('tbody:first, thead:first, tfoot:first');
                    if ($thb) thb.push($thb);
                    jQuery(rowsToDelete[ i ]).remove();
                }
            }

            for (i = 0; i < thb.length; i++) {
                var $thb = thb[i];
                if ($thb && !$thb.children().length) $thb.remove();
            }

            if (!$table.children().length) $table.remove();

            if ($tr_sel && $tr_sel[0]) {
                var $cell = $tr_sel.children(':first');
                this.editor.focusToNode($cell[0]);
            }
            else if ($table && $table[0]) {
                var cell = $table.find('th:first, td:first')[0];
                this.editor.focusToNode($cell);
            }
            this.$selected_node = null; }, this]);
        this.editor.execCommand('removeLine');
    };

    weEdContextMenuTableTools.prototype.addColumnBefore = function($selected_node) {
        this.editor.addCommand('addColumnBefore', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;
            this.insertColumn($selected_node, true);
            this.$selected_node = null; }, this]);
        this.editor.execCommand('addColumnBefore');
    };

    weEdContextMenuTableTools.prototype.addColumnAfter = function($selected_node) {
        this.editor.addCommand('addColumnAfter', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;
            this.insertColumn($selected_node);
            this.$selected_node = null; }, this]);
        this.editor.execCommand('addColumnAfter');
    };

    weEdContextMenuTableTools.prototype.removeColumn = function($selected_node) {
        this.editor.addCommand('removeColumn', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;
            var $table = $selected_node.parents('table:first'),
                map = weEdContextMenuTableTools.tableToArray($selected_node),
                startColIndex,
                endColIndex,
                rowsToDelete = [];

            // Figure out selected cells' column indices.
            for (var i = 0, rows = map.length; i < rows; i++) {
                for (var j = 0, cols = map[i].length; j < cols; j++) {
                    if (map[i][j][0] == $selected_node[0]) {
                        startColIndex = endColIndex = j;
                    }
                }
            }

            // Delete cell or reduce cell spans by checking through the table map.
            for (i = startColIndex; i <= endColIndex; i++) {
                for (j = 0; j < map.length; j++) {
                    var mapRow = map[j],
                        row = $table[0].rows[j],
                        cell = mapRow[i];

                    if (cell) {
                        if ((cell.attr('colspan') || 1) == 1 )
                            cell.remove();
                        // Reduce the col spans.
                        else
                            cell.attr('colspan', (cell.attr('colspan') || 1) - 1);

                        j += (cell.attr('rowspan') || 1) - 1;

                        if (!row.cells.length)
                            rowsToDelete.push(row);
                    }
                }
            }

            var firstRowCells = $table[0].rows[0] && $table[0].rows[0].cells;
            var $cell = (jQuery(firstRowCells[startColIndex])[0] ? jQuery(firstRowCells[startColIndex]) :
                (startColIndex ? (jQuery(firstRowCells[startColIndex - 1])[0] ? jQuery(firstRowCells[startColIndex - 1]) : null) : null));

            $table.find('tr:empty').remove();
            $table.find('thead:empty, tbody:empty, tfoot:empty').remove();
            if (!$table.children().length) $table.remove();

            // Where to put the cursor after columns been deleted?
            // 1. Into next cell of the first row if any;
            // 2. Into previous cell of the first row if any;
            // 3. Into table's parent element;
            //var cursorPosition =  new CKEDITOR.dom.element( firstRowCells[ startColIndex ] || ( startColIndex ? firstRowCells[ startColIndex - 1 ] : table.$.parentNode ) );

            // Delete table rows only if all columns are gone (do not remove empty row).

            if ($cell && $cell[0]) this.editor.focusToNode($cell[0]);
            else if ($table && $table[0]) {
                var cell = $table.find('th:first, td:first')[0];
                this.editor.focusToNode($cell);
            }
            this.$selected_node = null; }, this]);
        this.editor.execCommand('removeColumn');
    },

    weEdContextMenuTableTools.prototype.removeTable = function($selected_node) {
        this.editor.addCommand('removeTable', [function($selected_node) {
            $selected_node = $selected_node || this.$selected_node;
            $selected_node.parents('table:first').remove();
            this.$selected_node = null; }, this]);
        this.editor.execCommand('removeTable');
    };

    /* utilities functions */
    weEdContextMenuTableTools.tableToArray = function($selected_node) {
        /* based on CKEditor */
        var $table = $selected_node.parents('table:first');
        var rows = $table.find('tr');

        // Row and Column counters.
        var r = -1;
        var map = [];

        for (var i = 0; i < rows.length; i++) {
            r++ ;
            !map[r] && (map[r] = []);

            var c = -1 ;

            var cells = jQuery(rows[i]).find('td, th');

            for (var j = 0 ; j < cells.length ; j++) {
                var cell = jQuery(cells[j]) ;

                c++ ;
                while ( map[r][c] )
                    c++ ;

                var colspan = cell.attr('colspan') || 1;
                var rowspan = cell.attr('rowspan') || 1;

                for (var rs = 0; rs < rowspan; rs++) {
                    if (!map[r + rs]) map[r + rs] = [];

                    for (var cs = 0 ; cs < colspan; cs++) {
                        map[r + rs][c + cs] = jQuery(cells[j]) ;
                    }
                }
                c += colspan - 1 ;
            }
        }
        return map ;
    };

    weEdContextMenuTableTools.cellInCol = function(map, col_index) {
        var col = [];
        for (var r = 0; r < map.length; r++) {
            col.push( map[r][col_index]);
        }

        return col;
    };

    weEdContextMenuTableTools.cellIndex = function(map, $selected_node) {
        var startrow = $selected_node.parents('tr:first')[0].rowIndex;
        var startcolumn = 0;

        for (var startcolumn = 0; startcolumn < map[startrow].length; startcolumn++) {
            if (map[startrow][startcolumn][0] == $selected_node[0]) break;
        }
        return startcolumn;
    };

    weEdContextMenuTableTools.mergeableCells = function($selected_node, direction) {
        direction = direction || null;

        function _mergeable_cell(selected_node, direction) {

            var $selected_node = selected_node['node'];
            var rowspan = parseInt(selected_node['rowspan'] || $selected_node.attr('rowspan') || 1);
            var colspan = parseInt(selected_node['colspan'] || $selected_node.attr('colspan') || 1);
            var startrow = selected_node['startrow'] || null;
            var startcolumn = selected_node['startcolumn'] || 0;

            var map = weEdContextMenuTableTools.tableToArray($selected_node);

            if (!startrow) {
                startrow = $selected_node.parents('tr:first')[0].rowIndex;
                startcolumn = 0;
                var row = map[startrow];

                for (var startcolumn = 0; startcolumn < row.length; startcolumn++) {
                    if (row[startcolumn][0] == $selected_node[0]) break;
                }
            }

            var first_index = parseInt(startrow);
            var second_index = parseInt(startcolumn);
            var $target_cell = null;

            switch (direction) {
                case 'up':
                    first_index = ( startrow - rowspan );
                    break;
                case 'down':
                    first_index = ( startrow + rowspan );
                    break;
                case 'left':
                    second_index = ( startcolumn - colspan );
                    break;
                case 'right':
                    second_index = ( startcolumn + colspan );
                    break;
            }

            if ((first_index >= 0 && first_index < map.length) && map[first_index] && (second_index >= 0 && second_index < map[first_index].length)) {
                $target_cell = map[first_index][second_index];

                if (!$target_cell || ($target_cell[0] == $selected_node[0])) $target_cell = null;
                if ($target_cell && ('up' == direction || 'down' == direction) && (($target_cell.attr('colspan') || 1) != ($selected_node.attr('colspan') || 1))) $target_cell = null;
                if ($target_cell && ('left' == direction || 'right' == direction) && (($target_cell.attr('rowspan') || 1) != ($selected_node.attr('rowspan') || 1))) $target_cell = null;
            }
            return $target_cell;
        };

        var map = weEdContextMenuTableTools.tableToArray($selected_node);

        var startrow = $selected_node.parents('tr:first')[0].rowIndex;
        var startcolumn = 0;
        var row = map[startrow];

        for (var startcolumn = 0; startcolumn < row.length; startcolumn++) {
            if (row[startcolumn][0] == $selected_node[0]) break;
        }

        var selected_node_properties = {
            "node":     $selected_node,
            "rowspan":  ($selected_node.attr('rowspan') || 1),
            "colspan":  ($selected_node.attr('colspan') || 1),
            "startrow": startrow,
            "startcolumn": startcolumn
        };

        if (direction) return _mergeable_cell(selected_node_properties, direction);

        var cells = { "down": null, "right": null };

        var directions = ['down', 'up', 'right', 'left'];

        for (var i = 0; i < 4; i += 2) {
            var direction = directions[i];
            var mergeable_cell = _mergeable_cell(selected_node_properties, direction);
            cells[direction] = mergeable_cell;
        }

        return cells;
    };

    weEdContextMenuTableTools.prototype.insertColumn = function($selected_node, insertBefore) {
        function getCellColIndex($selected_node, isStart ) {
            var $row = $selected_node.parents('tr:first'),
                rowCells = $row[0].cells;

            var colIndex = 0;
            for (var i = 0; i < rowCells.length; i++) {
                var $mapCell = jQuery(rowCells[i]);

                colIndex += isStart ? 1 : ($mapCell.attr('colspan') || 1);
                if ($mapCell[0] == $selected_node[0])
                    break;
            }

            return colIndex -1;
        };

        function getColumnsIndices($selected_node, isStart ) {
            var retval = isStart ? Infinity : 0;
            var colIndex = getCellColIndex($selected_node, isStart);
            if (isStart ? colIndex < retval : colIndex > retval) retval = colIndex;
            return retval;
        }

        var $cell = null;

        var $table = $selected_node.parents('table:first');
            startCol =  getColumnsIndices($selected_node, 1),
            lastCol =  getColumnsIndices($selected_node),
            colIndex = insertBefore ? startCol : lastCol;

        var map = weEdContextMenuTableTools.tableToArray($selected_node),
            cloneCol = [],
            nextCol = [],
            height = map.length;

        for (var i = 0; i < height; i++) {
            cloneCol.push(map[i][colIndex]);
            var nextCell = insertBefore ? map[i][colIndex - 1] : map[i][colIndex + 1];
            nextCell && nextCol.push(nextCell);
        }

        for (i = 0; i < height; i++) {
            if (cloneCol[i]) {
                var cell;
                if ((cloneCol[i].attr('colspan') || 1) > 1 && nextCol.length && nextCol[i][0] == cloneCol[i][0]) {
                    cell = cloneCol[i];
                    cell.attr('colspan', (cell.attr('colspan') || 1) + 1);
                }
                else {
                    cell = cloneCol[i].clone();
                    cell.html('');
                    cell.removeAttr('colspan');
                    if (insertBefore) cloneCol[i].before(cell);
                    else cloneCol[i].after(cell);
                    cell.html('<br />');
                }

                if (!$cell) $cell = cell;

                i += (cell.attr('rowspan') || 1) - 1;
            }
        }
        if ($cell && $cell[0]) this.editor.focusToNode($cell[0]);
    };

    weEdContextMenuTableTools.prototype._ = function(key) {
        return this.editor._(key, this.plugin_name);
    };
};