/*
 * Copyright 2012 Sébastien Raud
 *
 * This file is part of beCms.
 *
 * beCms is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * beCms is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with beCms.  If not, see <http://www.gnu.org/licenses/>.
 */

if (undefined === weEdHtmlToXhtml) {
    var weEdHtmlToXhtml = function(options, editor) {
        this.editor = editor;
        this.plugin_name = 'html_to_xhtml';

        var self = this;

        this.options = jQuery.extend(true, {}, {
            "xhtml_tidy": {
                "doctype": self.editor.options.doctype,
                "dtd_path": self.editor.options["plugins_path"] + "XhtmlTidyDtd/",
                "dtd_file": self.editor.options.doctype + ".js",
                "dtd_loader": self.editor.loadJs
            }
        }, options || {});

        this.clean = XhtmlNode.prototype.clean;
        this.clean_node = XhtmlNode.prototype.cleanNode;


        this.editor.loadLanguage(this.plugin_name);

        this.editor.options["custom_buttons"]["cleanup"] = {
            "command": self.executeCleanup,
            "image": self.editor.options['images_path'] + "cleanup.gif",
            "type": "no-toggle",
            "title": self.editor._('cleanup', self.plugin_name),
        };

        // clean initial source
        var html = this.editor.$element.val();
            html = this.removeLastBr(html);
            html = this.cleanForIFrame(html);
            this.editor.$element.val(html);

        this.editor.bind('setDoctype.html_to_xhtml', function(data) {
            self.options.xhtml_tidy['doctype'] = data.doctype;
            self.options.xhtml_tidy['dtd_file'] = data.doctype + '.js';
        });

        this.editor.bind('updateTextarea.pre.html_to_xhtml', function(data) {
            if (data.html) {
                data.html = jQuery.trim(data.html);
                data.html = self.removeLastBr(data.html);
                data.html = self.cleanForTextarea(data.html);
            }
        });

        this.editor.bind('updateIFrame.pre.html_to_xhtml', function(data) {
            if (data.html) {
                data.html = jQuery.trim(data.html);
                data.html = self.removeLastBr(data.html);
                data.html = self.cleanForTextarea(data.html);
                data.html = self.cleanForIFrame(data.html);
            }
        });

        this.editor.bind('submit.html_to_xhtml', function(data) {
            var html = this.is_wysiwyg_mode ? this.$body.html() : this.$element.val();
            html = self.cleanForTextarea(html);
            this.updateTextarea(html);
        });

        this.editor.bind('showHtmlElementsPath.html_to_xhtml', function() {
            var l = this.html_path.length;
            for (var i = 0; i < l; i++) {
                if ('b' == this.html_path[i]) this.html_path[i] = 'strong';
                if ('i' == this.html_path[i]) this.html_path[i] = 'em';
                if ('u' == this.html_path[i] || 's' == this.html_path[i] || 'strike' == this.html_path[i]) this.html_path[i] = 'span';
            }
            this.statusbarLeftMessage(this.html_path.join(' &gt; '));
        });
    };

    weEdHtmlToXhtml.prototype.updateIFrame = function(html) {
        html = this.removeLastBr(html);
        html = this.cleanBrInPre(html);
        html = this.cleanForIFrame(html);
    };

    weEdHtmlToXhtml.prototype.removeLastBr = function(html) {
        var $d = jQuery('<div>').append(html);
        $d.find('p br:last-child, div br:last-child').filter(function(index) { return !jQuery(this)[0].nextSibling || '' == jQuery.trim(jQuery(jQuery(this)[0].nextSibling).text()); }).remove();
        return $d.html();
    };

    weEdHtmlToXhtml.prototype.executeCleanup = function($button) {
        var html = $button.weed.$body.html();
        $button.weed.updateIFrame(html);
        $button.weed.statusbarRightMessage('Nettoyage effectué', 2500);
        $button.weed.focusToFirstNode();
    };

    // textarea => iframe
    weEdHtmlToXhtml.prototype.cleanForIFrame = function(html) {
        var self = this;
        XhtmlNode.prototype.clean = function() {
            self.clean.apply(this);

            function change_node(node) {
                if ('strong' == node.name) node.changeTo('b');
                if ('em' == node.name) node.changeTo('i');

                if ('span' == node.name && node.hasAttribute('style')) {
                    var style = node.getAttributeValue('style').toLowerCase();
                    if (style.match(/text-decoration\s*\:\s*underline/)) node.name = 'u';
                    else if (style.match(/text-decoration\s*\:\s*line-through/)) node.name = 'strike';
                }

                if ('ul' == node.name || 'ol' == node.name) {
                    var parent = node.getParent();
                    if (parent && 'li' == parent.name) {
                        node.insertAfter(parent);
                    }
                }
            };

            var first_node = this;
            change_node(this);

            if (this.hasChildren()) {
                var node = this.getChildren();
                while (node && node != first_node) {
                    change_node(node);
                    if (node.hasChildren()) node = node.getChildren();
                    else if (node.getNext()) node = node.getNext();
                    else {
                        while (node.getParent() && node != first_node) {
                            node = node.getParent();
                            if (node.getNext()) {
                                node = node.getNext();
                                break;
                            }
                        }
                    }
                }
            }
        };

        var xhtml_node_clean = new XhtmlTidy(html, this.options['xhtml_tidy']);
        var html = xhtml_node_clean.clean();
        XhtmlNode.prototype.clean = this.clean;
        return html;
    };

    // iframe => textarea
    weEdHtmlToXhtml.prototype.cleanForTextarea = function(html) {
        var self = this;
        XhtmlNode.prototype.cleanNode = function(options) {
            if ('b' == this.name) this.changeTo('strong');
            if ('i' == this.name) this.changeTo('em');
            if ('u' == this.name.toLowerCase()) {
                this.changeTo('span');
                this.addAttribute('style', 'text-decoration: underline');
                this.cleanAttributes();
            }
            if ('s' == this.name.toLowerCase() || 'strike' == this.name.toLowerCase()) {
                this.changeTo('span');
                this.addAttribute('style', 'text-decoration: line-through');
                this.cleanAttributes();
            }
            self.clean_node.apply(this, [options]);
        };

        var xhtml_node_clean = new XhtmlTidy(html, this.options['xhtml_tidy']);
        html = xhtml_node_clean.clean();
        XhtmlNode.prototype.cleanNode = this.clean_node;
        return html;
    };


    function XhtmlTidy(html, options) {
        this.xhtml_tidy_version = "1.0.0";

        this.html = html;
        this.options = {};
        this.dtd = {};
        this.parser = null;

        this.checkOptions(options);

        if (this.loadDtd()) {
            this.parser = new HtmlParser(this);
            this.root_node = null;
        }
    };

    XhtmlTidy.prototype.checkOptions = function(options) {
        options = options || {};
        var defaults = {
            "doctype": "xhtml-1.0-strict",
            "dtd_path": null,
            "dtd_file": "dtd.js", // one file with all dtd or { "dtd": "all_dtd.js", "xhtml-1.0-strict": "xhtml-1.0-strict.js" }
            "dtd_loader": null
        };

        for (var i in defaults) {
            this.options[i] = options[i] ? options[i] : defaults[i];
        }
        if (!XhtmlTidy.tools.inArray(this.options['doctype'], ["xhtml-1.0-strict", "xhtml-1.0-transitional"])) this.options['doctype'] = "xhtml-1.0-strict";
    };

    XhtmlTidy.prototype.loadDtd = function() {
        var self = this;

        function load_dtd() {
            if (undefined === XhtmlTidy.DTD[self.options['doctype']]) {
                alert('DTD "' + self.options['doctype'] + '" not found');
                return false;
            }

            self.dtd = XhtmlTidy.DTD[self.options['doctype']];
            return true;
        };

        if (undefined === XhtmlTidy.DTD[this.options['doctype']] && this.options['dtd_loader'] && 'function' === typeof this.options['dtd_loader']) {
            var url = this.options['dtd_path'] || '';
            var file = null;
            if ('object' === typeof this.options['dtd_file']) {
                if (this.options['dtd_file'][this.options['doctype']]) file = this.options['dtd_file'][this.options['doctype']];
                else if (this.options['dtd_file']["dtd"]) file = this.options['dtd_file']["dtd"];
            }
            else if ('string' === typeof this.options['dtd_file']) file = this.options['dtd_file'];

            if (!file) {
                alert('no dtd file');
                return false;
            }
            url = url + file;

            this.options['dtd_loader'](url);
        }

        return load_dtd();
    };

    XhtmlTidy.prototype.parse = function() {
        if (!this.parser) return false;

        this.parser.parse(this.html);
        this.root_node = this.parser.node;
        return true;
    };

    XhtmlTidy.prototype.clean = function() {
        if (this.parse()) {
            this.root_node.clean(this.options);
            return this.root_node.toText().replace(/<\/?body(.*)>/g, '');
        }
        return this.html;
    };

    XhtmlTidy.tools = {
        "makeMap": function(array) {
            var object = {};
            for (var i in array) object[array[i]] = true;
            return object;
        },
        "trim": function(str) {
            if (!str) return '';
            return str.replace(/^\s*/, '').replace(/\s*$/, '');
        },
        "removeManyBlanks": function(str) {
            if (!str) return '';
            return str.replace(/\s+/g, ' ');
        },
        "inArray": function(value, array) {
            for (var i in array) {
                if (value == array[i]) return true;
            }
            return false;
        },
        "keyExists": function(value, object) {
            return 'undefined' != typeof object[value];
        }
    };

    XhtmlTidy.DTD = { };


    /*
     * XhtmlNode is a reprensentation of an xhtml element.
     */

    /**
     * Constructor.
     *
     * @param  string  name        Element name (eg : "a", "div", ...).
     * @param  array   attributes  An array of object name, value of attributes (eg : [{ "name": "href", "value": "http://www.fsf.org/" }, { "name": "id", "value": "link_to_fsf" }, ...]).
     */
    function XhtmlNode(name, attributes, xhtml_tidy) {
        /* node name, eg "a" */
        this.name = name;

        /* attributes list */
        this.attributes = [];
        this.count_attributes = 0;

        /* parent node */
        this.parent_node = null;

        /* previous and next siblings nodes */
        this.previous_node = null;
        this.next_node = null;

        /* children nodes */
        this.first_child = null;
        this.last_child = null;

        /* node text value, eg <a ...>text</a> */
        this.text = null;

        if (undefined !== attributes && !('length' in attributes) && 'xhtml_tidy_version' in attributes) {
            xhtml_tidy = attributes;
            attributes = null;
        }

        this.xhtml_tidy = xhtml_tidy;
        this.dtd = this.xhtml_tidy.dtd;
        if (attributes) this.setAttributes(attributes);

        this.is_empty = this.dtd.empty[name] || XhtmlNode.TEXT == name || false;
        this.is_inline = this.dtd.inline[name] || XhtmlNode.TEXT == name || false;
        this.is_block = ! this.is_inline;
    };

    /**
     * Indicates if node is an empty node.
     *
     * @return boolean
     */
    XhtmlNode.prototype.isEmpty = function() {
        return this.is_empty;
    };

    /**
     * Indicates if node is an inline node.
     *
     * @return boolean
     */
    XhtmlNode.prototype.isInline = function() {
        return this.is_inline;
    };

    /**
     * Indicates if node is a block node.
     *
     * @return boolean
     */
    XhtmlNode.prototype.isBlock = function() {
        return this.is_block;
    };

    /**
     * Indicates if a node is a specific node.
     *
     * @param  mixed  name  string / array, node name.
     * @return boolean
     */
    XhtmlNode.prototype.isNode = function(name) {
        if ('string' == typeof name) return this.name == name;
        else return XhtmlTidy.tools.inArray(this.name, name);
    };

    /**
     * Returns an attribute.
     *
     * @param  string  name  Attribute name.
     * @return object  { "name": "attribute_name", "value": "attribute_value" }, null if attribute not exists.
     */
    XhtmlNode.prototype.getAttribute = function(name) {
        name = name.toLowerCase();

        for (var i = 0; i < this.count_attributes; i++) {
            if (this.attributes[i].name.toLowerCase() == name) return this.attributes[i];
        }
        return null;
    };

    /**
     * Indicates if an attributes exists.
     *
     * @param  string  name  Attribute name.
     * @return boolean
     */
    XhtmlNode.prototype.hasAttribute = function(name) {
        name = name.toLowerCase();

        for (var i = 0; i < this.count_attributes; i++) {
            if (this.attributes[i].name.toLowerCase() == name) return true;
        }
        return false;
    };

    /**
     * Returns an attribute value.
     *
     * @param  string  name  Attribute name.
     * @return string
     */
    XhtmlNode.prototype.getAttributeValue = function(name) {
        name = name.toLowerCase();

        for (var i = 0; i < this.count_attributes; i++) {
            if (this.attributes[i].name.toLowerCase() == name) return this.attributes[i].value;
        }
        return null;
    };

    /**
     * Returns attributes.
     *
     * @return string
     */
    XhtmlNode.prototype.getAttributes = function() {
        return this.attributes;
    };

    /**
     * Add an attribute.
     *
     * @param  string  name   Attribute name.
     * @param  string  value  Attribute value.
     * @return void
     */
    XhtmlNode.prototype.addAttribute = function(name, value) {
        this.attributes.push({"name": name, "value": value, "escaped": value.replace(/(^|[^\\])"/g, '$1\\\"') }); // @ ???
        this.count_attributes++;
    };


    XhtmlNode.prototype.removeAttribute = function(name) {
        name = name.toLowerCase();

        for (var i = 0; i < this.count_attributes; i++) {
            if (this.attributes[i].name.toLowerCase() == name) {
                this.attributes = this.attributes.slice(0, i - 1).concat(this.attributes.slice(i + 1, this.count_attributes - 1));
                this.count_attributes = this.attributes.length;
                break;
            }
        }
    };

    /**
     * Sets attributes.
     *
     * @param  array   attributes  An array of object name, value of attributes (eg : [{ "name": "href", "value": "http://www.fsf.org/" }, { "name": "id", "value": "link_to_fsf" }, ...]).
     * @return void
     */
    XhtmlNode.prototype.setAttributes = function(attributes) {
        this.attributes = attributes;
        this.count_attributes = attributes.length;
    };

    /**
     * Returns the previous sibling node.
     *
     * @return XhtmlNode
     */
    XhtmlNode.prototype.getPrevious = function() {
        return this.previous_node;
    };

    /**
     * Sets the previous sibling node.
     *
     * @param  XhtmlNode  node  The previous sibling node.
     * @return void
     */
    XhtmlNode.prototype.setPrevious = function(node) {
        var parent = this.getParent();
        var is_first_child = (parent && parent.first_child == this ? true : false);

        this.previous_node = node;
        if (!node) return;
        if (node.getParent() != this.getParent()) node.setParent(this.getParent());
        if (node.getNext() != this) node.setNext(this);
        if (is_first_child) parent.first_child = node;
    }

    /**
     * Returns the next sibling node.
     *
     * @return XhtmlNode
     */
    XhtmlNode.prototype.getNext = function() {
        return this.next_node ;
    };

    /**
     * Sets the next sibling node.
     *
     * @param  XhtmlNode  node  The next sibling node.
     * @return void
     */
    XhtmlNode.prototype.setNext = function(node) {
        var parent = this.getParent();
        var is_last_child = (parent && parent.last_child == this ? true : false);

        this.next_node = node;

        if (!node) return;
        if (node.getParent() != this.getParent()) node.setParent(this.getParent());
        if (node.getPrevious() != this) node.setPrevious(this);
        if (is_last_child) parent.last_child = node;
    };

    /**
     * Returns the parent node.
     *
     * @return XhtmlNode
     */
    XhtmlNode.prototype.getParent = function() {
        return this.parent_node;
    };

    /**
     * Sets the parent node.
     *
     * @param  XhtmlNode  node  The parent node.
     * @return void
     */
    XhtmlNode.prototype.setParent = function(node) {
        this.parent_node = node;
        if (!node) return;
        if (!node.isChild(this)) node.addChild(this);
    };

    /**
     * Indicates if a tag name exists has parent node.
     *
     * @return boolean
     */
    XhtmlNode.prototype.hasParent = function(name) {
        var parent = this.getParent();
        while (parent) {
            if (parent.name == name) return true;
            parent = parent.getParent();
        }
        return false;
    };

    /**
     * Returns first child node.
     *
     * @return XhtmlNode
     */
    XhtmlNode.prototype.getChildren = function() {
        return this.first_child;
    };

    /**
     * Indicates if node has children nodes.
     *
     * @return boolean
     */
    XhtmlNode.prototype.hasChildren = function() {
        return this.first_child != null;
    };

    /**
     * Add a child node.
     *
     * @param  XhtmlNode  node  The new child node.
     * @return void
     */
    XhtmlNode.prototype.addChild = function(node) {
        if (!node) return;

        node.remove();

        if (!this.hasChildren()) {
            this.first_child = node;
            this.last_child = node;
            node.setPrevious(null);
            node.setNext(null);
        }
        else {
            this.last_child.setNext(node);
            this.last_child = node;
        }

        if (node.getParent() != this) node.setParent(this);
    };

    /**
     * Indicates if a node is a child of current node.
     *
     * @param  XhtmlNode  node  Node to test.
     * @return boolean
     */
    XhtmlNode.prototype.isChild = function(node) {
        var child = this.getChildren();
        while (child) {
            if (child == node) return true;
            child = child.getNext();
        }

        return false;
    };


    /**
     * Insert current node before node.
     *
     * @param  XhtmlNode  node
     * @return void
     */
    XhtmlNode.prototype.insertBefore = function(node) {
        this.remove();
        if (!node) return;

        var parent = node.getParent();
        var previous = node.getPrevious();

        this.parent_node = parent;

        if (parent && parent.first_child == node)  parent.first_child = this;
        if (previous) previous.setNext(this);
        this.setNext(node);
    };

    /**
     * Insert current node after node.
     *
     * @param  XhtmlNode  node
     * @return void
     */
    XhtmlNode.prototype.insertAfter = function(node) {
        this.remove();
        if (!node) return;

        var parent = node.getParent();
        var next = node.getNext();

        this.parent_node = parent;

        if (parent && parent.last_child == node) parent.last_child = this;
        if (next) next.setPrevious(this);
        this.setPrevious(node);
    };

    /**
     * Removes a child node.
     *
     * @param  XhtmlNode  node
     * @return boolean
     */
    XhtmlNode.prototype.removeChild = function(node) {
        if (!this.isChild(node)) return false;

        var is_first_child = (this.first_child == node);
        var is_last_child = (this.last_child == node);

        var previous = node.getPrevious();
        var next = node.getNext();

        if (is_first_child) this.first_child = next;
        if (is_last_child) this.last_child = previous;

        if (previous && next) previous.setNext(next);
        else if (previous) previous.setNext(null);
        else if (next) next.setPrevious(null);

        node.previous_node = null;
        node.next_node = null;
        node.parent_node = null;

        return true;
    };

    /**
     * Remove current node.
     *
     * @return void
     */
    XhtmlNode.prototype.remove = function() {
        var parent = this.getParent();

        if (parent) return parent.removeChild(this);

        var previous = this.getPrevious();
        var next = this.getNext();

        this.previous_node = null;
        this.next_node = null;
        this.parent_node = null;

        if (previous && next) previous.setNext(next);
        else if (previous) previous.setNext(null);
        else if (next) next.setPrevious(null);

        return true;
    };

    /**
     * Returns the node text.
     *
     * @return string
     */
    XhtmlNode.prototype.getText = function() {
        return this.text;
    };

    /**
     * Sets the node text.
     *
     * @param  string  text
     * @return void
     */
    XhtmlNode.prototype.setText = function(text) {
        this.text = ('pre' == this.name || this.hasParent('pre') ? text : XhtmlTidy.tools.removeManyBlanks(text));
    };

    /**
     * Executes cleanup functions.
     *
     * @return void
     */
    XhtmlNode.prototype.clean = function(options) {
        this.cleanTree();
        this.cleanOrphans();
        this.cleanTreeAttributes();

        if (XhtmlNode.CLEAN_NODES.length) {
            var node = XhtmlNode.CLEAN_NODES.shift();
            while (node) {
                node.cleanNode(options);
                node = XhtmlNode.CLEAN_NODES.shift();
            }
        }
    };

    /**
     * Adds node in XhtmlNode.CLEAN_NODES, to clean it.
     *
     * @return void
     */
    XhtmlNode.prototype.pushToClean = function() {
        if (!XhtmlTidy.tools.inArray(this, XhtmlNode.CLEAN_NODES)) {
            XhtmlNode.CLEAN_NODES.push(this);
        }
    };

    /**
     * Cleans orphans nodes (eg </tr>[empty text node]<td> : [empty text node] is removed).
     *
     * @return void
     */
    XhtmlNode.prototype.cleanOrphans = function() {
        // removes next orphans nodes
        var i_nodes = XhtmlNode.CLEAN_NODES.length;
        for (var i = i_nodes - 1; i >= 0; i--) {
            if (XhtmlNode.CLEAN_NODES[i].isOrphanNode()) {
                XhtmlNode.CLEAN_NODES[i].remove();
                XhtmlNode.CLEAN_NODES = XhtmlNode.CLEAN_NODES.slice(0, i).concat(XhtmlNode.CLEAN_NODES.slice(i + 1, i_nodes));
            }
        }
    }

    /**
     * Adds all children in XhtmlNode.CLEAN_NODES.
     *
     * @return void
     */
    XhtmlNode.prototype.cleanTree = function() {
        var first_node = this;

        this.pushToClean();

        if (this.hasChildren()) {
            var node = this.getChildren();
            while (node && node != first_node) {
                node.pushToClean();
                if (node.hasChildren()) node = node.getChildren();
                else if (node.getNext()) node = node.getNext();
                else {
                    while (node.getParent() && node != first_node) {
                        node = node.getParent();
                        if (node.getNext()) {
                            node = node.getNext();
                            break;
                        }
                    }
                }
            }
        }
    };

    /**
     * Clean attributes for node and children nodes.
     *
     * @return void
     */
    XhtmlNode.prototype.cleanTreeAttributes = function() {
        var first_node = this;

        first_node.cleanAttributes();

        if (this.hasChildren()) {
            var node = this.getChildren();
            while (node && node != first_node) {
                node.cleanAttributes();
                if (node.hasChildren()) node = node.getChildren();
                else if (node.getNext()) node = node.getNext();
                else {
                    while (node.getParent() && node != first_node) {
                        node = node.getParent();
                        if (node.getNext()) {
                            node = node.getNext();
                            break;
                        }
                    }
                }
            }
        }
    };

    /**
     * Executes cleanups node functions.
     *
     * @return boolean  unisignifiant.
     */
    XhtmlNode.prototype.cleanNode = function(options) {
        // change to text node if necessary
        if ((XhtmlNode.TEXT != this.name && !this.dtd.elements[this.name]) ||
            (('big' == this.name || 'small' == this.name || 'sub' == this.name || 'sup' == this.name) && this.hasParent('pre'))) {
            this.changeTo(XhtmlNode.TEXT);
            this.pushToClean();
            return true;
        }

        // removes node if necessary
        if ((this.isOrphanNode()) ||
            ('label' == this.name && this.hasParent('label')) ||
            (('img' == this.name || 'object' == this.name) && this.hasParent('pre'))) {
            if (this.getParent()) this.getParent().pushToClean();
            this.remove();
            return false;
        }

        this.cleanFormElements();
        this.cleanTableElements();
        this.cleanListsElements();

        var parent = this.getParent();
        if (parent) {
            if (XhtmlNode.TEXT != this.name && XhtmlNode.TEXT != parent.name && 'body' != parent.name) {
                if (!XhtmlTidy.tools.inArray(this.name, this.dtd.elements[parent.name].children)) {
                    // move current just after parent,
                    // move all next in new parent similar to this parent

                    // <body><h1><hr />xyz</h1></body> => <body><hr /><h1>xyz</h1></body>
                    if (!this.getPrevious()) {
                        this.insertBefore(parent);
                        this.pushToClean();
                        parent.pushToClean();
                        return true;
                    }

                    var can_duplicate_blocks = ['p', 'div', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6']

                    if ((parent.isInline() && 'a' != parent.name && 'label' != parent.name) ||
                        (XhtmlTidy.tools.inArray(parent.name, can_duplicate_blocks))) {
                        // <body><h1>abc<hr />xyz</h1></body> => <body><h1>abc</h1><hr /><h1>xyz</h1></body>
                        var new_parent = new XhtmlNode(parent.name, this.xhtml_tidy);

                        var next = this.getNext();
                        var next_next = null;

                        while (next) {
                            next_next = next.getNext();
                            new_parent.addChild(next);
                            next = next_next;
                        }

                        this.insertAfter(parent);
                        if (new_parent.hasChildren()) new_parent.insertAfter(this);
                        this.pushToClean();
                        new_parent.pushToClean();
                        return true;
                    }
                    else {
                        var next = this.getNext();
                        var next_next = null;
                        var insert_after = null;

                        this.insertAfter(parent);
                        insert_after = this;

                        while (next) {
                            next_next = next.getNext();
                            next.insertAfter(insert_after);
                            insert_after = next;
                            next = next_next;
                        }

                        parent.pushToClean();
                        return true;
                    }
                }
            }
            if ('body' == parent.name && this.isInline()) {
                // move inlines in P
                var previous = this.getPrevious();
                var next = this.getNext();

                if (previous && 'p' == previous.name) {
                    previous.addChild(this);
                    previous.pushToClean();
                    return true;
                }
                if (next && 'p' == next.name) {
                    if (!next.hasChildren()) {
                        next.addChild(this);
                    }
                    else {
                        this.insertBefore(next.getChildren());
                        this.clean();
                    }
                    this.pushToClean();
                    return true;
                }

                var new_parent = new XhtmlNode('p', this.xhtml_tidy);
                new_parent.insertBefore(this);
                new_parent.addChild(this);

                new_parent.pushToClean();
                return true;
            }
        }

        return true;
    };

    /* clean methods */

    /**
     * Indicates if node is an orphan node.
     *
     * @return boolean
     */
    XhtmlNode.prototype.isOrphanNode = function() {
        var preserve_if_attributes = ['button', 'span', 'div'];
        var preserve_if_empty = ['dd', 'li', 'option', 'textarea', 'td', 'th', 'iframe'];//, 'body', 'head', 'html'];

        if (this.hasParent('pre')) return false;

        var parent = this.getParent();
        var previous = this.getPrevious();
        var next = this.getNext();

        if (XhtmlTidy.tools.inArray(this.name, preserve_if_empty)) {
            return false;
        }

        if (XhtmlNode.TEXT == this.name) {
            if (!this.text || '' == XhtmlTidy.tools.trim(this.text)) {
                if (!this.hasChildren() && previous && this.dtd.block[previous.name]) {
                    return true;
                }
                if (!this.hasChildren() && next && this.dtd.block[next.name]) {
                    return true;
                }
                if (!this.hasChildren() && !previous && !next) {
                    return true;
                }
            }
        }
        else if (!this.isEmpty() && !this.hasChildren()) {
            if (XhtmlTidy.tools.inArray(this.name, preserve_if_attributes) && this.count_attributes){
                return false;
            }
            if (XhtmlTidy.tools.inArray(this.name, preserve_if_empty)) {
                return false;
            }
            else {
                return true
            }
        }

        return false;
    };

    /**
     * Clean forms elements.
     *
     * @return void
     */
    XhtmlNode.prototype.cleanFormElements = function() {
        var parent = this.getParent();
        var previous = this.getPrevious();
        var next = this.getNext();

        var new_parent_element = null;

        //var forms_elements = ['button', 'fieldset', 'input', 'label', 'legend', 'option', 'optgroup', 'select', 'textarea'];

        // legend in fieldset
        if ('legend' == this.name) {
            if (!this.hasParent('fieldset')) {
                new_parent_element = (previous && 'fieldset' == previous.name ? previous : (next && 'fieldset' == next.name ? next : null));

                if (new_parent_element) {
                    if (new_parent_element.hasChildren()) this.insertBefore(new_parent_element.getChildren());
                    else new_parent_element.addChild(this);

                    new_parent_element.pushToClean();
                    return true;
                }

                this.changeTo(XhtmlNode.TEXT);
                this.pushToClean();
                return true;
            }
        }

        // optgroup and options in select
        if (('optgroup' == this.name || 'option' == this.name) && !this.hasParent('select')) {
            this.changeTo(XhtmlNode.TEXT);
            this.pushToClean();
            return true;
        }

        // @todo textarea : convert children tags to text
        if ('textarea' == this.name) {
            ;
        }

        // controls in p
        if (XhtmlTidy.tools.inArray(this.name, ['button', 'input', 'label', 'select', 'textarea']) &&
            parent && 'form' == parent.name) {
            new_parent_element = (previous && 'p' == previous.name ? previous : (next && 'p' == next.name ? next : null));

            if (new_parent_element) {
                if (new_parent_element == previous || !new_parent_element.hasChildren()) new_parent_element.addChild(this);
                else this.insertBefore(new_parent_element.getChildren());
            }
            else {
                new_parent_element = new XhtmlNode('p', this.xhtml_tidy);
                new_parent_element.insertBefore(this);
                new_parent_element.addChild(this);
            }
            new_parent_element.pushToClean();
            return true;
        }

        // inlines in form in p
        if (this.isInline() && parent && 'form' == parent.name) {
            new_parent_element = (previous && 'p' == previous.name ? previous : (next && 'p' == next.name ? next : null));

            if (new_parent_element) {
                if (new_parent_element == previous || !new_parent_element.hasChildren()) new_parent_element.addChild(this);
                else this.insertBefore(new_parent_element.getChildren());
            }
            else {
                new_parent_element = new XhtmlNode('p', this.xhtml_tidy);
                new_parent_element.insertBefore(this);
                new_parent_element.addChild(this);
            }
            new_parent_element.pushToClean();
            return true;
        }

        return true;

    };

    /**
     * Add parents to table elements if necessary.
     *
     * @return void
     */
    XhtmlNode.prototype.cleanTableElements = function() {
        var parent = this.getParent();
        var previous = this.getPrevious();
        var next = this.getNext();

        var new_parent_element = null;

        var table_elements = ['td', 'th', 'tr', 'thead', 'tbody', 'tfoot', 'table'];

        // caption in table
        if ('caption' == this.name && (!parent || 'table' != parent.name)) {
            new_parent_element = (previous && 'table' == previous.name ? previous : (next && 'table' == next.name ? next : null));

            if (new_parent_element) {
                new_parent_element.addChild(this);
                new_parent_element.pushToClean();
                return true;
            }

            if (next && XhtmlTidy.tools.inArray(next.name, table_elements)) {
                var new_parent_element = new XhtmlNode('table', this.xhtml_tidy);
                new_parent_element.insertBefore(this);
                new_parent_element.addChild(this);
                new_parent_element.pushToClean();
                return true;
            }

            this.remove();
            return true;
        }

        // colgroup in table
        if ('colgroup' == this.name && (!parent || 'table' != parent.name)) {
            new_parent_element = (previous && 'table' == previous.name ? previous : (next && 'table' == next.name ? next : null));

            if (new_parent_element) {
                new_parent_element.addChild(this);
                new_parent_element.pushToClean();
                return true;
            }

            if (next && XhtmlTidy.tools.inArray(next.name, table_elements)) {
                var new_parent_element = new XhtmlNode('table', this.xhtml_tidy);
                new_parent_element.insertBefore(this);
                new_parent_element.addChild(this);
                new_parent_element.pushToClean();
                return true;
            }

            this.remove();
            return true;
        }

        // col in table
        if ('col' == this.name && (!parent || ('colgroup' != parent.name && 'table' != parent.name))) {
            new_parent_element = (previous && 'table' == previous.name ? previous : (next && 'table' == next.name ? next : null));

            if (new_parent_element) {
                new_parent_element.addChild(this);
                new_parent_element.pushToClean();
                return true;
            }

            if (next && XhtmlTidy.tools.inArray(next.name, table_elements)) {
                var new_parent_element = new XhtmlNode('table', this.xhtml_tidy);
                new_parent_element.insertBefore(this);
                new_parent_element.addChild(this);
                new_parent_element.pushToClean();
                return true;
            }

            this.remove();
            return true;
        }

        // th and td in tr, otherwise in txxx or table
        if (!this.isOrphanNode() && ('th' == this.name || 'td' == this.name) && (!parent || 'tr' != parent.name)) {
            new_parent_element = (previous && 'tr' == previous.name ? previous : (next && 'tr' == next.name ? next : null));

            // has tr in previous or next
            if (new_parent_element) {
                new_parent_element.addChild(this);
                new_parent_element.pushToClean();
                return true;
            }

            new_parent_element = (previous && ('thead' == previous.name || 'tfoot' == previous.name || 'tbody' == previous.name) ? previous :
                                (next && ('thead' == next.name || 'tfoot' == next.name || 'tbody' == next.name) ? next :
                                    (previous && 'table' == previous.name ? previous : (next && 'table' == next.name ? next : null))));

            parent_tr_element = new XhtmlNode('tr', this.xhtml_tidy);
            parent_tr_element.insertBefore(this);
            parent_tr_element.addChild(this);
            parent_tr_element.pushToClean();

            if (new_parent_element) {
                new_parent_element.addChild(parent_tr_element)
                new_parent_element.pushToClean();
            }

            return true;
        }

        // tr in table if not in table, thead, tbody or tfoot
        if ('tr' == this.name) {
            this.cleanTableTrElement();
            // parent must be thead, tfoot or tbody
            if (!parent || ('thead' != parent.name && 'tfoot' != parent.name && 'tbody' != parent.name)) {
                new_parent_element = (previous && XhtmlTidy.tools.inArray(previous.name, ['thead', 'tfoot', 'tbody']) ? previous :
                                    (next && XhtmlTidy.tools.inArray(next.name, ['thead', 'tfoot', 'tbody']) ? next : null));

                if (!new_parent_element) {
                    new_parent_element = new XhtmlNode('tbody', this.xhtml_tidy);
                    new_parent_element.insertBefore(this);
                }

                new_parent_element.addChild(this);
                new_parent_element.pushToClean();
                return true;
            }
        }

        if ('thead' == this.name || 'tfoot' == this.name || 'tbody' == this.name) {
            this.cleanTableTxElement();
            new_parent_element = (previous && 'table' == previous.name ? previous : (next && 'table' == next.name ? next : null));

            if (new_parent_element) {
                new_parent_element.addChild(this);
                new_parent_element.pushToClean();
                return true;
            }

            // txxx in table
            if (!parent || 'table' != parent.name) {
                new_parent_element = new XhtmlNode('table', this.xhtml_tidy);
                new_parent_element.insertBefore(this);
                new_parent_element.addChild(this);
                new_parent_element.pushToClean();
                return true;
            }
        }

        if ('table' == this.name) {
            this.cleanTableElement();
        }

        return true;
    };

    /**
     * Clean a colgroup.
     *
     * @return void
     */
    XhtmlNode.prototype.cleanTableColgroupElement = function() {
        if ('colgroup' != this.name) return ;

        if (!this.hasChildren() && !this.count_attributes) {
            this.remove();
        }
    };

    /**
     * Clean tr.
     *
     * @return void
     */
    XhtmlNode.prototype.cleanTableTrElement = function() {
        if ('tr' != this.name) return ;

        var child = this.getChildren();
        var next = null;
        while (child) {
            next = child.getNext();
            if ('td' != child.name && 'th' != child.name) {
                var new_parent = new XhtmlNode('td', this.xhtml_tidy);
                new_parent.insertBefore(child);
                new_parent.addChild(child);
            }
            child = next;
        }
    };

    /**
     * Clean thead, tfoot or tbody.
     *
     * @return void
     */
    XhtmlNode.prototype.cleanTableTxElement = function() {
        if ('thead' != this.name && 'tbody' != this.name && 'tfoot' != this.name) return ;

        var child = this.getChildren();
        var next = null;
        while (child) {
            next = child.getNext();
            if ('tr' != child.name) {
                var new_parent = new XhtmlNode('tr', this.xhtml_tidy);
                new_parent.insertBefore(child);
                new_parent.addChild(child);
                new_parent.cleanTableTrElement();
            }
            child = next;
        }
    };

    /**
     * Clean table element.
     *
     * @return void
     */
    XhtmlNode.prototype.cleanTableElement = function() {
        if ('table' != this.name) return;

        var child = this.getChildren();
        var next = null;
        while (child) {
            next = child.getNext();
            switch (child.name) {
                case 'colgroup':
                    child.cleanTableColgroupElement();
                    break;

                case 'thead':
                case 'tbody':
                case 'tfoot':
                    child.cleanTableTxElement();
                    break;

                case 'tr':
                    child.cleanTableTrElement();
                    break;
            }
            child = next;
        }

        var reorder_children = { "caption": [], "colgroup": [], "thead": [], "tbody": [], "tfoot": [] };

        var child = this.getChildren();
        var next = null;
        while (child) {
            next = child.getNext();
            switch (child.name) {
                case 'caption':
                    reorder_children["caption"].push(child);
                    this.removeChild(child);
                    break;

                case 'colgroup':
                    reorder_children["colgroup"].push(child);
                    this.removeChild(child);
                    break;

                case 'col':
                    var new_colgroup = new XhtmlNode('colgroup', this.xhtml_tidy);
                    child.moveTo(new_colgroup);
                    reorder_children["colgroup"].push(new_colgroup);
                    break;

                case 'thead':
                    reorder_children["thead"].push(child);
                    this.removeChild(child);
                    break;

                case 'tbody':
                    reorder_children["tbody"].push(child);
                    this.removeChild(child);
                    break;

                case 'tfoot':
                    reorder_children["tfoot"].push(child);
                    this.removeChild(child);
                    break;

                case 'tr':
                    var new_tbody = new XhtmlNode('tbody', this.xhtml_tidy);
                    new_tbody.addChild(child);
                    reorder_children["tbody"].push(new_tbody);
                    break;

                default:
                    this.removeChild(child);
                    break;
            }
            child = next;
        }

        if (!reorder_children["tbody"].length) {
            this.remove();
            return true;
        }

        if (reorder_children["caption"].length) this.addChild(reorder_children["caption"][reorder_children["caption"].length - 1]);//.moveTo(new_table);

        if (reorder_children["colgroup"].length) {
            for (var i = 0; i < reorder_children["colgroup"].length; i++) {
                this.addChild(reorder_children["colgroup"][i]);
            }
        }

        if (reorder_children["thead"].length) {
            this.addChild(reorder_children["thead"][0]);
            // only one thead
            if (reorder_children["thead"].length > 1) {
                for (var i = 1; i < reorder_children["thead"].length; i++) {
                    var child = reorder_children["thead"][i].getChildren();
                    var next = null;
                    while (child) {
                        next = child.getNext();
                        reorder_children["thead"][0].addChild(child);
                        child = next;
                    }
                }
            }
        }
        if (reorder_children["tfoot"].length) {
            this.addChild(reorder_children["tfoot"][0]);
            // only one tfoot
            if (reorder_children["tfoot"].length > 1) {
                for (var i = 1; i < reorder_children["tfoot"].length; i++) {
                    var child = reorder_children["tfoot"][i].getChildren();
                    var next = null;
                    while (child) {
                        next = child.getNext();
                        reorder_children["tfoot"][0].addChild(child);
                        child = next;
                    }
                }
            }
        }
        if (reorder_children["tbody"].length) {
            for (var i = 0; i < reorder_children["tbody"].length; i++)
                this.addChild(reorder_children["tbody"][i]);
        }
    };

    /**
     * Check if list items elements have list parent, add it if necessary.
     *
     * @return void
     */
    XhtmlNode.prototype.cleanListsElements = function() {
        var parent = this.getParent();
        var previous = this.getPrevious();
        var next = this.getNext();

        // li in ul or ol
        if ('li' == this.name && (!parent || ('ul' != parent.name && 'ol' != parent.name))) {
            if (previous && ('ul' == previous.name || 'ol' == previous.name)) {
                previous.addChild(this);
                previous.pushToClean();
                return true;
            }

            if (next && ('ul' == next.name || 'ol' == next.name)) {
                next.addChild(this);
                next.pushToClean();
                return true;
            }

            var new_ul_parent = new XhtmlNode('ul', this.xhtml_tidy);
            new_ul_parent.insertBefore(this);
            new_ul_parent.addChild(this);
            new_ul_parent.pushToClean();
            return true;
        }

        // <ul><li>..</li><ul>...</ul></ul> => <ul><li>..<ul>...</ul></li></ul>
        if (('ul' == this.name || 'ol' == this.name) && (previous && 'li' == previous.name)) {
            previous.addChild(this);
            previous.pushToClean();
            return true;
        }

        // dt, dd in dl
        if (('dt' == this.name || 'dd' == this.name) && (!parent || 'dl' != parent.name)) {
            if (previous && 'dl' == previous.name) {
                previous.addChild(this);
                previous.pushToClean();
                return true;
            }

            if (next && 'dl' == next.name) {
                next.addChild(this);
                next.pushToClean();
                return true;
            }

            var new_dl_parent = new XhtmlNode('dl', this.xhtml_tidy);
            new_dl_parent.insertBefore(this);
            new_dl_parent.addChild(this);
            new_dl_parent.pushToClean();
            return true;
        }

        // not dt child in dt => move has next dd
        // p change to br
        if (parent && 'dt' == parent.name && XhtmlNode.TEXT != this.name && !XhtmlTidy.tools.inArray(this.name, this.dtd.elements[parent.name].children)) {
            if ('p' == this.name) {
                this.changeTo('br');
                return true;
            }

            var new_dd_parent = new XhtmlNode('dd', this.xhtml_tidy);
            new_dd_parent.insertAfter(parent);
            new_dd_parent.addChild(this);
            new_dd_parent.pushToClean();
            return true;
        }


        return true;
    };

    /**
     * Cleans attributes.
     *
     * @return void
     */
    XhtmlNode.prototype.cleanAttributes = function() {
        if (XhtmlNode.TEXT == this.name) return;
        if ('undefined' == typeof this.dtd.elements[this.name]) return ;

        this.cleanAttributesOldToStyle();
        this.cleanAttributesFixLangXmlLang();
        this.cleanAttributesValidate();
        this.cleanAttributesRequired();
        //this.getDefaultAttributeValue();
        this.cleanAttributesEntities();
    };

    /**
     * Converts deprecated attributes to style.
     *
     * @return void
     */
    XhtmlNode.prototype.cleanAttributesOldToStyle = function() {
        // @todo check if attribute not exists
        if (XhtmlTidy.tools.inArray(this.name, ['body', 'table', 'tr', 'td', 'th']) && this.hasAttribute('bgcolor')) {
            this.addAttribute('style', 'background-color: ' + this.getAttributeValue('bgcolor'));
        }
        if ('caption' == this.name && this.hasAttribute('align')) {
            this.addAttribute('style', 'caption-side: ' + this.getAttributeValue('align'));
        }
        if (XhtmlTidy.tools.inArray(this.name, ['img', 'input', 'object']) && this.hasAttribute('align')) {
            var value = this.getAttributeValue('align').toLowerCase();
            if ('left' == value || 'right' == value) this.addAttribute('style', 'float: ' + value);
            else this.addAttribute('style', 'vertical-align: ' + value);
        }
        if ('table' == this.name && this.hasAttribute('align')) {
            var value = this.getAttributeValue('align').toLowerCase();
            if ('left' == value || 'right' == value) this.addAttribute('style', 'float: ' + value);
            else this.addAttribute('style', 'margin: auto');
        }
    };

    /**
     * Fix lang and xml:lang.
     * @see http://www.w3.org/TR/xhtml1/#C_7
     *
     * @return void
     */
    XhtmlNode.prototype.cleanAttributesFixLangXmlLang = function() {
        if (this.hasAttribute('lang') && !this.hasAttribute('xml:lang')) {
            this.addAttribute('xml:lang', this.getAttributeValue('lang'));
        }

        else if (this.hasAttribute('xml:lang') && !this.hasAttribute('lang')) {
            this.addAttribute('lang', this.getAttributeValue('xml:lang'));
        }
    };

    /**
     * Preserves validated attributes.
     *
     * @return void
     */
    XhtmlNode.prototype.cleanAttributesValidate = function() {
        if (XhtmlNode.TEXT == this.name) return;
        if ('undefined' == typeof this.dtd.elements[this.name]) return ;

        var current_styles = {};
        var current_classes = [];
        var attributes = [];

        for (var i = 0; i < this.count_attributes; i++) {
            var name = this.attributes[i].name.toLowerCase();
            var value = this.attributes[i].value;

            if (XhtmlTidy.tools.keyExists(name, this.dtd.elements[this.name].attributes)) {
                if ('style' == name) {
                    var styles = value.split(';');
                    for (var j = 0; j < styles.length; j++) {
                        var style_values = styles[j].split(':');
                        if (2 == style_values.length) {
                            style_values[0] = XhtmlTidy.tools.trim(style_values[0]);
                            style_values[1] = XhtmlTidy.tools.trim(style_values[1]);
                            if (style_values[0].length && style_values[1].length) current_styles[style_values[0].toLowerCase()] = style_values[1];
                        }
                    }
                }
                else if ('class' == name) {
                    current_classes.push(value);
                }
                else {
                    var expr = this.dtd.attributes[name][this.dtd.elements[this.name].attributes[name]];
                    if (!expr) attributes.push({ "name": name, "value": value });
                    else if (value.match(expr)) attributes.push({ "name": name, "value": value });
                }
            }
        }

        var style = [];
        for (var i in current_styles) style.push(i + ': ' + current_styles[i]);
        if (style.length) attributes.push({ "name": "style", "value": style.join('; ') });
        if (current_classes.length) attributes.push({ "name": "class", "value": XhtmlTidy.tools.trim(current_classes.join(' ')) });

        this.setAttributes(attributes);
    };

    /**
     * Add required attributes if necessary.
     *
     * @return void
     */
    XhtmlNode.prototype.cleanAttributesRequired = function() {
        if (XhtmlNode.TEXT == this.name) return;
        if ('undefined' == typeof this.dtd.elements[this.name]) return ;

        var required_attributes = this.dtd.elements[this.name]['required_attributes'];

        if (!required_attributes.length) return ;

        for (var i = 0; i < required_attributes.length; i++) {
            if (!this.hasAttribute(required_attributes[i]))
                this.addAttribute(required_attributes[i], this.getDefaultAttributeValue(required_attributes[i]));
        }
    };

    XhtmlNode.prototype.cleanAttributesEntities = function() {
        var attributes = [],
            map = { "&": "&amp;", "\"": "&quot;", "<": "&lt;", ">": "&gt;" };

        for (var i = 0; i < this.count_attributes; i++) {
            var name = this.attributes[i].name.toLowerCase();
            var value = this.attributes[i].value;

            value = value.replace(/[&"<>]/g, function(s) { return map[s]; });
            value = value.replace(/&amp;([a-z]+|(#\d+));/ig, "&$1;");

            attributes.push({ "name": name, "value": value });
        }
        this.setAttributes(attributes);
    };

    XhtmlNode.prototype.getDefaultAttributeValue = function(attribute) {
        switch (attribute) {
            case 'type':
                return ('style' == this.name ? 'text/css' : 'text/javascript');
                break;

            case 'col':
            case 'row':
                return '5';
                break;
        }
        //'action', 'alt', 'content', 'href', 'id', 'label', 'src':
        return '';
    };

    /**
     * Change node type to another.
     *
     * @param  string  name  New node name type.
     * @return void
     */
    XhtmlNode.prototype.changeTo = function(name) {
        this.name = name;
        this.is_empty = this.dtd.empty[name] || name == XhtmlNode.TEXT || false;
        this.is_inline = this.dtd.inline[name] || name == XhtmlNode.TEXT || false;
        this.is_block = ! this.is_inline;
    };

    XhtmlNode.prototype.toString = function() {
        var r = this.name +
                "\n\tparent : " + (this.getParent() ? this.getParent().name : '-') +
                "\n\tprevious : " + (this.getPrevious() ? this.getPrevious().name : '-') +
                "\n\tnext : " + (this.getNext() ? this.getNext().name : '-') +
                "\n\ttext : '" + (this.text ? this.text : '') + "'" +
                "\n\thasChildren : " + this.hasChildren();
        var child = this.first_child;
        var c = [];
        while (child) {
            c.push(child.name);
            child = child.getNext();
        }
        r += "[" + c.join(', ') + "]";

        return r + "\n";
    };

    XhtmlNode.prototype.toText = function(indent, level) {
        function start_tag(node) {
            var is_text = (XhtmlNode.TEXT == node.name);

            if (is_text && node.text) {
                if ((node.getPrevious() && node.getPrevious().isInline()) || (node.getParent() && node.getParent().isInline()) || (node.hasParent('pre'))) push_text(node.text);
                else {
                    push_new_line_text();
                    push_text(get_indent() + node.text);
                }
            }
            else if (!is_text) {
                if ((node.hasParent('pre') || (node.isInline() && 'option' != node.name && 'optgroup' != node.name)) && ((node.getPrevious() && node.getPrevious().isInline()) || (node.getParent() && node.getParent().isInline()))) push_text('<' + node.name)
                else {
                    push_new_line_text();
                    push_text(get_indent() + '<' + node.name);
                }
                for ( var i = 0; i < node.count_attributes; i++ )
                    push_text(' ' + node.attributes[i].name + '="' + node.attributes[i].value + '"');

                push_text((node.isEmpty() ? " /" : "") + ">");
                if ('br' == node.name && !node.hasParent('pre')) {
                    push_new_line_text();
                    push_text(get_indent());
                }
            }
        };

        function end_tag(node) {
            var is_text = (XhtmlNode.TEXT == node.name);

            if (!is_text && !node.isEmpty()) {
                if ((node.isInline() && 'select' == node.name) || (!node.isInline() && 'pre' != node.name && !node.hasParent('pre'))) {
                    push_new_line_text();
                    push_text(get_indent() + '</' + node.name + '>');
                }
                else {
                    push_text('</' + node.name + '>');
                }
            }
        };

        function get_indent() {
            var s_indent = '';
            for (var indent = 0; indent < (level * 2); indent++) s_indent += ' ';
            return s_indent;
        }

        function push_text(txt) {
            if ('' != txt)
                text.push(txt);
        }

        function push_new_line_text() {
            push_text('\n');
        }

        var level = 0;
        var text = [];
        var first_node = this;
        push_text(start_tag(this));
        level++;

        if (this.hasChildren()) {
            var node = this.getChildren();
            while (node && node != first_node) {
                start_tag(node);
                if (node.hasChildren()) {
                    node = node.getChildren();
                    level++;
                }
                else if (node.getNext()) {
                    end_tag(node);
                    node = node.getNext();
                }
                else {
                    while (node.getParent() && node != first_node) {
                        end_tag(node);
                        node = node.getParent();
                        level--;
                        if (node.getNext()) {
                            end_tag(node);
                            node = node.getNext();
                            break;
                        }
                    }
                }
            }
        }
        end_tag(this);
        return text.join('');
    };

    /* Cleaning status */
    XhtmlNode.CLEAN_IS_REMOVED              =   0;
    XhtmlNode.CLEAN_CONTINUE                =   1;
    XhtmlNode.CLEAN_REPASS                  =   2;
    XhtmlNode.CLEAN_IS_MOVED_BEFORE_PARENT  =   3;
    XhtmlNode.CLEAN_IS_MOVED_IN_PREVIOUS    =   4;

    XhtmlNode.CLEAN_NODES = [];

    /* specials nodes */
    XhtmlNode.TEXT          = '#text';
    XhtmlNode.COMMENT       = '#comment';

    /*
     * HtmlParser is an (x)html parser.
     *
     * Original code by John Resig (ejohn.org) http://ejohn.org/files/htmlparser.js
     */

    function HtmlParser(xhtml_tidy) {
        this.html = '';
        this.results = '';
        this.xhtml_tidy = xhtml_tidy;
        this.dtd = this.xhtml_tidy.dtd;
        this.node = new XhtmlNode('body', this.xhtml_tidy);
        this.current = this.node;
    };

    HtmlParser.prototype.parse = function(html) {
        this._parse(html);
    };

    HtmlParser.prototype._parse = function(html) {
        this.html = html;
        var self = this,

            start_tag = /^<([\w\:]+)((?:\s+[\w\:]+(?:\s*=\s*(?:(?:"[^"]*")|(?:'[^']*')|[^>\s]+))?)*)\s*(\/?)>/,
            end_tag = /^<\/([\w\:]+)[^>]*>/,
            attr = /([\w\:]+)(?:\s*=\s*(?:(?:"((?:\\.|[^"])*)")|(?:'((?:\\.|[^'])*)')|([^>\s]+)))?/g,

            empty = this.dtd.empty,
            block = this.dtd.block,
            inline = this.dtd.inline,
            close_self = XhtmlTidy.tools.makeMap(['colgroup', 'dd', 'dt', 'li', 'options', 'p', 'td', 'tfoot', 'th', 'thead', 'tr']),
            fill_attrs = this.dtd.attr_boolean,
            special = XhtmlTidy.tools.makeMap(['script', 'style']),

            index, chars, match, stack = [], last = this.html;

        stack.last = function() {
            return this[ this.length - 1 ];
        };

        while (this.html) {
            chars = true;

            // Make sure we're not in a script or style element
            if (!stack.last() || !special[stack.last()]) {

                // Comment
                if (this.html.indexOf("<!--") == 0) {
                    index = this.html.indexOf("-->");

                    if (index >= 0) {
                        this.commentTag(this.html.substring( 4, index ));
                        this.html = this.html.substring(index + 3);
                        chars = false;
                    }
                }
                // end tag
                else if (this.html.indexOf("</") == 0) {
                    match = this.html.match(end_tag);

                    if (match) {
                        this.html = this.html.substring(match[0].length);
                        match[0].replace(end_tag, parseEndTag);
                        chars = false;
                    }
                }
                // start tag
                else if (this.html.indexOf("<") == 0) {
                    match = this.html.match(start_tag);

                    if (match) {
                        this.html = this.html.substring(match[0].length);
                        match[0].replace(start_tag, parseStartTag);
                        chars = false;
                    }
                }

                if (chars) {
                    index = this.html.indexOf("<");

                    var text = index < 0 ? this.html : this.html.substring( 0, index );
                    this.html = index < 0 ? "" : this.html.substring( index );

                    this.text(text);
                }

            }
            else {
                // patch http://groups.google.com/group/envjs/browse_thread/thread/edd9033b9273fa58 - http://fu2k.org/alex/javascript/this.htmlparser/this.htmlparser.patch.20081012.js
                this.html = this.html.replace(new RegExp("([\\s\\S]*)<\\/" + stack.last() + "[^>]*>"), function(all, text){
                    text = text.replace(/<!--([\s\S]*?)-->/g, "$1").replace(/<!\[CDATA\[([\s\S]*?)\]\]>/g, "$1");
                    self.text( text );
                    return "";
                });

                parseEndTag("", stack.last());
            }

            if ( this.html == last )
                throw "Parse Error: " + this.html;
            last = this.html;
        }

        // Clean up any remaining tags
        parseEndTag();

        function parseStartTag(tag, tag_name, rest, unary ) {
            if (block[tag_name] ) {
                while (stack.last() && inline[stack.last()]) {
                    parseEndTag("", stack.last());
                }
            }

            if (close_self[tag_name] && stack.last() == tag_name) {
                parseEndTag("", tag_name);
            }

            unary = empty[tag_name] || !!unary;

            if (!unary)
                stack.push( tag_name );

                var attrs = [];

                rest.replace(attr, function(match, name) {
                    var value = arguments[2] ? arguments[2] :
                                arguments[3] ? arguments[3] :
                                arguments[4] ? arguments[4] :
                                fill_attrs[name] ? name : "";

                    attrs.push({
                        name: name,
                        value: value,
                        escaped: value.replace(/(^|[^\\])"/g, '$1\\\"') //"
                    });
                });

                self.startTag(tag_name, attrs, unary);
        };

        function parseEndTag(tag, tag_name) {
            // If no tag name is provided, clean shop
            if (!tag_name)
                var pos = 0;

            // Find the closest opened tag of the same type
            else
                for (var pos = stack.length - 1; pos >= 0; pos--)
                    if (stack[pos] == tag_name)
                        break;

            if (pos >= 0) {
                // Close all the open elements, up the stack
                for (var i = stack.length - 1; i >= pos; i--)
                        self.endTag(stack[i]);

                // Remove the open elements from the stack
                stack.length = pos;
            }
        };
    };

    HtmlParser.prototype.startTag = function(tag, attrs, unary) {
        tag = tag.toLowerCase();
        this.results += "<" + tag;

        for ( var i = 0; i < attrs.length; i++ )
            this.results += " " + attrs[i].name + '="' + attrs[i].escaped + '"';

        this.results += (unary ? " /" : "") + ">";

        var n = new XhtmlNode(tag, attrs, this.xhtml_tidy);
        this.current.addChild(n);

        if (!unary) this.current = n;
    };

    HtmlParser.prototype.endTag = function(tag) {
        tag = tag.toLowerCase();
        this.results += "</" + tag + ">";

        this.current = this.current.getParent();
    };

    HtmlParser.prototype.text = function(text) {
        this.results += text;

        var n = new XhtmlNode(XhtmlNode.TEXT, this.xhtml_tidy);
        this.current.addChild(n);
        n.setText(text);
    };

    HtmlParser.prototype.commentTag = function(text) {
        this.results += "<!--" + text + "-->";
        var n = new XhtmlNode(XhtmlNode.COMMENT, this.xhtml_tidy);
        n.setText(text);
        this.current.addChild(n);
    };

    HtmlParser.prototype.getRootNode = function() {
        if (!this.node) return new XhtmlNode(XhtmlNode.TEXT, this.xhtml_tidy);
        if (1 == this.node.count_children && 'body' == this.node.children[0].name) return this.node.children[0];
        return this.node;
    };
}