/*
 * Copyright 2012 Sébastien Raud
 *
 * This file is part of beCms.
 *
 * beCms is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * beCms is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with beCms.  If not, see <http://www.gnu.org/licenses/>.
 */

// jQuery tools

jQuery.extend({
    /**
     * Creates a jQuery html element.
     *
     * @param  string  name        Element name (eg. b, p, div...).
     * @param  object  attributes  Element attributes (eg. { "class": "css_class", "disabled": "disabled" }).
     * @return jQuery
     */
    createElement: function(name, attributes) {
        var $elt = jQuery(document.createElement(name));
        if (!attributes) return $elt;

        for (var key in attributes) {
            switch (key) {
                case 'class':
                    $elt.addClass(attributes[key]);
                    break;
                case 'html':
                    $elt.html(attributes[key]);
                    break;
                case 'text':
                    $elt.text(attributes[key]);
                    break;
                default:
                    $elt.attr(key, attributes[key]);
                    break;
            }
        }
        return $elt;
    }
});

/**
 * Sets or returns a style value for an element.
 * Returned style is lowercased.
 *
 * eg. jQuery('<a>').style('border', 'solid 1px #000'); // sets the "border" style value to "solid 1px #000'.
 * eg. var style = $element.style('text-align'); // returns the "text-align" style value.
 *
 * @param  string  name   Style name.
 * @param  mixed   value  string / object. Optional. The style value.
 * @return mixed
 */
if (jQuery.fn.jquery >= "1.7.2") {
    jQuery.fn.style = function(name, value) {
        if ( arguments.length === 2 && value === undefined ) {
            return this;
        }

        return jQuery.access( this, function( elem, name, value ) {
            var s = jQuery.style( elem, name, value );
            return s ? s.toLowerCase() : null;
        }, name, value, true );
    };
}
else {
    jQuery.fn.style = function(name, value) {
        if ( arguments.length === 2 && value === undefined ) {
            return this;
        }

        return jQuery.access( this, name, value, true, function( elem, name, value ) {
            var s = jQuery.style( elem, name, value );
            return s ? s.toLowerCase() : null;
        });
    };
}

/**
 * Indicates if a jQuery html element has a specific style.
 *
 * eg. jQuery('<a style="border: solid 1px #000">').hasStyle('border'); // check if border is a style, returns true
 * eg. jQuery('<a style="border: solid 1px #000">').hasStyle('border', 'solid 1px #fff'); // check if border style is "solid 1px #fff", returns false
 * eg. jQuery('<a style="color: blue; text-decoration: underline">').hasStyle({ "color": "blue", "text-decoration": "underline" }); // returns true
 *
 * @param  mixed   name   string / object. Style definition or styles definitions.
 * @param  string  value  Optional. Style value.
 * @return boolean
 */
jQuery.fn.hasStyle = function(name, value) {
    // browser can change the intial style definition,
    // we pass by an other object to get the value
    if ('string' === typeof name)
        return value ? this.style(name) === jQuery.createElement('div', { "style": name + ':' + value}).style(name) : null !== this.style(name);

    if (jQuery.isArray(name)) {
        var e = this, $d = jQuery.createElement('div'), r = true;
        jQuery.each(name, function(i, n) {
            r = r && null !== e.style(n);
        });
        return r;
    }

    if (jQuery.isPlainObject(name)) {
        var e = this, $d = jQuery.createElement('div'), r = true;
        jQuery.each(name, function(n, v) {
            r = r && v ? e.style(n) === $d.style(n, v).style(n) : null !== e.style(n);
        });
        return r;
    }

    return false;
};

/**
 * Returns the node name for a jQuery html element.
 * Node name is lowercased.
 *
 * @return string
 */
jQuery.fn.nodeName = function() {
    return this && this[0] ? this[0].nodeName.toLowerCase() : '';
};

/**
 * Indicates if a jQuery html element correspond to a node name.
 *
 * eg. jQuery('<a>').isNode('a'); // returns true
 * eg. jQuery('<p>').isNode('div, p'); // search if node is a div or p, return true
 *
 * @param  string  name  Node name.
 * @return boolean
 */
jQuery.fn.isNode = function(name) {
    if (!name || !this[0]) return false;
    return -1 != jQuery.inArray(this[0].nodeName.toLowerCase(), name.toLowerCase().split(/\s?,\s?/));
};


/*
 * weED
 * Wysiwyg and Extensible Editor
 */
window.onerror = function() { return true; };
if (undefined === weEd) {
    var weEd = function(options, $element) {
        this.init(options, $element);
    };

    // static attributes or methods
    weEd.initialBodyDimensions = { "width": jQuery('body').width(), "height": jQuery('body').height() };
    weEd.cssSupport = { "opacity": ('opacity' in document.body.style) };

    var _weEdp_ = weEd.prototype;

    _weEdp_.init = function(options, $element) {
        options = options || {};
        var self = this;

        this.options = jQuery.extend({}, {
            "buttons": "format-block,bold,italic,underline,strikethrough,subscript,superscript,nobreakspace,|,bullets,numbering,|,outdent,indent,|,alignleft,center,alignright,justify,|,undo,redo,|,rule,image,link,unlink,table,|,display-css,show-elements,source,fullscreen",
            "custom_buttons": {},

            "doctype":  "xhtml-1.0-strict",

            "css": [],
            "html_classes": null,
            "html_id": null,
            "body_classes": null,
            "body_id": null,
            "body_style": null,
            "wysiwym_css": null,

            "lang": "fr",
            "plugins": [],

            "height": "auto",
            "width":  "auto",

            "base_path":    null,
            "lang_path":    null,
            "images_path":  null,
            "plugins_path": null,

            "wysiwym_mode": false,

            "ready": null
        }, options);

        // check path
        // base path to load languages files and plugins
        this.options['base_path'] = this.options['base_path'] ||
                         jQuery('script').filter(function(index, elem) { return elem.src && /weed(.*)?\.js$/i.test(elem.src) }).attr('src').replace(/weed(.*)?\.js$/i, '') ||
                         '',
        this.options['lang_path'] = this.options['lang_path'] || this.options['base_path'] + 'lang/';
        this.options['images_path'] = this.options['images_path'] || this.options['base_path'] + 'images/';
        this.options['plugins_path'] = this.options['plugins_path'] || this.options['base_path'] + 'plugins/';
        if (!this.options['wysiwym_css']) this.options['wysiwym_css'] = this.options['base_path'] + 'weEd-wysiwym.css';

        // events bind
        this.events = {},
        // plugins
        this.plugins = {},

        this.is_init = false,

        // initial element
        this.$initial_element = $element,
        // textarea element
        this.$element = $element,

        // jQuery ui elements
        this.$container = null,
        this.$toolbar = null,
        this.$iframe = null,
        this.$statusbar = null,
        this.$statusbar_left = null,
        this.$statusbar_right = null,
        this.$doc = null,
        this.$head = null,
        this.$body = null,


        // states
        this.is_wysiwyg_mode = true,
        // is managing later
        this.is_wysiwym_mode = false,
        this.is_css_active   = true,
        this.is_fullscreen   = false,
        this.is_design_mode  = false,
        //this.is_enabled = true,

        // buttons
        this.buttons_click = {},
        this.toolbar_buttons = {},

        // current path in html code
        this.html_path = [],

        // bookmark for ie
        this.bookmark_range = null,

        // language
        this.lang = { "__plugins": {} },

        this.commands = {},
        this.shortcuts = {"ctrl": {}, "alt": {}, "shift": {}};
        //this.shortcuts = [];

        // defaults buttons properties
        this.buttons = {
            "format-block":     { "dialog":  self.installFormatBlockDialog, "type": "toggle" },
            "bold":             { "command": "bold", "shortcut": "Ctrl+B" },
            "italic":           { "command": "italic", "shortcut": "Ctrl+I" },
            "underline":        { "command": "underline", "shortcut": "Ctrl+U" },
            "strikethrough":    { "command": "strikethrough" },
            "subscript":        { "command": "subscript" },
            "superscript":      { "command": "superscript" },
            "nobreakspace":     { "command": "inserthtml", "command_value": "&nbsp;", "type": "no-toggle", "shortcut": "ctrl+ " },
            "bullets":          { "name":    "bullets", "command": "insertunorderedlist" },
            "numbering":        { "name":    "numbering", "command": "insertorderedlist" },
            "outdent":          { "command": "outdent", "type": "no-toggle" },
            "indent":           { "command": "indent", "type": "no-toggle" },
            "alignleft":        { "command": "justifyleft",   "select": function(state, $button) { align_select_fn(state, $button, 'alignleft'); }},
            "center":           { "command": "justifycenter", "select": function(state, $button) { align_select_fn(state, $button, 'center'); }},
            "alignright":       { "command": "justifyright",  "select": function(state, $button) { align_select_fn(state, $button, 'alignright'); }},
            "justify":          { "command": "justifyfull",   "select": function(state, $button) { align_select_fn(state, $button, 'justify'); }},
            "undo":             { "command": "undo", "type": "no-toggle" },
            "redo":             { "command": "redo", "type": "no-toggle" },
            "rule":             { "command": "inserthorizontalrule", "type": "no-toggle" },
            "image":            { "dialog":  self.installImageDialog, "shortcut": "Ctrl+P" },
            "link":             { "dialog":  self.installLinkDialg, "shortcut": "Ctrl+L" },
            "unlink":           { "command": "unlink", "type": "no-toggle" },
            "table":            { "dialog":  self.installTableDialog, "shortcut": "Ctrl+T" },

            // display css
            "show-elements":    { "command": self.toggleWisiwym, "select": function(state, $button) {
                    if (!self.is_wysiwym_mode) $button.removeClass('weed-button-selected');
                    else if (!$button.hasClass('weed-button-selected')) $button.addClass('weed-button-selected');
                }
            },

            "display-css":      { "command": self.toggleCss, "select": function(state, $button) {
                    if (!self.is_css_active) $button.removeClass('weed-button-selected');
                    else if (!$button.hasClass('weed-button-selected')) $button.addClass('weed-button-selected');
                }
            },

            "source":           { "command": self.toggle, "enable": function() { return true; } },

            "fullscreen":       { "command": self.toggleFullscreen, "enable": function() { return true; }, "shortcut": "Alt+F" },

            "|":                {},
            "||":               {}
        };

        // function to manage align functions
        var align_select_fn = function(state, $button, type) {
            if (!state) $button.removeClass('weed-button-selected');
            else {
                var a = ['alignleft', 'center', 'alignright', 'justify'], pos = -1;
                if (-1 != (pos = jQuery.inArray(type, a))) a.splice(pos, 1);
                jQuery.each(a, function(index, name) { self.toolbar_buttons[name].selected = false; self.toolbar_buttons[name].removeClass('weed-button-selected'); });
                if (!$button.hasClass('weed-button-selected')) $button.addClass('weed-button-selected');
            }
        };

        this.loadLanguage();
        this.loadEvents();
        this.loadPlugins();
        this.initEditor();
    };

    _weEdp_.initEditor = function() {
        var self = this;

        if (!this.$element.isNode('textarea')) {
            this.$element = jQuery.createElement('textarea', { "text": this.$initial_element.html() });
            this.$initial_element.after(this.$element).hide();
        }

        this.$container = jQuery.createElement('div', { "class": "weed-container" });
        if (jQuery.browser.msie && jQuery.browser.version < 9) this.$container.addClass('weed-ie-old-support');

        this.$element.before(this.$container);

        this.$toolbar = jQuery.createElement('div', { "class": "weed-toolbar" });
        this.$container.append(this.$toolbar);
        var $toolbar_group = jQuery.createElement('ul', { "class": "weed-toolbar-group" });
        this.$toolbar.append($toolbar_group, jQuery.createElement('div', { "class": "weed-clear" }));

        // make buttons
        this.installButtons($toolbar_group);

        // iframe
        this.$iframe = jQuery.createElement('iframe', { "frameborder": 0, "src": this.options.base_path + "weEd-" + this.options.doctype + ".html", "class": "weed-frame" })
                        .bind('load', function() { self.initIFrame.apply(self); });

        this.$container.append(this.$iframe);
        this.$element.hide();

        // statusbar
        this.$statusbar = jQuery.createElement('div', { "class": "weed-statusbar" });
        this.$statusbar_left = jQuery.createElement('span', { "class": "weed-statusbar-left", "text": this._('Loading...') });
        this.$statusbar_right = jQuery.createElement('span', { "class": "weed-statusbar-right" });
        this.$statusbar.append(this.$statusbar_left, this.$statusbar_right);
        this.$container.append(this.$statusbar);
    };

    _weEdp_.initIFrame = function() {
        //if (this.is_init) return;

        this.$doc = this.$iframe[0].contentWindow ? jQuery(this.$iframe[0].contentWindow.document) : jQuery(this.$iframe[0].contentDocument);

        var self = this;
        setTimeout(function() {
            self.enableDesignMode();

            setTimeout(function() {
                self.setHeaderBody();

                if (jQuery.browser.msie && !self.is_init) {
                    self.is_init = true;
                    self.updateIFrame('<p>' + self._('Loading...') + '</p>');
                    return;
                }

                self.$iframe.unbind('load.weed');

                if (self.options.html_classes) {
                    if ('string' === typeof self.options.html_classes && '' !== self.options.html_classes) self.$body.parent('html').addClass(self.options.html_classes);
                    else if (jQuery.isArray(self.options.html_classes) && self.options.body_classes.length) self.$body.parent('html').addClass(self.options.html_classes.join(' '));
                }
                if (self.options.html_id) self.$body.parent('html').attr('id', self.options.html_id);

                if (self.options.body_classes) {
                    if ('string' === typeof self.options.body_classes && '' !== self.options.body_classes) self.$body.addClass(self.options.body_classes);
                    else if (jQuery.isArray(self.options.body_classes) && self.options.body_classes.length) self.$body.addClass(self.options.body_classes.join(' '));
                }
                if (self.options.body_id) self.$body.attr('id', self.options.body_id);
                if (self.options.body_style) self.$body.attr('style', self.options.body_style);

                self.updateIFrame(self.$element.val());

                self.is_init = true;

                self.$element.parents('form:first').bind('submit.weed', function(e) { self.submit(e); });
                self.preventLinkClick();
                self.preserveAbsoluteUrls();
                self.$iframe.before(self.$element);

                self.insertCss(self.options['wysiwym_css'], 'none');

                // set css
                if (self.options.css) {
                    var css = self.options.css;
                    self.options.css = [];
                    var l = css.length;
                    for (var i = 0; i < l; i++) self.addCss(css[i], true);
                    self.selectButton('display-css', true);
                }

                if (self.options.wysiwym_mode) {
                    self.toggleWisiwym();
                }

                // bind doc after designMode for ie
                self.$doc
                    .bind('click.doc mouseup.doc', function(e) { self.saveRange(e); self.showHtmlElementsPath(); })
                    .bind('keypress.doc keydown.doc keyup.doc', function(e) {
                        self.saveRange(e);
                        if (e.keyCode >= 33 && e.keyCode <= 40) self.showHtmlElementsPath();
                    })
                    .bind('keydown.doc-shortcuts', function(e) {
                        var key = e.which,
                            $button = null;

                        if (e.ctrlKey && self.shortcuts['ctrl'][key]) $button = self.shortcuts['ctrl'][key];
                        else if (e.altKey && self.shortcuts['alt'][key]) $button = self.shortcuts['alt'][key];
                        else if (e.shiftKey && self.shortcuts['shift'][key]) $button = self.shortcuts['shift'][key];

                        if ($button) {
                            $button.trigger('click');
                            e.preventDefault();
                        }
                    })
                    .bind('focus.doc', function() { self.restoreRange(); });

                self.resize();
                self.initPlugins();

                self.statusbarLeftMessage('');

                self.focusToNode(self.$body[0]); // required for chrome
                self.focusToFirstNode();
                self._ready();
            }, 0);
        }, 0);
    };

    // ui management : buttons and dialog

    _weEdp_.buttonClick = function($button, e) {
        if (!$button.enabled) return ;
        var r = null; // return value
        $button.weed.focus();

        switch ($button.type) {
            case 'none':
                if ($button.dialog) r = $button.dialog.show();
                else {
                    if ($button.command && 'string' === typeof $button.command) r = this.execCommand($button.command, $button.command_value);
                    else if (jQuery.isFunction($button.command)) r = $button.command.apply(this, [$button, $button.command_value]);
                    $button.select(!$button.selected);
                }
                break;

            case 'toggle':
                if (!$button.selected) {
                    if ($button.dialog) r = $button.dialog.show();
                    else {
                        if ($button.command && 'string' === typeof $button.command) r = this.execCommand($button.command, $button.command_value);
                        else if (jQuery.isFunction($button.command)) r = $button.command.apply(this, [$button, $button.command_value]);
                        $button.select(!$button.selected);
                    }
                }
                else if ($button.dialog) $button.dialog.hide();
                break;

            case 'no-toggle':
                $button.select(true);
                if ($button.dialog) r = $button.dialog.show();
                else {
                    if ($button.command && 'string' === typeof $button.command) r = this.execCommand($button.command, $button.command_value);
                    else if (jQuery.isFunction($button.command)) r = $button.command.apply(this, [$button, $button.command_value]);
                }
                setTimeout(function() { $button.select(false); }, 250);
                break;
        }

        $button.weed.focus();
        if (!$button.dialog) $button.weed.showHtmlElementsPath();

        return (undefined == r ? $button : r);
    };

    _weEdp_.installButtons = function($toolbar_group) {
        var self = this;
        var buttons = this.options.buttons.split(',');
        var ln = buttons.length;

        var buttons_properties = jQuery.extend(true, self.buttons, self.options.custom_buttons);

        for (var i = 0; i < ln; i++) {
            var key = buttons[i], short_specials = null, short_key = null;
            if (buttons_properties[key]) {
                var $button = this.getButton(key, buttons_properties[key]);
                $toolbar_group.append($button);
                if (buttons_properties[key]['shortcut']) {
                    short_specials = /(ctrl\+)|(alt\+)|(shift\+)/ig.exec(buttons_properties[key]['shortcut']);
                    if (short_specials) {
                        short_specials = short_specials[0].replace('+', '').toLowerCase();
                        short_key = buttons_properties[key]['shortcut'].replace(/(ctrl\+)|(alt\+)|(shift\+)/ig, '').toUpperCase().charCodeAt(0);
                        this.shortcuts[short_specials][short_key] = $button;
                    }
                }
                if ('|' !== key && '||' !== key) this.toolbar_buttons[key] = $button;
            }
        }
    };

    _weEdp_.getButton = function(key, properties) {
        var self = this;

        if ('|' === key) return jQuery.createElement('li', { "class": "weed-button weed-separator" });
        if ('||' === key) return jQuery.createElement('li', { "class": "weed-button weed-newline" });

        var default_properties = { "name": key, "title": this._(key) + (properties['shortcut'] ? ' ' + properties['shortcut'] : ''), "command": null, "command-value": null, "dialog": null, "$dialog": null, "image": null, "css-style": null, "init": false, "enabled": true, "enable": null, "selected": false, "select": null, "shortcut": null, "html_path_update": null, "weed": self, "type": "none" };
        properties = !properties ? default_properties : jQuery.extend(default_properties, properties);

        var $button =   jQuery.extend(
                            jQuery.createElement('li', { "class": "weed-button button-" + key, "title": properties['title'] || self._(key) + (properties['shortcut'] ? ' ' + properties['shortcut'] : '') }),
                            properties
                        );

        // fix enable and select functions
        if ($button.enable && jQuery.isFunction($button.enable)) {
            $button._enable = $button.enable;
            $button.enable = function(state) {
                if (undefined === state) return $button.enabled;
                var r = $button._enable.apply($button, [state, $button]);
                $button.enabled = (r && 'boolean' === typeof r ? r : state);
                return (undefined == r ? $button : r);
            };
        }
        else {
            // state : true / false, enabled / disabled
            // if no state, returns the current enabled status
            $button.enable = function(state) {
                if (undefined === state) return $button.enabled;
                if (state) $button.removeClass('weed-button-disabled');
                else if (!$button.hasClass('weed-button-disabled')) $button.addClass('weed-button-disabled');
                $button.enabled = state;
            };
        }

        if ($button.select && jQuery.isFunction($button.select)) {
            $button._select = $button.select;
            $button.select = function(state) {
                if (undefined === state) return $button.selected;
                var r = $button._select.apply($button, [state, $button]);
                $button.selected = (r && 'boolean' === typeof r ? r : state);
                return (undefined == r ? $button : r);
            };
        }
        else {
            $button.select = function(state) {
                if (undefined === state) return $button.selected;
                if (!state) $button.removeClass('weed-button-selected');
                else if (!$button.hasClass('weed-button-selected')) $button.addClass('weed-button-selected');
                $button.selected = state;
                return $button;
            };
        }

        $button.bind('click', function(e) {
            self.buttonClick($button, e);
        });

        if (false === $button.enabled) $button.enable(false);
        if ($button.selected) $button.select(true);

        if ($button.init && jQuery.isFunction($button.init)) {
            $button.init($button);
        }

        if ($button.image && '' != $button.image) {
            $button.css("background-image", 'url(' + $button.image + ')');
        }

        if ($button['css-style']) {
            $button.style($button['css-style']);
        }

        if ($button.dialog) $button.dialog = this.getDialogInstance($button);

        return $button;
    };

    _weEdp_.selectButton = function(button, select) {
        button = undefined === button ? '*' : button;
        select = undefined === select || 'boolean' !== typeof select ? true : select;

        var buttons = { "b": "bold", "strong": "bold", "i": "italic", "em": "italic", "u": "underline", "strike": "strikethrough", "s": "strikethrough",
                        "sub": "subscript", "sup": "superscript", "ul": "bullets", "ol": "numbering", /**/ "hr": "rule", "img": "image", "a": "link",
                        "alignleft": "alignleft", "center": "center", "alignright": "alignright", "justify": "justify", "table": "table" };

        if ('*' === button) button = buttons;

        if (jQuery.isArray(button) || jQuery.isPlainObject(button)) {
            var self = this;
            jQuery.each(button, function(index, name) {
                if (buttons[name]) name = buttons[name];
                select_button.apply(self, [name]);
            });
        }
        else {
            name = buttons[button] ? buttons[button] : button;
            select_button.apply(this, [name]);
        }

        function select_button(button) {
            if (this.toolbar_buttons[button]) {
                this.toolbar_buttons[button].select(this.toolbar_buttons[button].dialog && this.toolbar_buttons[button].dialog.is_open ? true : select);
            }
        };
    };

    _weEdp_.enableButton = function(button, enable) {
        this.trigger('enableButton', { "button": button, "enable": enable });
    };

    _weEdp_._enableButton = function(data) {
        data.button = undefined === data.button ? '*' : data.button;
        data.enable = undefined === data.enable || 'boolean' !== typeof data.enable ? true : data.enable;

        if ('*' === data.button) {
            for (var i in this.toolbar_buttons)
                this.toolbar_buttons[i].enable(data.enable);
        }
        else if (this.toolbar_buttons[data.button])
            this.toolbar_buttons[data.button].enable(data.enable);
    };

    _weEdp_.resize = function(width, height) {
        this.trigger('resize', { "width": width, "height": height });
    };

    _weEdp_._resize = function(data) {

        if (!this.$container.is(':visible')) return ;

        var self = this;

        if (!data.width) data.width = this.options.width;
        if (!data.height) data.height = this.options.height;

        if ('auto' === data.width) {
            this.$container.style({ "width": 1, "height": 1 });
            var $parent = this.$container.parent(':first');

            if (!$parent.isNode('body')) data.width = $parent.width();
            else data.width = weEd.initialBodyDimensions.width - 50;
        }

        if ('auto' === data.height) data.height = '250px';

        // bug ie7+, opera 9.x...
        jQuery.each([this.$iframe, this.$element], function(index, $elt) {
            $elt.style({ "width": 1, "height": 1 });
        });

        this.$container.style({"width": data.width, "height": data.height});

        jQuery.each([this.$iframe, this.$element], function(index, $elt) {
            var h = (parseInt(self.$container.innerHeight()) - parseInt(self.$toolbar.outerHeight()) - 25 - /*parseInt(self.$statusbar.outerHeight()) */ (parseInt($elt.outerHeight()) - parseInt($elt.innerHeight())));
            var w = (parseInt(self.$container.innerWidth()) - (parseInt($elt.outerWidth()) - parseInt($elt.innerWidth())));
            $elt.style({ "width": w, "height": h });
        });
    };

    _weEdp_.toggleFullscreen = function() {
        var $w = jQuery(window);
        var self = this;

        if (this.is_fullscreen) {
            $w.unbind('resize.weed-fullscreen')
            this.$container.style({ "left": "", "position": "", "top": "", "z-index": "" });
            this.resize();
        }
        else {
            this.$container.style({ "left": 0, "position": "fixed", "top": 0, "z-index": 99 });
            this.resize($w.width(), $w.height());
            $w.unbind('resize.weed-fullscreen')
              .bind('resize.weed-fullscreen', function() { self.resize($w.width(), $w.height()); });
        }
        this.is_fullscreen = !this.is_fullscreen;
    };

    _weEdp_.getDialogInstance = function($element) {
        var dialog = function($element) {
            this.$element = $element;
            this.weed = $element.weed;
            this.fn = $element.dialog;

            this.$tabs = [];
            this.$tab_container = null;
            this.$body = [];
            this.$body_container = null;

            this.$elements = null;
            this.buttons = {};
            this.hide_auto_clear = false;

            this.update_function = null;
            this.is_updated = true;
            this.display_tab = 0;
            this.is_open = false;

            this.errors = [];

            var self = this;

            var methods = {
                init: function() {
                    self.$tabs = [];
                    self.$tab_container = null;
                    self.$body = [];
                    self.$body_container = null;

                    self.$elements = null;
                    self.hide_auto_clear = true;
                    return self;
                },
                addTab: function(label, $body_tab) {
                    var $span_tab = jQuery.createElement('span');

                    $span_tab.data('tab-data', {
                        "label": null,
                        "label_short": null
                    });

                    $span_tab
                        .bind('mouseover.dialog', function() { jQuery(this).text(jQuery(this).data('tab-data').label); })
                        .bind('mouseout.dialog', function() { if (!jQuery(this).hasClass('weed-dialog-tab-selected')) jQuery(this).text(jQuery(this).data('tab-data').label_short); })
                        .bind('click.dialog', function() {
                            for (var i in self.$tabs) {
                                self.$tabs[i].removeClass('weed-dialog-tab-selected').mouseout();
                            }
                            jQuery(this).addClass('weed-dialog-tab-selected');
                            for (var i in self.$body) self.$body[i].hide();
                            if ($body_tab) $body_tab.show();
                            jQuery(this).text(jQuery(this).data('tab-data').label);
                        });

                    self.setTabLabel(label, $span_tab);
                    self.$tabs.push($span_tab);

                    if ($body_tab) {
                        if (!$body_tab.hasClass('weed-dialog-tab')) $body_tab.addClass('weed-dialog-tab');
                        self.$body.push($body_tab.hide());
                    }
                    return self;
                },
                setTabLabel: function(label, tab) {
                    var $tab = (isNaN(parseInt(tab)) ? tab : (tab >= 0 && tab < self.$tabs.length ? self.$tabs[tab] : null));
                    if (!$tab) return ;

                    $tab.data('tab-data').label = label;
                    $tab.data('tab-data').label_short = (label.length > 8 ? label.substring(0, 8) + '...' : label);
                },
                setBody: function($body) {
                    if ('string' === typeof $body) $body = jQuery($body);
                    self.$body = [$body];
                    return self;
                },
                hide: function() {
                    self.$element.$dialog.hide(150);
                    self.$element.select(false);
                    self.is_updated = false;
                    if (self.hide_auto_clear) {
                        self.init();
                        self.$element.$dialog = null;
                    }
                    self.weed.hideDialog($element);
                    self.is_open = false;
                    return self;
                },
                show: function() {
                    self.fn(self.$element);
                    self.update();
                    if (self.display_tab >= 0 && self.display_tab < self.$tabs.length) {
                        for (var i in self.$body) self.$body[i].hide();
                        self.$tabs[self.display_tab].click();
                    }
                    self.$element.select(true);
                    self.$element.$dialog.show(150, function() {
                        jQuery(self.$element.$dialog.find('input:visible, select:visible, textarea:visible').get(0)).focus();
                    });
                    self.weed.executeDialog($element);
                    self.is_open = true;
                    return self;
                },
                build: function() {
                    var $dialog = jQuery.createElement('div', { "class": "weed-dialog-box" });
                    if (self.$tabs.length) {
                        self.$tab_container = jQuery.createElement('p').addClass('weed-dialog-tabs');
                        for (var i in self.$tabs) self.$tab_container.append(self.$tabs[i]);
                        $dialog.append(self.$tab_container);
                    }

                    self.$body_container = jQuery.createElement('div').addClass('weed-dialog-body');
                    for (var i in self.$body) self.$body_container.append(self.$body[i]);
                    $dialog.append(self.$body_container);

                    if (self.$elements) self.$body_container.append(self.$elements);
                    self.is_updated = false;
                    self.$element.$dialog = $dialog;
                    return self;
                },
                update: function(callback) {
                    if (callback && jQuery.isFunction(callback)) self.update_function = callback;
                    else if (self.update_function && !self.is_updated) {
                        self.update_function(self.$element);
                        self.is_updated = true;
                    }
                    return self;
                },
                getDialog: function() {
                    return self.$element.$dialog;
                },
                displayErrors: function() {
                    if (self.errors.length) {
                        var $ul_errors = jQuery.createElement('ul').addClass('weed-dialog-error');
                        jQuery.each(self.errors, function(index, value) {
                            if ('object' === typeof(value)) {
                                $ul_errors.append(jQuery.createElement('li', { "text": value['error'] }));
                                value['element'].addClass('weed-dialog-error');
                            }
                            else
                                $ul_errors.append(jQuery.createElement('li', { "text": value }));
                        });
                        if (!self.$body_container) self.$body_container = jQuery.createElement('div', { "class": "weed-dialog-body" });
                        self.$body_container.prepend($ul_errors);
                    }
                },
                clearErrors: function() {
                    if (self.$body_container) self.$body_container.children('ul.weed-dialog-error').remove();
                    jQuery.each(self.errors, function(index, value) {
                            if ('object' === typeof(value)) {
                                value['element'].removeClass('weed-dialog-error')
                            }
                    });
                    self.errors = [];
                },
                addButtons: function(buttons) {
                    var $p_buttons = jQuery.createElement('p').addClass('weed-dialog-buttons');
                    if (buttons['ok']) {
                        var $ok = jQuery.createElement('input').attr('type', 'button').addClass('weed-dialog-button-ok').val(self.weed._('ok'));
                        if (buttons['ok']['callback']) {
                            $ok.bind('click.ok', function(e) {
                                self.clearErrors();
                                buttons['ok']['callback'].call(self, self.$element, self.$element.data('weed-button-dialog'), e);
                            });
                        }
                        self.buttons['ok'] = $ok;
                        $p_buttons.append($ok);
                    }
                    if (buttons['cancel']) {
                        var $cancel = jQuery.createElement('input').attr('type', 'button').addClass('weed-dialog-button-cancel').val(self.weed._('cancel'));
                        if (buttons['cancel']['callback']) {
                            $cancel.bind('click.cancel', function(e) {
                                self.clearErrors();
                                buttons['cancel']['callback'].call(self, self.$element);
                            });
                        }
                        else {
                            $cancel.bind('click.cancel', function(e) {
                                self.hide();
                            });
                        }
                        self.buttons['cancel'] = $cancel;
                        $p_buttons.append(((buttons['ok']) ? ' ' : ''), $cancel);
                    }
                    self.$elements = $p_buttons;
                    return self;
                }
            };
            var namespace = dialog.prototype;
            jQuery.each(methods, function(name, fct) {
                namespace[name] = fct;
            });

            self.init();
        };

        if (!$element.weed) $element.weed = this;
        if (!$element.$dialog) $element.$dialog = null;
        if (!$element.select) $element.select = function(select) { return (undefined == select) || $element;  };


        return new dialog($element);
    };

    _weEdp_.executeDialog = function($button) {
        this.trigger('executeDialog', { "$button": $button });
    };

    _weEdp_._executeDialog = function(data) {
        this.displayOverlay(data.$button);
        this.positionDialog(data.$button);
        this.$container.append(data.$button.$dialog);
    };

    _weEdp_.hideDialog = function($button) {
        this.trigger('hideDialog', { "$button": $button });
    };

    _weEdp_._hideDialog = function(data) {
        this.hideOverlay();
        this.focus();
        this.selectButton(this.getHtmlElementsPath(), true);
    },

    _weEdp_.positionDialog = function($button) {
        this.trigger('positionDialog', { "$button": $button });
    };

    _weEdp_._positionDialog = function(data) {
        $dialog = data.$button.$dialog;
        var offset, left, top;
        if (data.$button) {
          offset = data.$button.offset();
          left = --offset.left;
          top = offset.top + data.$button.height();
        }
        else {
          offset = $toolbar.offset();
          left = Math.floor((this.$toolbar.width() - $dialog.width()) / 2) + offset.left;
          top = offset.top + this.$toolbar.height() - 2;
        }

        $dialog.css({left: left, top: top, position: 'absolute', 'z-index': 1000});
    },

    _weEdp_.displayOverlay = function($button) {
        var self = this,
            position = this.$container.offset(),
            dimensions = { "width": this.$container.outerWidth(), "height": this.$container.outerHeight(), "top": position.top, "left": position.left },
            destroy;

        var $overlay = this.$container.find('div.overlay');
        if (!$overlay.length) $overlay = jQuery.createElement('div', { "class": "weed-overlay" });

        destroy = function(e) {
            if ($button && $button.dialog) {
                $button.select(false);
                $button.dialog.hide();
                self.hideDialog($button);
            }
            else
                self.hideOverlay();
            self.$doc.unbind('keyup.overlay');
        };
        $overlay.style(dimensions)
            .bind('click.overlay', function(e) {
                destroy(e);
            });
        this.$doc.bind('keyup.overlay', function(e) {
            if (27 === e.keyCode) destroy(e);
        });
        $button.dialog.$element.$dialog.bind('keyup.overlay', function(e) {
            if (27 === e.keyCode) destroy(e);
        });
        this.$container.append($overlay);
        return $overlay;
    },

    _weEdp_.hideOverlay = function() {
        this.$container.children('div.weed-overlay').remove();
    },

    _weEdp_.installFormatBlockDialog = function($button) {
        if (!$button.$dialog) {
            var $selected_node = jQuery($button.weed.getSelectedNode());
            var $ul = jQuery.createElement('ul').addClass('weed-selector-list');
            var $li_list = [];

            jQuery.each({ "clear" : "clear", "div": "div", "p": "paragraph", "h1": "heading1", "h2": "heading2", "h3": "heading3", "h4": "heading4", "h5": "heading5", "h6": "heading6", "pre": "formatted", "blockquote": "blockquote", "address": "address" }, function(key, name) {
                var $li = jQuery.createElement('li').text($button.weed._(name))
                            .bind('click', function(e) {
                                var $this = jQuery(this);
                                if ('clear' != key) {
                                    if ($this.hasClass('selected')) jQuery(this).removeClass('selected');
                                    else $this.addClass('selected');
                                }
                                $button.weed.execCommand('FormatBlock', '<' + key + '>');
                                $button.dialog.hide();
                            })
                            .bind('mouseover', function(e) {
                                if ('clear' != key)
                                    $button.weed.statusbarRightMessage('&lt;' + key + '&gt; ... &lt;/' + key + '&gt;');
                            })
                            .bind('mouseout', function(e) {
                                $button.weed.statusbarRightMessage('');
                            });
                if ($selected_node[0] && $selected_node.isNode(key) || $selected_node.parents(key).length) $li.addClass('selected');
                $ul.append($li);
                $li_list.push($li);

                if ('div' === key)
                    $ul.append(jQuery.createElement('li', { "style": "border-top: solid 1px #999; margin: 0; padding: 0 5px;" }));
            });

            $button.dialog.setBody($ul);
            $button.dialog.update(function($button) {
                var $selected_node = jQuery($button.weed.getSelectedNode());
                    jQuery.each(['div', 'p', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'pre', 'blockquote', 'address'], function(index, key) {
                        if ($selected_node[0] && $selected_node.isNode(key) || $selected_node.parents(key).length) $li_list[index + 1].addClass('selected');
                        else $li_list[index + 1].removeClass('selected');
                    })
            });

            $button.dialog.build();
            $button.dialog.is_updated = true;
            $button.dialog.hide_auto_clear = false;
        }
    };

    _weEdp_.installImageDialog = function($button) {
        if (!$button.$dialog) {
            var $input_list = [];
            var $div = jQuery.createElement('div');
            var $p = jQuery.createElement('p');

            var $label = jQuery.createElement('label', {"for": "weed-image-url", "text": $button.weed._('url') + ' : '});
            var $input = jQuery.createElement('input', {"type": "text", "id": "weed-image-url", "class": "weed-dialog-input-long"});
            $p.append($label, $input);
            $div.append($p);
            $input_list.push($input);

            $p = jQuery.createElement('p');
            $label = jQuery.createElement('label', {"for": "weed-image-alt", "text": $button.weed._('alt') + ' : '});
            $input = jQuery.createElement('input', {"type": "text", "id": "weed-image-alt", "class": "weed-dialog-input-long"});
            $p.append($label, $input);
            $div.append($p);
            $input_list.push($input);

            $p = jQuery.createElement('p');
            $label = jQuery.createElement('label', {"for": "weed-image-title", "text": $button.weed._('title') + ' : '});
            $input = jQuery.createElement('input', {"type": "text", "id": "weed-image-title", "class": "weed-dialog-input-long"});
            $p.append($label, $input);
            $div.append($p);
            $input_list.push($input);

            $p = jQuery.createElement('p');
            $label = jQuery.createElement('label', {"for": "weed-image-width", "text": $button.weed._('width') + ' : '});
            $input = jQuery.createElement('input', {"type": "text", "id": "weed-image-width", "class": "weed-dialog-input-small"});
            $p.append($label, $input);
            $div.append($p);
            $input_list.push($input);

            $p = jQuery.createElement('p');
            $label = jQuery.createElement('label', {"for": "weed-image-height", "text": $button.weed._('height') + ' : '});
            $input = jQuery.createElement('input', {"type": "text", "id": "weed-image-height", "class": "weed-dialog-input-small"});
            $p.append($label, $input);
            $div.append($p);
            $input_list.push($input);

            $button.dialog.setBody($div);
            $button.dialog.addButtons({"ok": {"callback": function($button) {
                var src   = jQuery.trim($input_list[0].val());
                var alt   = jQuery.trim($button.weed.protectData($input_list[1].val()));
                var title = jQuery.trim($button.weed.protectData($input_list[2].val()));
                var w = jQuery.trim($input_list[3].val());
                var h = jQuery.trim($input_list[4].val());
                if ('' != src && 'http://' != src) {
                    var temp_id = 'http://abc/' + $button.weed.getTempId() + '.png'; // to prevent call to server
                    $button.weed.execCommand('insertimage', temp_id);
                    var $img = $button.weed.$doc.find('body img[src$="' + temp_id + '"]');
                    $img.attr('src', src);
                    if ('' != alt) $img.attr('alt', alt);
                    if ('' != title) $img.attr('title', title);
                    if (/^\d+$/.test(w)) $img.attr('width', w);
                    if (/^\d+$/.test(h)) $img.attr('height', h);
                    $button.dialog.hide();
                }
                else {
                    $button.dialog.errors.push({ "error": $button.weed._('url_required'), "element": $input_list[0].parent() });
                    $button.dialog.displayErrors();
                }
            }}, "cancel": {}});
            $button.dialog.update(function($button) {
                var $node = jQuery($button.weed.getSelectedNode());
                $button.dialog.clearErrors();
                if (!$node[0] || !$node.isNode('img')) {
                    $input_list[0].val('http://');
                    for (var i = 4; i > 0; i--) $input_list[i].val('');
                }
                else {
                    $input_list[0].val($node.attr('src'));
                    $input_list[1].val($node.attr('alt') || '');
                    $input_list[2].val($node.attr('title') || '');
                    $input_list[3].val($node.attr('width') || '');
                    $input_list[4].val($node.attr('height') || '');
                }
            });
            $button.dialog.build();
            $button.dialog.hide_auto_clear = false;
        }
    };

    _weEdp_.installLinkDialg = function($button) {
        if (!$button.$dialog) {
            var $input_list = [];
            var $div = jQuery.createElement('div');
            var $p = jQuery.createElement('p');

            var $label = jQuery.createElement('label', {"for": "weed-link-url", "text": $button.weed._('url') + ' : '});
            var $input = jQuery.createElement('input', {"type": "text", "id": "weed-link-url", "class": "weed-dialog-input-long"});
            $p.append($label, $input);
            $div.append($p);
            $input_list.push($input);

            $p = jQuery.createElement('p');
            $label = jQuery.createElement('label', {"for": "weed-link-title", "text": $button.weed._('title') + ' : '});
            $input = jQuery.createElement('input', {"type": "text", "id": "weed-link-title", "class": "weed-dialog-input-long"});
            $p.append($label, $input);
            $div.append($p);
            $input_list.push($input);

            $button.dialog.setBody($div);
            $button.dialog.addButtons({"ok": {"callback": function($button) {
                var href = jQuery.trim($input_list[0].val());
                var title = jQuery.trim($button.weed.protectData($input_list[1].val()));
                if ('' != href && 'http://' != href) {
                    var temp_id = 'http://abc' + $button.weed.getTempId() + '.html'; // to prevent call to server
                    $button.weed.execCommand('createlink', temp_id);
                    var $a = $button.weed.$doc.find('body a[href$="' + temp_id + '"]');
                    $a.attr('href', href);
                    if ('' != title) $a.attr('title', title);
                    $button.weed.preventLinkClick();
                    $button.dialog.hide();
                }
                else {
                    $button.dialog.errors.push({ "error": $button.weed._('url_required'), "element": $input_list[0].parent() });
                    $button.dialog.displayErrors();
                }
            }}, "cancel": {}});
            $button.dialog.update(function($button) {
                var $node = jQuery($button.weed.getSelectedNode());
                if ($node.isNode('img') && $node.parents('a:first')) $node = $node.parents('a:first');
                $button.dialog.clearErrors();

                if (('' === $button.weed.getSelectedText()) && (!$node || !$node.isNode('img, a'))) {
                    $button.dialog.errors.push("Aucun texte sélectionné.");
                    $button.dialog.displayErrors();
                    $input_list[0].val('http://').attr('disabled', 'disabled');
                    $input_list[1].val('').attr('disabled', 'disabled');
                    $button.dialog.buttons['ok'].hide();
                    return;
                }

                $input_list[0].removeAttr('disabled');
                $input_list[1].removeAttr('disabled');
                $button.dialog.buttons['ok'].show();

                if (!$node[0] || !$node.isNode('a')) {
                    $input_list[0].val('http://');
                    $input_list[1].val('');
                }
                else {
                    $input_list[0].val($node.attr('href'));
                    $input_list[1].val($node.attr('title') || '');
                }
            });
            $button.dialog.build();
            $button.dialog.hide_auto_clear = false;
        }
    };

    _weEdp_.installTableDialog = function($button) {
        if (!$button.$dialog) {
            var $input_list = [];
            var $div = jQuery.createElement('div');
            var $p = jQuery.createElement('p');

            var $label = jQuery.createElement('label', {"for": "weed-table-rows", "text": $button.weed._('number_of_rows') + ' : '});
            var $input = jQuery.createElement('input', {"type": "text", "id": "weed-table-rows", "class": "weed-dialog-input-small"});
            $p.append($label, $input);
            $div.append($p);
            $input_list.push($input);

            $p = jQuery.createElement('p');
            $label = jQuery.createElement('label', {"for": "weed-table-cols", "text": $button.weed._('number_of_cols') + ' : '});
            $input = jQuery.createElement('input', {"type": "text", "id": "weed-table-cols", "class": "weed-dialog-input-small"});
            $p.append($label, $input);
            $div.append($p);
            $input_list.push($input);

            $p = jQuery.createElement('p');
            $label = jQuery.createElement('label', {"for": "weed-table-headers", "text": $button.weed._('headers') + ' : '});
            $input = jQuery.createElement('select', {"id": "weed-table-headers", "class": "weed-dialog-input-small"})
                        .append(jQuery.createElement('option', {"value": "", "text": $button.weed._('none'), "selected": "selected"}))
                        .append(jQuery.createElement('option', {"value": "first-row", "text": $button.weed._('first-row')}))
                        .append(jQuery.createElement('option', {"value": "first-col", "text": $button.weed._('first-col')}))
                        .append(jQuery.createElement('option', {"value": "both", "text": $button.weed._('both')}))
            $p.append($label, $input);
            $div.append($p);
            $input_list.push($input);

            $p = jQuery.createElement('p');
            $label = jQuery.createElement('label', {"for": "weed-table-caption", "text": $button.weed._('caption') + ' : '});
            $input = jQuery.createElement('input', {"type": "text", "id": "weed-table-caption", "class": "weed-dialog-input-long"});
            $p.append($label, $input);
            $div.append($p);
            $input_list.push($input);

            $p = jQuery.createElement('p');
            $label = jQuery.createElement('label', {"for": "weed-table-summary", "text": $button.weed._('summary') + ' : '});
            $input = jQuery.createElement('textarea', {"id": "weed-table-summary", "class": "weed-dialog-input-long"});
            $p.append($label, $input);
            $div.append($p);
            $input_list.push($input);

            $button.dialog.setBody($div);
            $button.dialog.addButtons({"ok": {"callback": function($button) {
                var r = $input_list[0].val(), c = $input_list[1].val();
                if (/^\d+$/.test(r) && /^\d+$/.test(c)) {
                    var headers = $input_list[2].val();
                    var caption = jQuery.trim($button.weed.protectData($input_list[3].val()));
                    var summary = jQuery.trim($button.weed.protectData($input_list[4].val()));
                    var html = '<table' + (('' != summary) ? ' summary="' + summary + '"' : '') + ' class="weed-insert-new-table">';
                    if ('' != caption) html += '<caption>' + caption + '</caption>';
                    if ('first-row' === headers || 'both' === headers) {
                        html += '<thead>'
                        for (var ir = 0; ir < r && ir < 1; ir++) {
                            html += '<tr>';
                            for (var ic = 0; ic < c; ic++)  html += '<th' + ((!ic && 'both' === headers) ? ' scope="row"' : ((ic && 'both' === headers) ? ' scope="col"' : ' scope="col"')) + '></th>';
                            html += '</tr>';
                        }
                        html += '</thead>';
                    }
                    html += '<tbody>';
                    for (var ir = ('first-row' === headers || 'both' === headers) ? 1 : 0; ir < r; ir++) {
                        html += '<tr>';
                        for (var ic = 0; ic < c; ic++) {
                            if (!ic && ('first-col' === headers || 'both' === headers)) html += '<th scope="row"></th>';
                            else html += '<td></td>';
                        }
                        html += '</tr>';
                    }
                    html += '</tbody></table>';
                    $button.weed.execCommand('inserthtml', html);
                    var $table = $button.weed.$doc.find('body table.weed-insert-new-table').removeClass('weed-insert-new-table');
                    $button.dialog.hide();
                    if ('' === $table.attr('class')) $table.removeAttr('class');
                    var $cell = $table.find('th:first, td:first');
                    if ($cell && $cell[0]) $button.weed.focusToNode($cell[0]);
                }
                else {
                    if (! /^\d+$/.test(r)) $button.dialog.errors.push({ "error": self._('rows_integer'), "element": $input_list[0].parent() });
                    if (! /^\d+$/.test(c)) $button.dialog.errors.push({ "error": self._('cols_integer'), "element": $input_list[1].parent() });
                    $button.dialog.displayErrors();
                }
            }}, "cancel": {}});
            $button.dialog.update(function($button) {
                $button.dialog.clearErrors();
                for (var i = 4; i >= 0; i--) $input_list[i].val('');
            });
            $button.dialog.build();
            $button.dialog.hide_auto_clear = false;
        }
    };

    // end ui management : buttons and dialog


    // common editor methods

    _weEdp_.statusbarLeftMessage = function(message) {
        this.trigger('statusbarLeftMessage', { "message": message });
    };

    _weEdp_.statusbarRightMessage = function(message, duration) {
        this.trigger('statusbarRightMessage', { "message": message, "duration": duration || 2500 });
    };

    _weEdp_.getHtmlElementsPath = function() {
        var $node = jQuery(this.getSelectedNode(true));
        var node_name = $node.nodeName();
        var path = ('#text' != node_name) ? [node_name] : [];

        $node.parents().each(function() {
            var node_name = jQuery(this).nodeName();
            if ('html' === node_name) return;
            if ('#text' !== node_name) path.push(node_name);
        });

        this.html_path = path.reverse();

        try {
            var $node = jQuery(this.getSelectedHtml());
            if ($node && $node[0]) {
                function push_node($node, html_path) {
                    var node_name = $node.nodeName();
                    if (-1 === jQuery.inArray(node_name, html_path)) html_path.push(node_name);
                    $node.children().each(function() { push_node(jQuery(this), html_path); });
                }
                push_node($node, this.html_path);
            }
        }catch(ex) {};

        return this.html_path;
    };

    _weEdp_.showHtmlElementsPath = function() {
        this.trigger('showHtmlElementsPath');
    };

    _weEdp_._showHtmlElementsPath = function() {
        if (this.is_wysiwyg_mode) {
            var $node = jQuery(this.getSelectedNode(true));
            var path = this.getHtmlElementsPath();
            this.selectButton('*', false);
            this.selectButton(path, true);

            // specials buttons : align
            var $node = jQuery(this.getSelectedNode());
            if ($node.isNode('div, img')) {
                if ($node.hasStyle('float', 'left')) this.selectButton('alignleft', true);
                else if ($node.hasStyle('float', 'right')) this.selectButton('alignright', true);
                else if ($node.hasStyle({ "display": "block", "margin-left": "auto", "margin-right": "auto" })) this.selectButton('center', true);
                else this.selectButton('justify', true);
            }
            else {
                var style = $node.style('text-align');
                if ('left' === style || 'right' === style) this.selectButton('align' + style, true);
                else if ('' !== style) this.selectButton(style, true);
            }
            this.statusbarLeftMessage(path.join(' &gt; '));
        }
    };

    // css management and wysiwym mode

    _weEdp_.setBodyId = function(id) {
        this.$body.attr('id', id);
    };

    _weEdp_.removeBodyId = function() {
        this.$body.removeAttr('id');
    };

    _weEdp_.setBodyClasses = function(classes) {
        this.$body.attr('class', classes);
    };

    _weEdp_.removeBodyClasses = function() {
        this.$body.removeAttr('class');
    };

    // because $head.append(css) not works with ie
    _weEdp_.insertCss = function(css, media) {
        media = media || 'all';

        //var $css = jQuery.createElement('link', { "rel": "stylesheet", "type": "text/css", "href": css, "media": media, "weed:media": media });
        var head = this.$doc[0].getElementsByTagName("head")[0];
        if (head) {
            var css_node = this.$doc[0].createElement('link');
                css_node.type = 'text/css';
                css_node.rel = 'stylesheet';
                css_node.href = css;
                css_node.media = media;
            head.appendChild(css_node);
            jQuery(css_node).attr('weed:media', media);
        }
    };

    _weEdp_.toggleCss = function() {
        this.displayCss(!this.is_css_active)
        this.is_css_active = !this.is_css_active;
        this.selectButton('show-elements', this.is_wysiwym_mode);
    };

    _weEdp_.displayCss = function(state, css) {
        state = undefined === state ? true : state;
        // if ! css : all
        css = 'string' === typeof css ? css : undefined === css ? null : css.href;

        if (css) var $css = this.$head.find('link[type="text/css"][href="' + css + '"]');
        else var $css = this.$head.find('link[type="text/css"][href!="' + this.options['wysiwym_css'] + '"]');

        if (!$css.length) return ;
        if (!state) $css.attr('media', 'none');
        else $css.each(function() { var $this = jQuery(this); $this.attr('media', $this.attr('weed:media')); });
    };

    _weEdp_.addCss = function(css, media, display) {
        if (2 === arguments.length && 'boolean' === typeof media) {
            display = media;
            media = null;
        }
        if (1 === arguments.length) {
            display = true;
            media = null;
        }

        if ('string' === typeof css) css = {"href": css, "media": media ? media : "all"};
        if ('object' === typeof css) {
            if (!css.href) return false;
            if (!css.media) css['media'] = media ? media : "all";
        }

        if (!css.href || !css.media) return false;

        var $css = jQuery(this.$doc[0].head).find('link[type="text/css"][href="' + css.href + '"]');
        if (!$css.length) {
            this.options.css.push(css);
            this.insertCss(css.href, css.media);
            this.$head.find('link[type="text/css"][href="' + this.options['wysiwym_css'] + '"]').remove();
            this.insertCss(this.options['wysiwym_css'], this.is_wysiwym_mode ? 'all' : 'none');
        }

        return true;
    };

    _weEdp_.removeCss = function(css) {
        if ('string' === typeof css) {
            if ('all' === css) {
                this.options.css = [];
                this.$head.find('link[type="text/css"][href!="' + this.options['wysiwym_css'] + '"]').remove();
                return ;
            }
            css = {"href": css, "media": media ? media : "all"};
        }
        if ('object' === typeof css) {
            if (!css.href) return false;
            if (!css.media) css['media'] = css.media ? css.media : (media ? media : "all");
        }

        if (!css.href || !css.media) return false;

        var exists = false;
        var l = this.options.css.length;
        for (var i = 0; i < l && !exists; i++) {
            if (this.options.css[i].href === css.href) {
                this.options.css.splice(i, 1);
                return true;
            }
        }
    };

    // wysiwym mode
    _weEdp_.toggleWisiwym = function() {
        if (!this.is_wysiwym_mode)
            this.$head.find('link[type="text/css"][href="' + this.options['wysiwym_css'] + '"]').attr('media', 'all');
        else
            this.$head.find('link[type="text/css"][href="' + this.options['wysiwym_css'] + '"]').attr('media', 'none');

        this.is_wysiwym_mode = !this.is_wysiwym_mode;
        this.selectButton('show-elements', this.is_wysiwym_mode);
    };

    // css management and wysiwym mode

    _weEdp_.toggle = function() {
        this.trigger('toggle');
    };

    _weEdp_._toggle = function() {
        if (this.is_wysiwyg_mode) {
            this.$doc.find('a, img').each(function() {
                var $this = jQuery(this);
                if ($this.attr('weed:initialsrc')) {
                    if ($this.isNode('a')) $this.attr('href', $this.attr('weed:initialsrc'));
                    else $this.attr('src', $this.attr('weed:initialsrc'));
                    $this.removeAttr('weed:initialsrc');
                }
            });
            // restore body, required for ie
            this.setHeaderBody();
            this.updateTextarea(this.$body.html());
            this.$iframe.hide();
            this.$element.show();
        }
        else {
            this.updateIFrame(this.$element.val());
            this.$element.hide();
            this.$iframe.show();
        }
        this.is_wysiwyg_mode = !this.is_wysiwyg_mode;
        this.enableButton('*', this.is_wysiwyg_mode);
        this.focusToFirstNode();
    };

    _weEdp_.updateIFrame = function(html) {
        this.trigger('updateIFrame', { "html": jQuery.trim(html || '') });
    };

    _weEdp_._updateIFrame = function(data) {
        data.html = jQuery.trim(data.html || '');
        data.html = this.preventBrPre(data.html, 1);
        //data.html = this.preventNoBreakSpaces(data.html, 1);

        this.$body.html(data.html);

        this.preventLinkClick();
        this.preserveAbsoluteUrls();
        this.fixTable();

        this.saveRange();
        this.focusToFirstNode();
    };

    _weEdp_.updateTextarea = function(html) {
        //alert('updateTextarea\n'+html);
        //html = this.preventNoBreakSpaces(html, 2);
        this.trigger('updateTextarea', { "html": jQuery.trim(html || '') });
    };

    _weEdp_._updateTextarea = function(data) {
        data.html = jQuery.trim(data.html || '');
        data.html = this.preventBrPre(data.html, 2);
//        data.html = this.preventNoBreakSpaces(data.html, 2);
        this.$element.val(data.html);
    };

    _weEdp_.enableDesignMode = function() {
        if (this.is_design_mode) return ;
        if (this.$doc && this.$doc[0]) {
            if ('contentEditable' in this.$doc[0]) this.$doc[0].contentEditable = true;
            if ('designMode' in this.$doc[0]) {
                this.$doc[0].designMode = 'on';
                try { this.$doc[0].execCommand('useCSS', false, false); } catch (ex) { }
                try { this.$doc[0].execCommand('styleWithCSS', false, false); } catch (ex) { }
            }
            this.is_design_mode = true;
        }
        //this.setHeaderBody();
    };

    _weEdp_.disableDesignMode = function() {
        if (!this.is_design_mode) return ;
        if (this.$doc && this.$doc[0]) {
            if ('contentEditable' in this.$doc[0]) this.$doc[0].contentEditable = false;
            if ('designMode' in this.$doc[0]) this.$doc[0].designMode = 'off';
            this.is_design_mode = false;
        }
    };

    // end common editor methods

    // command management
    _weEdp_.addCommand = function(command, fn) {
        this.commands[command] = fn;
    };

    _weEdp_.execUserCommand = function(command, value) {
        var fn = this.commands[command];
        if (undefined !== fn) {
            if (jQuery.isFunction(fn)) this.commands[command].call(value || null);
            else if (jQuery.isArray(fn) && 2 == fn.length) fn[0].apply(fn[1], [value || null]);
        }
    };

    _weEdp_.execCommand = function(command, value) {
        this.trigger('execCommand', { "command": command, "value": value });
    };

    _weEdp_._execCommand = function(data) {
        var command = data.command, value = data.value;
        if (command && this.$doc && this.$doc[0]) {
            this.focus();
            this.restoreRange();

            if (jQuery.browser.msie && command.toLowerCase() === "inserthtml") {
                var r = this.getRange();
                r.pasteHTML(value);
                r.collapse(false);
                r.select();
            }
            else {
                // FF !
                try { this.$doc[0].execCommand('useCSS', false, false); } catch (ex) { }
                try { this.$doc[0].execCommand('styleWithCSS', false, false); } catch (ex) { }
                // ! FF !

                var selected_node = this.getSelectedNode();

                if (-1 != jQuery.inArray(command, ['justifyleft', 'justifyright', 'justifycenter', 'justifyfull'])) {
                    var $selected_node = jQuery(selected_node);
                    if ($selected_node.isNode('div, img')) {
                        switch (command) {
                            case 'justifyleft':
                                if ($selected_node.hasStyle('float', 'left')) $selected_node.style('float', '');
                                else $selected_node.style({'display': '', 'float': 'left', 'margin-left': '', 'margin-right': ''});
                                break;

                            case 'justifyright':
                                if ($selected_node.hasStyle('float', 'right')) $selected_node.style('float', '');
                                else $selected_node.style({'display': '', 'float': 'right', 'margin-left': '', 'margin-right': ''});
                                break;

                            case 'justifycenter':
                                if ($selected_node.hasStyle({ "display": "block", "margin-left": "auto", "margin-right": "auto" }))
                                    $selected_node.style({'display': '', 'float': '', 'margin-left': '', 'margin-right': ''});
                                else
                                    $selected_node.style({'display': 'block', 'float': '', 'margin-left': 'auto', 'margin-right': 'auto'});
                                break;

                            case 'justifyfull':
                                $selected_node.style({'display': '', 'float': '', 'margin-left': '', 'margin-right': ''});
                                break;
                        }
                    }
                    else {
                        var style = $selected_node.style('text-align');
                        switch (command) {
                            case 'justifyleft':
                                if ('left' === style) $selected_node.style('text-align', '');
                                else $selected_node.style('text-align', 'left');
                                break;

                            case 'justifyright':
                                if ('right' === style) $selected_node.style('text-align', '');
                                else $selected_node.style('text-align', 'right');
                                break;

                            case 'justifycenter':
                                if ('center' === style) $selected_node.style('text-align', '');
                                else $selected_node.style('text-align', 'center');
                                break;

                            case 'justifyfull':
                                if ('justify' === style) $selected_node.style('text-align', '');
                                else $selected_node.style('text-align', 'justify');
                                break;
                        }
                    }
                    if ('' === jQuery.trim($selected_node.attr('style'))) $selected_node.removeAttr('style');
                }
                else if (('indent' != command.toLowerCase() && 'outdent' != command.toLowerCase()) || ('li' === selected_node.nodeName.toLowerCase())) {
                    switch (value) {
                        case '<clear>':
                            // special : move node and next siblings nodes after parent
                            var range = this.getRange();
                            var $node = range.startContainer && range.startContainer.childNodes.length ? jQuery(range.startContainer.childNodes[range.startOffset]) : jQuery(this.getSelectedNode(true));
                            var $exec_node = $node;
                            var $parent = null;
                            var $nodes_to_move = [];

                            if (!$node || !$node.length) break;

                            if ($node.isNode('div, p, h1, h2, h3, h4, h5, h6, pre, blockquote, address')) {
                                var $c = $node.contents();
                                $node.after($c);
                                $node.remove();
                                break;
                            }

                            // get parent
                            //search first node before specific parent
                            $node.parents().each(function() {
                                if (-1 != jQuery.inArray(this.nodeName.toLowerCase(), ['div', 'p', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'pre', 'blockquote', 'address'])) {
                                    $parent = jQuery(this);
                                    return false;
                                }
                                else
                                    $exec_node = jQuery(this);
                            });

                            var elt = $exec_node[0];
                            while (elt) {
                                $nodes_to_move.push(jQuery(elt));
                                elt = elt.nextSibling;
                            }
                            if ($parent && $parent.length) {
                                var len = $nodes_to_move.length
                                for (var i = len - 1; i >= 0; i--) $parent.after($nodes_to_move[i]);
                                if ('' === $parent.html()) $parent.remove();
                            }

                            break;

                        case '<div>':
                            var has_div = false;
                            var $selected_node = jQuery(this.getSelectedNode());

                            $selected_node.parents().andSelf().each(function() {
                                var $this = jQuery(this);
                                if ('div' === this.nodeName.toLowerCase()) {
                                    $content = $this.contents();
                                    $this.replaceWith($content);
                                    has_div = true;
                                    return ;
                                }
                            });

                            if (!has_div)
                                this.$doc[0].execCommand(command, false, value || null);
                            break;


                        case '<blockquote>':
                            var has_blockquote = false;
                            var $selected_node = jQuery(this.getSelectedNode());

                            $selected_node.parents().andSelf().each(function() {
                                var $this = jQuery(this);
                                if ('blockquote' === this.nodeName.toLowerCase()) {
                                    $content = $this.contents();
                                    $this.replaceWith($content);
                                    has_blockquote = true;
                                    return ;
                                }
                            });

                            if (!has_blockquote) {
                                var children = ['address', 'blockquote', 'del', 'div', 'dl', 'fieldset', 'form', 'h1', 'h2', 'h3', 'h4', 'h5','h6', 'hr', 'ins', 'noscript', 'ol', 'p', 'pre', 'script', 'table', 'ul'];

                                this.$doc[0].execCommand(command, false, value);

                                var $previous = null;
                                this.$doc.find('blockquote').contents().each(function() {
                                    var $this = jQuery(this);
                                    if ((this.nodeType && 3 === this.nodeType) || (-1 === jQuery.inArray(this.nodeName.toLowerCase(), children))) {
                                        if (!$previous) {
                                            $previous = jQuery.createElement('p');
                                            $this.before($previous);
                                        }
                                        $previous.append($this);
                                    }
                                    else
                                        $previous = $this;
                                });
                            }
                            break;

                        default:
                            if (undefined !== this.commands[command])
                                this.execUserCommand(command, value || null);
                            else
                                this.$doc[0].execCommand(command, false, value || null);
                            break;
                    }
                    this.showHtmlElementsPath();
                }
            }

            this.saveRange();
            this.focus();
            this.fixTable();
        }
    };
    // end command management

    // range and nodes management
    _weEdp_.focus = function() {
        this.trigger('focus');
    };

    _weEdp_._focus = function() {
        var self = this;
        if (this.is_wysiwyg_mode) {
            setTimeout(function() {
                if (self.$iframe[0].contentWindow) self.$iframe[0].contentWindow.focus();
                else if (self.$iframe[0].contentDocument && self.$iframe[0].contentDocument.documentElement) self.$iframe[0].contentDocument.documentElement.focus();
                self.showHtmlElementsPath();
            }, 0);
        }
        else if (this.$element.is(':visible')) {
            if (this.$element[0].createTextRange) {
                var range = this.$element[0].createTextRange();
                range.move('character', 0);
                range.select();
            }
            else if (this.$element[0].selectionStart) {
                this.$element[0].focus();
                this.$element[0].setSelectionRange(0, 0);
            }
            else this.$element.focus();
        }
    };

    _weEdp_.focusToFirstNode = function() {
        if (!this.is_wysiwyg_mode) return ;
        this.trigger('focusToFirstNode');
    };

    _weEdp_._focusToFirstNode = function() {
        if (!this.is_wysiwyg_mode) return ;

        this.focus();
        var $node = this.$body;
        var node = null;

        if ($node.isNode('body') && $node.children().length) {
            node = $node.children(':first')[0];
            if ('table' === node.nodeName.toLowerCase()) {
                node = jQuery(node).find('caption:first,th:first,td:first')[0];
            }
            else {
                var nn = [];
                function get_text_node(node) {
                    if (nn.length) return ;
                    if (node.nodeType == 3) {
                        if (!/^\s*$/.test(node.nodeValue)) nn.push(node);
                    } else {
                        for (var i = 0, len = node.childNodes.length && !nn.length; i < len; ++i) get_text_node(node.childNodes[i]);
                    }
                }
                get_text_node(node);
                if (nn.length) node = nn[0].parentNode;
            }
        }
        this.focusToNode((node ? node : $node[0]));
    };

    _weEdp_.focusToNode = function(node, focus_to_end) {
        if (!this.is_wysiwyg_mode || !node) return ;

        focus_to_end = focus_to_end || false;
        var range = this.getRange(),
            txt_node = (-1 === jQuery.inArray(node.nodeName.toLowerCase(), ['img']));

        if (range.selectNodeContents) {
            var selection = this.getSelection();
            if (selection) {
                try {
                    selection.removeAllRanges();
                    if (txt_node) {
                        range.selectNodeContents(node);
                    }
                    else {
                        range.selectNode(node);
                    }

                    selection.addRange(range); //@fix it 24/07/2012 ???

                    if (txt_node) {
                        if (focus_to_end) selection.collapseToEnd();
                        else selection.collapse(node, 0);
                    }
                }catch(ex) {}
            }
            this.saveRange();
            this.focus();
        }
        else if (range.moveToElementText) {
            try {
                range = this.$body[0].createTextRange();
                range.moveToElementText(node);
                range.collapse(true);
                range.select();
                node.focus();
                this.saveRange();
                this.showHtmlElementsPath();
            }catch(ex) {  }
        }
    };

    _weEdp_.emphasisNode = function($node, emphase) {
        if (emphase) $node.addClass('weed-css-class-selected');
        else {
            $node.removeClass('weed-css-class-selected');
            if ('' == $node.attr('class')) $node.removeAttr('class');
        }
    };

    if (jQuery.browser.msie) { // check ie version < 9 ??
        _weEdp_.saveRange = function() {
            this.bookmark_range = this.getRange();
        };

        _weEdp_.restoreRange = function() {
            try {
                var range = this.getRange();
                if (this.bookmark_range) {
                    if (range.getBookmark) {
                        this.bookmark_range.select();
                    }
                    else {
                        var selection = this.getSelection();
                        if ('removeAllRanges' in selection && 'addRange' in selection) {
                            selection.removeAllRanges();
                            selection.addRange(this.bookmark_range);
                            if (range.getStartContainer) selection.collapse(range.getStartContainer(), 0);
                        }
                    }
                }
            }catch(ex) {};
        };
    }
    else {
        _weEdp_.saveRange = function(e) { if (e && -1 !== jQuery.inArray(e.target.nodeName.toLowerCase(), ['img'])) { this.focusToNode(e.target); }};
        _weEdp_.restoreRange = function() {};
    }

    _weEdp_.getRange = function() {
        var selection = this.getSelection();
        if (selection && selection.createRange) return selection.createRange();
        if (selection && selection.rangeCount) return selection.getRangeAt(0);
        return document.createRange();
    };

    _weEdp_.getSelection = function() {
        if (this.$doc[0].selection) return this.$doc[0].selection;
        else if (this.$doc[0].parentWindow) return this.$doc[0].parentWindow.getSelection();
        else if (this.$doc[0].defaultView) return this.$doc[0].defaultView.getSelection();
    };

    _weEdp_.getSelectedText = function() {
        this.restoreRange();
        var selection = this.getSelection();
        if (!selection.toString) return this.getRange().text;
        return selection.toString();
    };

    _weEdp_.getSelectedHtml = function() {
        this.restoreRange();
        var range = this.getRange();
        if ('htmlText' in range) return range.htmlText;
        return 'cloneContents' in range ? jQuery.createElement('layer').append(range.cloneContents()).html() : '';
    };

    _weEdp_.getSelectedNode = function(text_node) {
        text_node = text_node || false;

        this.restoreRange();

        var range = this.getRange();
        var element = null;

        if (range.setStart) {
            // based on tinymce editor
            element = range.commonAncestorContainer;
            // Handle selection a image or other control like element such as anchors
            if (!range.collapsed) {
                if (range.startContainer == range.endContainer) {
                    if (range.startOffset - range.endOffset < 2) {
                        if (range.startContainer.hasChildNodes()) {
                            element = range.startContainer.childNodes[range.startOffset];
                        }
                    }
                }
                else {
                    // get child between startContainer and endContainer
                    var node = null, start_index = -1, end_index = -1;
                    for (var i = 0; i < element.childNodes.length; i++) {
                        node = element.childNodes[i];
                        if (node == range.startContainer) start_index = i;
                        if (node == range.endContainer) end_index = i;
                        if (-1 !== start_index && -1 !== end_index) break;
                    }
                    if (start_index >= 0 && end_index >= 0 && start_index + 1 == end_index - 1) {
                        while (element.hasChildNodes()) {
                            element = element.childNodes[start_index + 1];
                            start_index = -1;
                        }
                    }
                }
            }
            if (element && element.nodeType && 3 === element.nodeType && !text_node) {
                return element.parentNode;
            }
            return element;
        }
        return range.item ? range.item(0) : range.parentElement();
    };

    _weEdp_.getSelectedParentNode = function() {
        this.restoreRange();
        var range = this.getRange();
        var parent = range.commonAncestorContainer ? range.commonAncestorContainer :
            range.parentElement ? range.parentElement():
            range.item(0);
        if (parent && parent.nodeType && parent.nodeType != 1) parent = parent.parentNode;
        return parent;
    };
    // end range and nodes management

    _weEdp_.submit = function(e, ajax) {
        this.trigger('submit', { "event": e, "ajax": ajax });
    };

    _weEdp_._submit = function(data) {
        if (this.is_wysiwyg_mode) {
            this.updateTextarea(this.$body.html());
        }
    };

    // tools
    _weEdp_.preventLinkClick = function() {
        this.$doc.find('a').bind('click.weed-no-click-link', function(e) { e.preventDefault(); });
    };

    _weEdp_.preventBrPre = function(html, mode) {
        mode = mode || (this.is_wysiwyg_mode ? 1 : 2);

        html = html.replace(/\r\n/g, '\n').replace(/\n/g, '\uffff');

        var matches = html.match(/<pre[^>]*>(.*?)<\/pre>/mgi);
        if(matches) {
            if (1 === mode) {
                for(var i = 0; i < matches.length; i++) {
                    html = html.replace(matches[i], matches[i].replace(/<br(\s*\/?)>/gi, '\uffff').replace(/\uffff/g, '<br />'));
                }
            }
            else {
                for(var i = 0; i < matches.length; i++) {
                    html = html.replace(matches[i], matches[i].replace(/<br(\s*\/?)>/gi, '\uffff'));
                }
            }
        }

        html = html.replace(/\uffff/g, '\n');
        return html;
    };

    _weEdp_.preventNoBreakSpaces = function(html, mode) {
        mode = mode || (this.is_wysiwyg_mode ? 1 : 2);
        html = html.replace(new RegExp(String.fromCharCode(8203), 'g'), '');

        return 2 === mode ? html : html.replace(/&nbsp;/g, '&#8203;&nbsp;&#8203;');
    };

    _weEdp_.preserveAbsoluteUrls = function() {
        this.$doc.find('a, img').each(function() {
            var $this = jQuery(this);
            $this.attr('weed:initialsrc', ($this.isNode('a') ? $this.attr('href') : $this.attr('src')));
        });
    };

    _weEdp_.fixTable = function() {
        if (!this.$doc.children().length) return ;

        var $last_child = this.$doc.children().last();
        if ($last_child.isNode('table')) {
            this.$body.append('<p><br /></p>');
        }
    };

    _weEdp_.setHeaderBody = function() {
        // restore body, required for ie
        this.$doc = this.$iframe[0].contentWindow ? jQuery(this.$iframe[0].contentWindow.document) : jQuery(this.$iframe[0].contentDocument);
        this.$head = this.$doc.find('head');
        this.$body = this.$doc.find('body');
    };

    _weEdp_.getTempId = function() {
        var now = new Date();
        return 'weed-temp-id-' + now.getTime();
    };

    _weEdp_.protectData = function(txt) {
        var map = { "&": "&amp;", "\"": "&quot;", "<": "&lt;", ">": "&gt;" };
        txt = txt.replace(/[&"<>]/g, function(s) { return map[s]; });
        txt = txt.replace(/&amp;([a-z]+|(#\d+));/ig, "&$1;");
        return txt;
    };

    _weEdp_.camelize = function(s) {
        return s.replace (/(?:^|[_])(\w)/g, function (_, c) {
            return c ? c.toUpperCase() : '';
        });
    };

    _weEdp_.underscorize = function(s) {
        return s.replace(/([A-Z])/g, function(c){
            return c ? '_' + c.toLowerCase() : '';
        }).replace(/^_|_$/g, '');
    };

    _weEdp_._setDoctype = function(data) {
        if (-1 !== jQuery.inArray(data.doctype, ['xhtml-1.0-strict', 'xhtml-1.0-transitional'])) {
            this.options.doctype = data.doctype;
        }
    };

    _weEdp_.setDoctype = function(doctype) {
        this.trigger('setDoctype', { "doctype": doctype });
    };

    _weEdp_._ready = function(data) {
        if (this.options.ready && jQuery.isFunction(this.options.ready)) {
            this.options.ready();
        }
    };
    // end tools

    // plugins
    _weEdp_.loadJs = function(file) {
        jQuery.ajax({ "url": file, "async": false, "dataType": "script" });
    };

    _weEdp_.hasPlugin = function(name) {
        return undefined !== this.plugins[name] || undefined !== this.plugins[this.underscorize(name.replace(/^weEd/i, ''))];
    };

    _weEdp_.loadPlugins = function() {
        // plugins are an array of object with plugins definitions
        // definition is :
        //  - name       : the plugin name ;
        //  - class_name : object class name, optional ;
        //  - file       : the plugin file, optional ;
        //  - path       : the plugin file, if the file is not in the plugins dir ;
        //  - options    : plugin options / configuration.
        if (this.options.plugins && this.options.plugins.length) {
            var l = this.options.plugins.length;
            for (var i = 0; i < l; i++) {
                this.loadPlugin(this.options.plugins[i]);
            }
        }
    };

    _weEdp_.loadPlugin = function(plugin_info) {
        if (!plugin_info) return ;
        var n_c = plugin_info.name ? this.underscorize(plugin_info.name.replace(/^weEd/i, '')) : (plugin_info.class_name ? this.underscorize(plugin_info.class_name.replace(/^weEd/i, '')) : null);

        if (!n_c) return ;

        plugin_info.normalized_name = n_c;

        if (!plugin_info.name) plugin_info.name = n_c;
        if (!plugin_info.path) plugin_info.path = this.options['plugins_path'];
        if (!plugin_info.file) plugin_info.file = this.underscorize(plugin_info.name) + '.js';
        if (!plugin_info.class_name) plugin_info.class_name = 'weEd' + this.camelize(plugin_info.name);
        if (!plugin_info.options) plugin_info.options = {};

        // if class exists
        if (undefined === window[plugin_info.class_name]) {
            this.loadJs(plugin_info.path + plugin_info.file);
            if (undefined === window[plugin_info.class_name]) return ;
        }

        var obj = eval(plugin_info.class_name);
        plugin_info.instance = new obj(plugin_info.options, this);
        this.plugins[plugin_info.normalized_name] = plugin_info; // store plugin with the normalized name
    };

    _weEdp_.initPlugins = function() {
        for (var i in this.plugins) {
            var p = this.plugins[i].instance;
            if ('init' in p) {
                p.init();
            }
        }
    };

    // end plugins

    // language
    _weEdp_.loadLanguage = function(plugin) {
        var self = this;

        if (plugin && self.lang.__plugins[plugin]) return ;

        url = !plugin ? this.options['lang_path'] + this.options.lang + '.js' :
                this.options['lang_path'] + plugin + '-' + this.options.lang + '.js';

        jQuery.ajax({ "url": url, "async": false, "dataType": "json", "success":  function(lng) {
            var l = self.lang;
            if (plugin) {
                self.lang.__plugins[plugin] = {};
                l = self.lang.__plugins[plugin];
            }
            jQuery.extend(true, l, lng);
        }});
    };

    _weEdp_._ = function(key, plugin) {
        return (plugin && this.lang.__plugins[plugin] && 'object' === typeof this.lang.__plugins[plugin] && this.lang.__plugins[plugin][key] ?  this.lang.__plugins[plugin][key] :
                (this.lang[key] ? this.lang[key] : key));
    };
    // end language

    // events
    _weEdp_.loadEvents = function() {
        var self = this;
        // smalls functions are here, otherwise it's a method callback with event name prefixed by an underscore
        this.bind({
            "statusbarLeftMessage.__weed": [ { "message": "" }, function(data) { this.$statusbar_left.html(data.message); }],

            "statusbarRightMessage.__weed": [ { "message": "", "duration": 2500 }, function(data) {
                this.$statusbar_right.html(data.message || "");
                this.focus();
                var self = this;
                if (this.statusbar_right_message_timeout) clearTimeout(this.statusbar_right_message_timeout);
                this.statusbar_right_message_timeout = setTimeout(function resetMessage() { self.$statusbar_right.html(""); }, data.duration || 2500);
            }],

            "showHtmlElementsPath.__weed": self._showHtmlElementsPath,
            "resize.__weed": [ { "width": null, "height": null }, self._resize ],
            "executeDialog.__weed": [ { "$button": null }, self._executeDialog ],
            "positionDialog.__weed": [ { "$button": null }, self._positionDialog ],
            "hideDialog.__weed": [ { "$button": null }, self._hideDialog],
            "toggle.__weed": self._toggle,
            "enableButton.__weed": [ { "button": null, "enable": null }, self._enableButton ],
            "updateIFrame.__weed": [ { "html": "" }, self._updateIFrame ],
            "updateTextarea.__weed": [ { "html": "" }, self._updateTextarea ],
            "focus.__weed": self._focus,
            "focusToFirstNode.__weed": self._focusToFirstNode,
            "execCommand.__weed": [ { "command": null, "value": null }, self._execCommand],
            "submit.__weed": [ { "event": null, "ajax": false }, self._submit ],
            "setDoctype.__weed": [ { "doctype": "xhtml-1.0-strict" }, self._setDoctype ]
        });
    };

    _weEdp_.bind = function(name, data, fn) {
        // bind from object
        if (jQuery.isPlainObject(name)) {
            for (var i in name) {
                if (jQuery.isArray(name[i])) {
                    if (1 === name[i].length) this.bind(i, name[i][0]);
                    else if (2 === name[i].length) this.bind(i, name[i][0], name[i][1]);
                }
                else if (jQuery.isFunction(name[i])) this.bind(i, name[i]);
            }
            return ;
        }

        if (jQuery.isFunction(data)) {
            fn = data;
            data = null;
        }

        if (!jQuery.isFunction(fn)) return ;

        // many bind
        var n = name.split(/\s/);
        var l = n.length;
        if (l > 1) {
            for (var i = 0; i < l; i++)
                this.bind(n[i], data, fn);
            return ;
        }

        var names = name.split('.');
        var event = {
            "type": {
                    "name": names[0],
                    "position": (names.length >= 2 && -1 !== jQuery.inArray(names[1], ['pre', 'post']) ? names[1] : 'post'),
                    "namespace": (2 === names.length && -1 === jQuery.inArray(names[1], ['pre', 'post']) ? names[1] : (3 === names.length ? names[2] : null))
            },
            "fn": fn,
            "data": (undefined !== data && jQuery.isPlainObject(data) ? data : {})
        }

        if (undefined === this.events[event.type.name]) this.events[event.type.name] = { "pre": [], "post": [] };
        this.events[event.type.name][event.type.position].push(event);
    };

    _weEdp_.trigger = function(name, data) {
        data = data && jQuery.isPlainObject(data) ? data : {};

        var names = name.split('.');
        var type = {
            "name": names[0],
            "position": (names.length >= 2 && -1 !== jQuery.inArray(names[1], ['pre', 'post']) ? names[1] : null),
            "namespace": (2 === names.length && -1 === jQuery.inArray(names[1], ['pre', 'post']) ? names[1] : (3 === names.length ? names[2] : null))
        };

        this.executeEvents(type, data);
    };

    _weEdp_.executeEvents = function(type, data) {
        if (undefined === this.events[type.name]) return;
        var e = this.events[type.name];

        // data copy to preserve originals values
        var _data = jQuery.extend(true, {}, data);

        // get data from __weed namespace
        var l = e['post'].length;
        for (var i = 0; i < l; i++) {
            if ('__weed' === e['post'][i].type.namespace) {
                _data = jQuery.extend(true, {}, e['post'][i].data, _data);
            }
        }

        // executes callbacks
        var events = [];
        if (!type.position) {
            jQuery.each(['pre', 'post'], function(index, position) {
                var l = e[position].length;
                for (var i = 0; i < l; i++) events.push(e[position][i]);
            });
        }
        else events = e[type.position];
        var l = events.length;

        for (var i = 0; i < l; i++) {
            var event_type = events[i].type;

            if (((type.name === event_type.name) &&
                ((!type.position) || (type.position === event_type.position)) &&
                ((!type.namespace) || (type.namespace === event_type.namespace)))) {
                    _data = jQuery.extend(true, {}, events[i].data, _data);
                    events[i].fn.apply(this, [_data]);
            }
        }
    };

    _weEdp_.unbind = function(name) {
        // many unbind
        var n = name.split(/\s/);
        var l = n.length;
        if (l > 1) {
            for (var i = 0; i < l; i++)
                this.unbind(n[i]);
            return ;
        }

        var names = name.split('.');
        var type = {
            "name": names[0],
            "position": (names.length >= 2 && -1 !== jQuery.inArray(names[1], ['pre', 'post']) ? names[1] : null),
            "namespace": (2 === names.length && -1 === jQuery.inArray(names[1], ['pre', 'post']) ? names[1] : (3 === names.length ? names[2] : null))
        };

        if (!type.position) {
            var self = this;
            jQuery.each(['pre', 'post'], function(index, position)  {
                type.position  = position;
                self.removeEvents(type);
            });
        }
        else this.removeEvents(type);

    };

    _weEdp_.removeEvents = function(type) {
        if (undefined === this.events[type.name]) return;

        var new_events = [];
        var e = this.events[type.name];
        var l = e[type.position].length;
        for (var i = 0; i < l; i++) {
            var event_type = e[type.position][i].type;

            // removes __weed namespace only if it's specified
            if (false === ((type.name === event_type.name) &&
                ((!type.position) || (type.position === event_type.position)) &&
                ((!type.namespace && '__weed' !== event_type.namespace) || (type.namespace === event_type.namespace)))) {
                    new_events.push(e[type.position][i]);
            }
        }

        e[type.position] = new_events;
    };
    // end events
};

jQuery.fn.weEd = function(options) {
    return this.each(function() {
        var $this = $(this);
        $this.data('weEd', new weEd(options, $this));
    });
};
