/*
 * Copyright 2012 Sébastien Raud
 *
 * This file is part of beCms.
 *
 * beCms is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * beCms is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with beCms.  If not, see <http://www.gnu.org/licenses/>.
 */

/*jslint plusplus: true, regexp: true, newcap: true, maxerr: 10 */
"use strict";
/*global jQuery: true, window:true, document: true, confirm: true, escape: true, weEd: true, CodeMirror: true */

if (undefined === window.console) {
    window.console = {};
    window.console.log = function (s) {};
}

jQuery(function () {

    if (!jQuery.createElement) { // make jquery element
        jQuery.extend({
            createElement: function (name, attributes) {
                var $elt = jQuery(document.createElement(name)), key = null;
                if (!attributes) { return $elt; }

                //jQuery.each(attributes, function (key, value) {
                for (key in attributes) {
                    if (attributes.hasOwnProperty(key)) {
                        switch (key) {
                        case 'class':
                            $elt.addClass(attributes[key]);
                            break;
                        case 'html':
                            $elt.html(attributes[key]);
                            break;
                        case 'text':
                            $elt.text(attributes[key]);
                            break;
                        default:
                            $elt.attr(key, attributes[key]);
                            break;
                        }
                    }
                }
                return $elt;
            }
        });
    }
});

/*
beCms
beCms.mvc
beCms.ui
beCms.ajax
beCms.session
beCms.tools
*/
(function (window, undefined) {
    window.be_cms_instance = window.be_cms_instance || null;
    var beCms = {
        init: function () {
            if (window.be_cms_instance) { return window.be_cms_instance; }

            window.be_cms_instance = this;
            var self = this,

            // shortcuts
                $html = jQuery('html'),
                $body = jQuery('body'),
                $window = jQuery(window);

            // configuration

            // configuration array
            self.configuration = {};

            /**
             * Loads, or reloads, the configuration from the html page.
             *
             * @return void
             */
            self.loadConfiguration = function () {
                self.configuration = {};
                jQuery('input[name^="_config"]').each(function () {
                    var $this = jQuery(this);
                    self.setConfiguration($this.attr('name').replace('_config', ''), $this.val());
                });
            };

            /**
             * Returns a configuration value from a key.
             *
             * @param  string  key            Key.
             * @param  mixed   default_value  Default value to return if key not exists.
             * @return mixed
             */
            self.getConfiguration = function (key, default_value) {
                default_value = default_value || null;
                var split_key = key.replace(/^\/+/, '').replace(/\/+$/, '').split(/\/+/),
                    config = self.configuration,
                    i = 0;
                for (i = 0; i < split_key.length; i++) {
                    if (i !== split_key.length - 1) {
                        if (config[split_key[i]]) {
                            config = config[split_key[i]];
                        } else {
                            return default_value;
                        }
                    } else {
                        return config[split_key[i]] || default_value;
                    }
                }
                return default_value;
            };

            /**
             * Sets a configuration value to a key.
             *
             * @param  string  key    Key.
             * @param  mixed   value  Value.
             * @return void
             */
            self.setConfiguration = function (key, value) {
                var split_key = key.replace(/^\/+/, '').replace(/\/+$/, '').split(/\/+/),
                    config = self.configuration,
                    i = 0;

                for (i = 0; i < split_key.length; i++) {
                    if (i < split_key.length - 1) {
                        if (!config[split_key[i]]) {
                            config[split_key[i]]  = {};
                        }
                        config = config[split_key[i]];
                    } else {
                        config[split_key[i]] = value;
                    }
                }
            };
            // end configuration


            // mvc
            self.mvc = {
                /**
                 * Dispatches javascripts actions from the page name.
                 *
                 * @return void
                 */
                dispatch: function () {
                    var controller,
                        f;
                    self.loadConfiguration();
                    self.session.ieFix();
                    self.session.indicator();

                    controller = self.tools.getControllerBaseName() + 'Controller';
                    if (jQuery.isFunction(this[controller])) {
                        this[controller].apply(this);
                    } else {
                        this.defaultController();
                    }

                    // editor resize bug with Chrome if not visible
                    f = function () {
                        if (!self.ui.components.editor_instance.length || self.ui.components.editor_instance.length == self.getConfiguration('beCms/editor/loaded')) {
                            self.ui.components.menuSelectItem();
                        }
                        else {
                            setTimeout(f, 500);
                        }
                    };
                    f();
                },

                /**
                 * Dispatches javascripts actions from ajax.
                 *
                 * @return void
                 */
                ajaxDispatch: function ($obj, e) {
                    var is_link = 'a' === $obj[0].nodeName.toLowerCase(),
                        action = is_link ? self.tools.getUriValue($obj.attr('href'), 'action') : $obj.val(),
                        uri = is_link ? $obj.attr('href') : $obj.parents('form:first').attr('action'),
                        re = null,
                        controller = null;

                    if (jQuery.browser.msie && 'button' === $obj[0].nodeName.toLowerCase()) { // ie bug, value for button is the button content
                        re = /value="([\w\d\-\_]+)"/i.exec(jQuery.createElement('div').append($obj.clone()).html() || '');
                        if (re.length >= 2) {
                            action = re[1];
                        }
                    }
                    if (!action) {
                        return true;
                    }
                    // controller name. For modules, controller name start with 'module' to signify "current module"
                    controller = (self.tools.getControllerBaseName(uri) + 'Ajax' + self.tools.camelize(action) + 'Controller').replace(/(^modules)/, 'module');

                    if (jQuery.isFunction(this[controller])) {
                        e.preventDefault();
                        e.stopPropagation();
                        e.stopImmediatePropagation();

                        // get ajax request
                        self.ajax.request(is_link ? $obj : $obj.closest('form'), action, this[controller]);
                    }
                },

                /**
                 * Default controller. Used if controller not exists.
                 *
                 * @return void
                 */
                defaultController: function () {
                    self.setConfiguration('current_page/title', jQuery('title').html());

                    if (!jQuery('div.loading').length) {
                        self.ui.components.messageBox(jQuery('<ul><li>Chargement de la page...</li></ul>'), 'loading');
                    }
                    self.ui.components.menu();
                    self.ui.components.help();
                    self.ui.events.notify();

                    self.ui.events.changesIndicator();
                    self.ui.events.ajaxify(null);
                    self.ui.components.externalLinks();
                    self.ui.components.imagesLightbox();
                    self.ui.events.confirmationMessage(jQuery('.ico-delete'), 'Voulez-vous vraiment supprimer cet élément ?');

                    var f = function () {
                        var js_to_load = self.getConfiguration('beCms/files/js/load', []),
                            js_loaded = self.getConfiguration('beCms/files/js/loaded', []),
                            success_to_play = self.getConfiguration('beCms/files/js/success_to_play', []);

                        if (js_to_load.length !== js_loaded.length) { // @todo check it
                            setTimeout(f, 500);
                        } else {
                            jQuery.each(success_to_play, function (index, func) { func(); });
                            self.setConfiguration('beCms/files/js/load', []);
                            self.setConfiguration('beCms/files/js/loaded', []);
                            self.setConfiguration('beCms/files/js/success_to_play', []);
                            self.ajax.loadRessources();
                            jQuery('div.loading').remove();
                        }
                    };
                    setTimeout(f, 500);
                },

                /**
                 * Default controller for ajax responses. Used if controller not exists.
                 *
                 * @parameters  jQuery  $html        jQuery object, html content. Optional.
                 * @return void
                 */
                defaultAjaxController: function ($html) {
                    $html = $html || $body;
                    self.ui.components.help($html);
                    self.ui.events.changesIndicator($html);
                    self.ui.events.ajaxify($html);
                    self.ui.components.externalLinks($html);
                    self.ui.events.confirmationMessage($html.find('.ico-delete'), 'Voulez-vous vraiment supprimer cet élément ?');
                },

                // pages

                loginController: function () {
                    var $form = jQuery('form');
                    if (!(jQuery.browser.msie && jQuery.browser.version <= 7)) {
                        self.ui.effects.centering($form);
                    }
                    self.tools.formFocus($form);
                },

                homeController: function () {
                    var $dashbord = jQuery('div.dashbord'),
                        max_height = 0,
                        dashbord_height;

                    this.defaultController();

                    if ($dashbord.length) {
                        dashbord_height = function() {
                            $dashbord
                                .each(function() { max_height = Math.max(max_height, jQuery(this).height()); })
                                .height(max_height);
                        }
                        $window.bind('resize.dashbord', dashbord_height);
                        dashbord_height();
                    }
                },

                structureController: function () {
                    this.defaultController();
                    self.ui.components.treeview();
                },

                editController: function () {
                    this.defaultController();
                    if (jQuery('.ui-editor-html, .ui-editor-php, .ui-editor-xml').length) {
                        self.ui.components.editor();
                        jQuery('div:not([id]).nav li:nth-child(2)').bind('click.resize-editor', function () { // resize editor on first click
                            if (self.ui.components.editor_instance[0]) {
                                if ('resize' in self.ui.components.editor_instance[0]) {
                                    $(window).trigger('resize.editor-0');
                                    self.ui.components.editor_instance[0].focus();
                                } else if (self.ui.components.editor_instance[0].refresh) {
                                    self.ui.components.editor_instance[0].refresh();
                                }
                            }
                            //jQuery(this).unbind('click.resize-editor');
                        });
                    }

                    self.ui.components.treeview();
                    if (jQuery('textarea#page_css_files').length) {
                        self.ui.components.toSelect(jQuery('textarea#page_css_files'), jQuery('textarea#page_css_list'));
                        // bind click with editorvar editor = jQuery('.ui-editor-html, .ui-editor-php, .ui-editor-xml').data('editor');
                        if (self.ui.components.editor_instance[0] && self.ui.components.editor_instance[0].updateTextarea) {
                            jQuery('a#page_css_files_up_select, a#page_css_files_down_select, a#page_css_files_add_list, a#page_css_files_remove_list').bind('click.beCms/tools/syncEditorCss', function () {
                                // get list
                                var a = [];
                                jQuery('select#page_css_files_select_in').find('option').each(function () { a.push(self.getConfiguration('request/protocol') + '://' + self.getConfiguration('base_uri_ressources_path') + jQuery(this).val()); });
                                self.tools.syncEditorCss(self.ui.components.editor_instance[0], a);
                            });
                        }
                    }
                    if (jQuery('textarea#page_js_files').length) {
                        self.ui.components.toSelect(jQuery('textarea#page_js_files'), jQuery('textarea#page_js_list'));
                    }

                    this.editControllerManageDisabled();

                    if (self.ui.components.editor_instance[0] && self.ui.components.editor_instance[0].setDoctype) {
                        jQuery('select#page_doctype').bind('change.update-doctype', function () {
                            self.ui.components.editor_instance[0].setDoctype(jQuery(this).val());
                        });
                    }

                    if ('php' === jQuery('input#page_type').val()) {
                        jQuery('select#page_type_mime').bind('change.update-doctype', function(e) {
                            if ('xml' === jQuery(this).children('option:selected').val()) {
                                jQuery('select#page_doctype').closest('p').addClass('hide');
                            }
                            else {
                                jQuery('select#page_doctype').closest('p').removeClass('hide');
                            }
                        });
                    }
                    if ('html' === jQuery('input#page_type').val()) {
                        jQuery('input#page_body_id').bind('change.update-body-id', function() {
                            var val = jQuery(this).val();
                            if ('' === jQuery.trim(val))
                                self.ui.components.editor_instance[0].removeBodyId();
                            else
                                self.ui.components.editor_instance[0].setBodyId(val);
                        });
                        jQuery('input#page_body_classes').bind('change.update-body-classes', function() {
                            var val = jQuery.trim(jQuery(this).val() + ' beCms');
                            self.ui.components.editor_instance[0].setBodyClasses(val);
                        });
                    }

                    if (jQuery('button#action_pusblish').length) {
                        self.ui.events.confirmationMessage(jQuery('button#action_pusblish'), 'Voulez-vous vraiment passer ce brouillon en page réelle ?');
                    }

                    // bind editor for ajax
                    jQuery('form#edit').bind('ajax', function (e) {
                        if (self.ui.components.editor_instance[0]) {
                            if (self.ui.components.editor_instance[0].submit) {
                                self.ui.components.editor_instance[0].submit(e, true);
                            } else if (self.ui.components.editor_instance[0].save) {
                                self.ui.components.editor_instance[0].save();
                            }
                        }
                    });
                },

                editControllerManageDisabled: function () {
                    if (jQuery('input#page_locked_id').length && self.session.getUser() !== jQuery('input#page_locked_id').val()) {
                        jQuery('button, input[type!="hidden"], select, textarea').attr("disabled", "disabled");
                        if (self.ui.components.editor_instance[0] && self.ui.components.editor_instance[0].enabled) {
                            self.ui.components.editor_instance[0].enabled(false);
                        }
                        jQuery('div.nav a.text-button-ico')
                            .not('.ico-cancel, .link-edit-draft, .link-edit-real')
                            .filter(function () { return (0 === jQuery(this).parents('ul.ui-treeview').length && 0 === jQuery(this).parents('div#nav').length); })
                            .addClass('disabled')
                            .bind('click.beCms/editController', function (e) {
                                e.preventDefault();
                                e.stopImmediatePropagation();
                                e.stopPropagation();
                                return false;
                            });
                    }
                    if ('draft' === jQuery('input#type').val()) {
                        jQuery('input#page_slug, input#page_cache').attr('disabled', 'disabled');
                    }

                    if ('php' === jQuery('input#page_type').val()) {
                        if ('xml' === jQuery('select#page_type_mime option:selected').val()) {
                            jQuery('select#page_doctype').closest('p').addClass('hide');
                        }
                    }
                },

                filesController: function () {
                    var $bt_list_dir = jQuery('#files_document_list_dir_submit'),
                        $bt_list_img = jQuery('#files_image_list_dir_submit');

                    this.defaultController();

                    $bt_list_dir.hide();
                    $bt_list_img.hide();

                    jQuery('#files_document_list_dir').bind('change', function()  {
                        $bt_list_dir.trigger('click');
                    });
                    jQuery('#files_image_list_dir').bind('change', function()  {
                        $bt_list_img.trigger('click');
                    });
                },

                configurationController: function () {
                    this.defaultController();
                    self.ui.components.toSelect(jQuery('textarea#configuration_pages_default_css'), jQuery('textarea#configuration_pages_default_css_list'));
                    self.ui.components.toSelect(jQuery('textarea#configuration_pages_default_js'), jQuery('textarea#configuration_pages_default_js_list'));
                },

                usersController: function () {
                    this.defaultController();
                },

                modulesController: function () {
                    var module_a,
                        module_key,
                        fn;

                    if (module_a = /.+\/modules\/([\w\d\-\_]+)($|[#\&\?\/.]+)/i.exec(window.location.href.toLowerCase())) {
                            module_key = module_a[1];
                    }

                    if (module_key) {
                        self.ajax.loadJs('/js/back/modules/' + module_key + '/controllers.js');
                        // controller name start with 'module' to signify "current module"
                        fn = (self.tools.getControllerBaseName(window.location.href) + 'Controller').replace(/^modules/, 'module');
                        if (jQuery.isFunction(this[fn])) {
                            this[fn].apply(self);
                        }
                    }

                    this.defaultController();
                },

                // ajax

                structureAjaxEditController: function ($link, xml, $html) {
                    this.structureAjaxEditableLighbox($html);
                },

                structureAjaxAddController: function ($link, xml, $html) {
                    this.structureAjaxEditableLighbox($html);
                },

                structureAjaxSaveController: function ($form, xml, $html) {
                    if (xml.get('messages/success', []).length) {
                        var id = $form.find('input#structure_id').val(),
                            $u = null,
                            $e = null,
                            $ul = null;

                        if ('' !== id) {
                            $u = jQuery(xml.get('line')).children('p');
                            jQuery('li[id="e-' + id + '"] > p').replaceWith($u);
                            self.ui.components.treeview(jQuery('li[id="e-' + id + '"]').closest('ul'));
                        } else {
                            // add new child
                            $u = jQuery(xml.get('line'));
                            if (null !== (id = $form.find('input#structure_parent_id').val().match(/([\w\d\.]+)$/))) {
                                id = id[1];
                                $e = jQuery('li[id="e-' + id + '"]');
                                $ul = $e.children('ul');
                                if (!$ul.length) {
                                    $ul = jQuery.createElement('ul');
                                    $e.append($ul);
                                }
                                $ul.append($u);
                                if (1 === $ul.children('li').length) {
                                    $e.children('p').find('span.ui-treeview-none')
                                        .removeClass('ui-treeview-none').addClass('ui-treeview-minus');
                                    self.ui.components.treeview($e.closest('ul'));
                                }
                            }
                        }
                        this.defaultAjaxController($u);
                    }
                    this.structureAjaxEditableLighbox($html);
                },

                structureAjaxDeleteController: function ($link, xml) { //reprendre
                    if (xml.get('messages/success', []).length) {
                        var id = self.tools.getUriValue($link.attr('href'), 'id'),
                            $e = null,
                            $u = null,
                            $this = null;

                        if (id) {
                            $e = jQuery('li[id="e-' + id + '"]'); // id contains a dot

                            $u = $e.children('ul');
                            if ($u.length) {
                                $u.children('li').each(function(index, $this) {
                                    $e.after($this);
                                });
                            }
                            $u = $e.closest('ul');
                            $e.remove();
                            if (!$u.children().length) {
                                $u.prev('p').find('span:first')
                                    .removeClass('ui-treeview-minus ui-treeview-plus').addClass('ui-treeview-none')
                                    .unbind('click.beCms/ui/components/treeview');
                                $u.remove();
                            }
                        }
                        self.ui.components.$lightbox.hide();
                    }
                },

                structureAjaxClearCacheController: function ($link, xml) {
                },

                structureAjaxUpController: function ($link, xml, $html) {
                    this.structureAjaxUpDownController($link, xml, $html, 'up');
                },

                structureAjaxDownController: function ($link, xml, $html) {
                    this.structureAjaxUpDownController($link, xml, $html, 'down');
                },

                structureAjaxUpDownController: function ($link, xml, $html, up_down) { //reprendre
                    if (xml.get('messages/success', []).length) {
                        var id = self.tools.getUriValue($link.attr('href'), 'id'),
                            $e = null;

                        if (id) {
                            $e = jQuery('li[id="e-' + id + '"]'); // id contains a dot
                            if ('down' === up_down) {
                                $e.next().after($e);
                            } else {
                                $e.prev().before($e);
                            }

                            this.structureAjaxEditableLighbox($html);
                        }
                    }
                },

                structureAjaxEditableLighbox: function ($html) {
                    self.ui.components.htmlLightbox($html);
                },

                editAjaxSaveController: function ($form, xml, $html) {
                    this.editAjaxUpdateAll(xml);
                },

                editAjaxPublishController: function ($form, xml, $html) {
                    this.editAjaxUpdateAll(xml);
                },

                editAjaxEditController: function ($link, xml, $html) {
                    this.editAjaxUpdateAll(xml, true);
                },

                editAjaxLockController: function ($link, xml, $html) {
                    this.editAjaxUnlockController($link, xml, $html);
                },

                editAjaxUnlockController: function ($link, xml, $html) {
                    this.editAjaxUpdateLock(xml);
                    this.editAjaxUpdateButtons(xml);
                    this.defaultAjaxController();
                },

                editAjaxCopyController: function ($link, xml, $html) {
                    this.editAjaxUpdateAll(xml, true);
                },

                editAjaxDeleteController: function ($link, xml, $html) {
                    this.editAjaxUpdateAll(xml);
                },

                editAjaxUpdateAll: function (xml, is_new_content) {
                    //reset_editor_undo = reset_editor_undo || false;
                    var update_val = ['is_draft', 'title', 'slug', 'cache', 'state', 'type', 'redirection', 'body_id', 'body_classes'],
                        i = 0,  // @todo comment and rename vars
                        html = null,
                        v = null,
                        $route = null;

                    is_new_content = is_new_content || false;

                    if (self.ui.components.editor_instance[0] && self.ui.components.editor_instance[0].changeIndicatorUnbind) {
                        self.ui.components.editor_instance[0].changeIndicatorUnbind();
                    }

                    for (i in update_val) {
                        if (update_val.hasOwnProperty(i)) {
                            jQuery('#page_' + update_val[i]).val(xml.get('page/' + update_val[i]));
                        }
                    }

                    if ('php' === xml.get('page/type')) {
                        jQuery('select#page_type_mime').removeAttr('selected');
                        jQuery('select#page_type_mime option[value="' + xml.get('page/type_mime') + '"]').attr('selected', 'selected');
                    } else {
                        jQuery('#page_type_mime').val(xml.get('page/type_mime'));
                    }

                    jQuery('input#type').val(1 === parseInt(xml.get('page/is_draft'), 10) ? 'draft' : '');
                    jQuery('textarea#page_metadata_description').val(xml.get('page/metadata/description'));
                    jQuery('textarea#page_metadata_keywords').val(xml.get('page/metadata/keywords'));

                    jQuery('select#page_layout option').removeAttr('selected');
                    jQuery('select#page_layout option[value="' + xml.get('page/layout') + '"]').attr('selected', 'selected');

                    jQuery('select#page_doctype option').removeAttr('selected');
                    jQuery('select#page_doctype option[value="' + xml.get('page/doctype') + '"]').attr('selected', 'selected');

                    jQuery('textarea#page_css_list').val(xml.get('page/css_files_list'));
                    jQuery('textarea#page_css_files').val(xml.get('page/css_files')).trigger('update');

                    // update css
                    jQuery('a#page_css_files_up_select').trigger('click.beCms/tools/syncEditorCss');

                    jQuery('textarea#page_js_list').val(xml.get('page/js_files_list'));
                    jQuery('textarea#page_js_files').val(xml.get('page/js_files')).trigger('update');

                    // update editor
                    html = xml.get('page/content');
                    if (is_new_content && self.ui.components.editor_instance[0] && self.ui.components.editor_instance[0].updateTextarea) {
                        self.ui.components.editor_instance[0].updateTextarea(html);
                        self.ui.components.editor_instance[0].updateIFrame(html);
                        self.ui.components.editor_instance[0].enabled(true);
                        self.ui.components.editor_instance[0].plugins.undo_redo.instance.undo_buffer = [];
                        self.ui.components.editor_instance[0].plugins.undo_redo.instance.redo_buffer = [];
                        self.ui.components.editor_instance[0].plugins.undo_redo.instance.switchButtons();
                        // update body_id and body_classes
                        v = xml.get('page/body_id');
                        if (!v || '' === v) {
                            self.ui.components.editor_instance[0].$body.removeAttr('id');
                        } else {
                            self.ui.components.editor_instance[0].$body.attr('id', v);
                        }

                        v = xml.get('page/body_classes');
                        if (!v || '' === v) {
                            self.ui.components.editor_instance[0].$body.removeAttr('class').addClass('beCms');
                        } else {
                            self.ui.components.editor_instance[0].$body.removeAttr('class').addClass(v + ' beCms');
                        }
                    } else if (self.ui.components.editor_instance[0] && self.ui.components.editor_instance[0].setValue) {
                        self.ui.components.editor_instance[0].setValue(html);
                    }

                    // update h2
                    jQuery('div.main #ui-page-title').html(xml.get('misc/title'));
                    this.editAjaxUpdateButtons(xml);
                    this.editAjaxUpdateLock(xml);

                    // restore all
                    jQuery('button, input, select, textarea').removeAttr("disabled");
                    jQuery('a.ico').removeClass('disabled').unbind('click.beCms/pagesController');
                    this.editControllerManageDisabled();

                    if (jQuery('button#action_pusblish').length) {
                        self.ui.events.confirmationMessage(jQuery('button#action_pusblish'), 'Voulez-vous vraiment passer ce brouillon en page réelle ?');
                    }

                    $route = jQuery('span#page_route');
                    $route.html(xml.get('misc/preview'));

                    // update element title in page list
                    jQuery('ul.pages-list').find('li[id="e-' + xml.get('page/id') + '"] > p strong').html(xml.get('page/title'));

                    this.defaultAjaxController();

                    if (self.ui.components.editor_instance[0] && self.ui.components.editor_instance[0].changeIndicatorBind) {
                        self.ui.components.editor_instance[0].changeIndicatorBind();
                    }
                },

                editAjaxUpdateLock: function (xml) {
                    // update locked info
                    var locked_id = xml.get('page/locked/id'),
                        update_val = null,
                        i = 0,
                        $e = null,
                        $locked = null,
                        locked = null;

                    if (locked_id) {
                        update_val = ['id', 'firstname', 'lastname'];
                        for (i in update_val) {
                            if (update_val.hasOwnProperty(i)) {
                                $e = jQuery('input#page_locked_' + update_val[i]);
                                if (!$e.length) {
                                    $e = jQuery.createElement('input', { "type": "hidden", "id": "page_locked_" + update_val[i], "name": "page[locked][" + update_val[i] + "]" });
                                    jQuery('form#edit').prepend($e);
                                }
                                $e.val(xml.get('page/locked/' + update_val[i]));
                            }
                        }
                        jQuery('input#page_locked').remove();
                    } else {
                        $e = jQuery('input#page_locked');
                        if (!$e.length) {
                            $e = jQuery.createElement('input', { "type": "hidden", "id": "page_locked", "name": "page[locked]" });
                            jQuery('form#edit').prepend($e);
                        }
                        $e.val(0);
                        update_val = ['id', 'firstname', 'lastname'];
                        for (i in update_val) {
                            if (update_val.hasOwnProperty(i)) {
                                jQuery('input#page_locked_' + update_val[i]).remove();
                            }
                        }
                    }

                    $locked = jQuery('p#ui-page-locked');
                    locked = xml.get('misc/locked');
                    if (!$locked.length) {
                        $locked = jQuery.createElement('p', { "id": "ui-page-locked", "class": "ico ico-lock" });
                        jQuery('div.main #ui-page-title').after($locked);
                    }
                    if ('' !== locked) {
                        $locked.html(locked).show();
                    } else {
                        $locked.html('').hide();
                    }
                },

                editAjaxUpdateButtons: function (xml) {
                    jQuery('p#ui-page-actions').html(xml.get('misc/buttons'));
                },

                filesAjaxEditController: function ($link, xml, $html) {
                    self.ui.components.htmlLightbox($html);
                },

                filesAjaxModifyController: function ($form, xml, $html) {
                    self.ui.components.htmlLightbox($html);
                    $body.find('td[id="file_' + $html.find('input#file_id').val() + '"]').html($html.find('input#file_title').val());
                },

                filesAjaxListController: function ($form, xml, $html) {
                    var type = $form.find('input[name="type"]').val();
                    jQuery('#' + type).replaceWith($html);
                    self.ui.components.imagesLightbox($html);
                },

                filesAjaxDeleteController: function ($link, xml, $html) {
                    // success
                    if (xml.get('messages/success', []).length) {
                        var $tr = $link.closest('tr'),
                            $table = $tr.closest('table');

                        $tr.nextAll('tr').toggleClass('odd');
                        $tr.remove();

                        if (1 === $table.find('tr').length) {// no children
                            $table.remove();
                        }
                    }
                },

                configurationAjaxSaveController: function ($form, xml, $html) {
                    if (window.location.href.replace(/\#.*/, '') !== xml.get('url')) {
                        window.location = xml.get('url');
                    }

                    $form.find('input[id^="configuration_"], textarea[id^="configuration_"]').each(function () {
                        var $this = jQuery(this),
                            name = $this.attr('name'),
                            config_key = null;

                        if (!name || '' === name) {
                            return true;
                        }

                        config_key = name.replace(/[\[\]]/g, '/').replace(/\/+/g, '/').replace(/\/$/, '');
                        if ('input' === this.nodeName.toLowerCase() && 'checkbox' === $this.attr('type')) {
                            if (1 === parseInt(xml.get(config_key), 10)) {
                                $this.attr('checked', 'checked');
                            } else {
                                $this.removeAttr('checked');
                            }
                        } else {
                            $this.val(xml.get(config_key));
                        }
                    });

                    // update extension
                    $form.find('select#configuration_pages_default_extension option').removeAttr('selected');
                    $form.find('select#configuration_pages_default_extension option[value="' + xml.get('configuration/pages/default_extension') + '"]').attr('selected', 'selected');

                    // update layout
                    var $e = $form.find('select#configuration_pages_default_layout'),
                        layouts = null,
                        i = 0,
                        css = null,
                        list = [],
                        js = null;

                    $e.children().remove();
                    layouts = xml.get('layouts');
                    for (i = 0; i < layouts.length; i++) {
                        $e.append(jQuery.createElement('option', { "value": layouts[i], "text": layouts[i] }));
                    }
                    $e.find('option[value="' + xml.get('configuration/pages/default_layout', '') + '"]').attr('selected', 'selected');

                    // update css and js
                    css = xml.get('css');
                    list = [];
                    for (i = 0; i < css.length; i++) {
                        list.push(css[i]);
                    }
                    jQuery('textarea#configuration_pages_default_css_list').val(list.join(';'));
                    jQuery('textarea#configuration_pages_default_css').val(xml.get('configuration/pages/default_css')).trigger('update');

                    js = xml.get('js');
                    list = [];
                    for (i = 0; i < js.length; i++) {
                        list.push(css[i]);
                    }
                    jQuery('textarea#configuration_pages_default_js_list').val(list.join(';'));
                    jQuery('textarea#configuration_pages_default_js').val(xml.get('configuration/pages/default_js')).trigger('update');
                },

                usersAjaxEditController: function ($link, xml, $html) {
                    self.ui.components.htmlLightbox($html);
                },

                usersAjaxAddController: function ($link, xml, $html) {
                    self.ui.components.htmlLightbox($html);
                },

                usersAjaxSaveController: function ($link, xml, $html) {
                    self.ui.components.htmlLightbox($html);
                    var html_list = xml.get('html_list'),
                        $html_list = null;

                    if (html_list) {
                        $html_list = jQuery(html_list);
                        self.mvc.defaultAjaxController($html_list);
                        $body.find('table#users').replaceWith($html_list);
                    }
                },

                usersAjaxDeleteController: function ($link, xml, $html) {
                    // success
                    if (xml.get('messages/success', []).length) {
                        var $tr = $link.closest('tr'),
                            $table = $tr.closest('table');

                        $tr.nextAll('tr').toggleClass('odd');
                        $tr.remove();
                    }
                },

                moduleAjaxInstallController:function ($link, xml, $html) {
                    self.mvc.modulesAjaxUpdateAll($link, xml);
                },

                moduleAjaxUninstallController:function ($link, xml, $html) {
                    self.mvc.modulesAjaxUpdateAll($link, xml);

                    var module_key = self.getConfiguration('module/key'),
                        installed = xml.get('module_installed/' + module_key, false),
                        $li,
                        l,
                        i;

                    if (module_key && !installed) {
                        jQuery('div#main div.main form, div#main div.main div')
                            .filter(function() {
                                return !jQuery(this).hasClass('module-conf')
                            })
                            .remove();

                        $li = jQuery('div#main div.nav ul li');
                        l = $li.length;

                        for (i = 0; i < l - 2; i++) {
                            jQuery($li.get(i)).remove();
                        }
                        jQuery($li.get(0)).bind('click');
                    }
                },

                moduleAjaxUpController:function ($link, xml, $html) {
                    self.mvc.modulesAjaxUpdateAll($link, xml);
                },

                moduleAjaxDownController:function ($link, xml, $html) {
                    self.mvc.modulesAjaxUpdateAll($link, xml);
                },

                modulesAjaxUpdateAll: function ($link, xml) {
                    var $section = jQuery('div#section-installed'),
                        $legend = $section.find('p.legend'),
                        $html = jQuery(xml.get('html_modules_installed'));

                    self.mvc.defaultAjaxController($html);
                    $section.html('').append($legend, $html);

                    $section = jQuery('div#section-availables'),
                    $legend = $section.find('p.legend'),
                    $html = jQuery(xml.get('html_modules_availables'));

                    self.mvc.defaultAjaxController($html);
                    $section.html('').append($legend, $html);
                }
            };

            // session
            self.session = {
                /**
                 * Create a cookie with the given name and value and other optional parameters.
                 * Based on jQuery cookie plugin.
                 * @name $.cookie
                 * @cat Plugins/Cookie
                 * @author Klaus Hartl/klaus.hartl@stilbuero.de
                 */
                cookie: function (name, value, options) {
                    var expires = '',
                        date,
                        path,
                        domain,
                        secure,
                        i,
                        cookie_value = null,
                        cookies,
                        cookie;

                    if ('undefined' !== typeof value) { // name and value given, set cookie
                        options = options || {};
                        if (value === null) {
                            value = '';
                            options.expires = -1;
                        }

                        if (options.expires && ('number' === typeof options.expires || options.expires.toUTCString)) {
                            if ('number' === typeof options.expires) {
                                date = new Date();
                                date.setTime(date.getTime() + (options.expires * 24 * 60 * 60 * 1000));
                            } else {
                                date = options.expires;
                            }
                            expires = '; expires=' + date.toUTCString(); // use expires attribute, max-age is not supported by IE
                        }
                        // CAUTION: Needed to parenthesize options.path and options.domain
                        // in the following expressions, otherwise they evaluate to undefined
                        // in the packed version for some reason...
                        path = options.path ? '; path=' + (options.path) : '';
                        domain = options.domain ? '; domain=' + (options.domain) : '';
                        secure = options.secure ? '; secure' : '';
                        document.cookie = [name, '=', encodeURIComponent(value), expires, path, domain, secure].join('');
                    } else { // only name given, get cookie
                        if (document.cookie && '' !== document.cookie) {
                            cookies = document.cookie.split(';');
                            for (i = 0; i < cookies.length; i++) {
                                cookie = jQuery.trim(cookies[i]);
                                // Does this cookie string begin with the name we want?
                                if (cookie.substring(0, name.length + 1) === (name + '=')) {
                                    cookie_value = decodeURIComponent(cookie.substring(name.length + 1));
                                    break;
                                }
                            }
                        }
                        return cookie_value;
                    }
                },

                /**
                 * Returns user id.
                 *
                 * @return string
                */
                getUser: function () {
                    return this.cookie('user');
                },

                ieFix: function () {
                    if (jQuery.browser.msie && jQuery.browser.version <= 8) {
                        self.ajax.loadCss('/css/back/ie.css');
                        $body.addClass('ie ie' + parseInt(jQuery.browser.version, 10));
                    }
                },

                $session_indicator: null,
                $msg_box: null,
                id_timeout_session: null,

                indicator: function () {
                    var delay = self.getConfiguration('server/session/duration');
                    if (!delay) {
                        return;
                    }
                    delay = parseInt(delay, 10) * 1000; // seconds to milliseconds

                    if (!self.session.$session_indicator) {
                        self.session.$session_indicator = jQuery.createElement('span', { "class": "ui-session-status status-ok", "title": "Indicateur de fin de session." });
                        jQuery('div#header span').before(self.session.$session_indicator);
                    } else {
                        clearTimeout(self.session.id_timeout_session);
                        self.session.$session_indicator
                            .removeClass('status-alert').removeClass('status-ko')
                            .addClass('status-ok').attr('title', 'Indicateur de fin de session.');

                        if (self.session.$msg_box) {
                            self.session.$msg_box.remove();
                            self.session.$msg_box = null;
                        }
                    }

                    self.session.id_timeout_session = setTimeout(function () {
                        self.session.$session_indicator.removeClass('status-ok').addClass('status-alert').attr('title', 'Votre session expire dans moins de 10 minutes.');
                        self.session.$msg_box = self.ui.components.messageBox('Votre session expire dans moins de 10 minutes.<br />Pensez à enregistrer votre travail.', 'info');
                        self.ui.effects.scrollTo(self.session.$msg_box);

                        clearTimeout(self.session.id_timeout_session);

                        self.session.id_timeout_session = setTimeout(function () {
                            self.session.$msg_box.remove();
                            self.session.$msg_box = null;

                            self.session.$session_indicator.removeClass('status-alert').addClass('status-ko').attr('title', 'Votre session expire dans moins de 5 minutes.');
                            self.session.$msg_box = self.ui.components.messageBox('Votre session expire dans moins de 5 minutes.<br />Pensez à enregistrer votre travail.', 'warning');
                            self.ui.effects.scrollTo(self.session.$msg_box);

                            clearTimeout(self.session.id_timeout_session);
                            self.session.id_timeout_session = setTimeout(function () {
                                self.session.$msg_box.remove();
                                self.session.$msg_box = null;

                                self.session.$session_indicator.attr('title', 'Votre session est terminée.');
                                self.session.$msg_box = self.ui.components.messageBox('Votre session est terminée.', 'error');
                                self.ui.effects.scrollTo(self.session.$msg_box);
                            }, 4 * 60 * 1000); // end session, 1 minute before real end

                        }, 5 * 60 * 1000); // second message in 5 min, 6 min before end
                    }, delay - (11 * 60 * 1000)); // first message 11 min before end
                }
            };

            // ajax
            self.ajax = {
                /**
                 * Returns data from php data controller.
                 *
                 * @param  string  type  Type of data. @see /libs/app/back/controllers.php : data_controller().
                 * @return object
                 */
                data: function (type) {
                    var uri = self.tools.getUri('data', {"type": type, "fullpath": 1 }),
                        o_xml = null;

                    self.session.indicator(); // restart session time

                    jQuery.ajax({
                        type:       'GET',
                        url:        uri,
                        dataType:   'xml',
                        async:      false,
                        success:    function (xml, textStatus, jqXHR) {
                            var o = null;
                            o_xml = self.ajax.xmlResponseToObject(xml);
                            for (o in o_xml.data) {
                                if (o_xml.data.hasOwnProperty(o) && 'string' === typeof o_xml.data[o]) {
                                    o_xml.data[o] = {"length": 0};
                                }
                            }
                        },
                        error:      function (jqXHR, textStatus, errorThrown) {
                            self.ui.components.notify(jQuery('erreur ajax : ' + textStatus.toString()), 'error');
                        }
                    });
                    return o_xml.data;
                },

                /**
                 * Loads a javascript file.
                 *
                 * @param  string   file   Javascript file path.
                 * @return void
                 */
                loadJs: function (file, async, success, error) {
                    var js_to_load = self.getConfiguration('beCms/files/js/load', []),
                        js_loaded = self.getConfiguration('beCms/files/js/loaded', []),
                        success_to_play,
                        s, e;

                    if (-1 !== jQuery.inArray(file, js_to_load)) {
                        if ('function' === typeof success) {
                            if (-1 !== jQuery.inArray(file, js_loaded)) {
                                success();
                            } else {
                                success_to_play = self.getConfiguration('beCms/files/js/success_to_play', []);
                                success_to_play.push(success);
                                self.setConfiguration('beCms/files/js/success_to_play', success_to_play);
                            }
                        }

                        return;
                    }

                    js_to_load.push(file);
                    self.setConfiguration('beCms/files/js/load', js_to_load);
                    async = async || false;

                    s = function () {
                        js_loaded = self.getConfiguration('beCms/files/js/loaded', []);
                        js_loaded.push(file);
                        self.setConfiguration('beCms/files/js/loaded', js_loaded);
                        if ('function' === typeof success) {
                            success();
                        }
                    };

                    e = function () {
                        js_loaded = self.getConfiguration('beCms/files/js/loaded', []);
                        js_loaded.push(file);
                        self.setConfiguration('beCms/files/js/loaded', js_loaded);
                        if ('function' === typeof error) {
                            error();
                        }
                    };

                    file = self.getConfiguration('request/protocol') + '://' + self.getConfiguration('base_uri_ressources_path') + file;
                    jQuery.ajax({ "url": file, "async": async, "dataType": "script", "success": s, "error": e});
                },

                /**
                 * Loads css file.
                 *
                 * @param  file  string, Css file path.
                 * @return void
                 */
                loadCss: function (file) {
                    if (jQuery('link[href="' + file + '"]').length) {
                        return;
                    }

                    file = self.getConfiguration('request/protocol') + '://' + self.getConfiguration('base_uri_ressources_path') + file;

                    var $head = jQuery('head'),
                        $css;
                    $head.append('<link>');

                    $css = $head.children(":last");
                    $css.attr({
                        "rel":  "stylesheet",
                        "type": "text/css",
                        "href": file
                    });
                },

                loadRessources: function () {
                    var load,
                        i,
                        l,
                        fn;

                    function assign_fn(name) {
                        var u_name = self.tools.underscore(name),
                            fn_name = u_name + '_init';

                        return function() {
                            if (jQuery.isFunction(window[fn_name]))
                                window[fn_name].apply(self);
                            else if (self.modules[u_name] && jQuery.isFunction(self.modules[u_name].init))
                                self.modules[u_name].init.apply(self.modules[u_name]);
                        };
                    }

                    load = self.getConfiguration('modules/load/js');
                    if (load) {
                        load = load.split(';');
                        l = load.length;
                        for (i = 0; i < l; i++) {
                            fn = assign_fn(load[i].split('/')[0]);
                            this.loadJs('/js/back/modules/' + load[i], true, fn);
                        }
                    }

                    load = self.getConfiguration('modules/load/css');
                    if (load) {
                        load = load.split(';');
                        l = load.length;
                        for (i = 0; i < l; i++) {
                            this.loadCss('/css/back/modules/' + load[i]);
                        }
                    }
                },

                /**
                 * Sends a link or a form by ajax.
                 *
                 * @param  jQuery    $obj        jQuery element : a or form.
                 * @param  string    action      Action name.
                 * @param  function  controller  Controller to apply the results.
                 * @return void
                 */
                request: function ($obj, action, controller) {
                    var is_link = 'a' === $obj[0].nodeName.toLowerCase(),
                        data;

                    $obj.trigger('ajax');
                    data = is_link ? 'ajax=1' : self.tools.serialize($obj) + '&action=' + escape(action) + '&ajax=1';

                    self.session.indicator(); // restart session time

                    jQuery.ajax({
                        type:       is_link ? 'GET' : 'POST',
                        url:        is_link ? $obj.attr('href') : $obj.attr('action'),
                        data:       data,
                        dataType:   'xml',
                        async:      false,
                        success:    function (xml, textStatus, jqXHR) {
                            var r = self.ajax.xmlResponseToObject(xml);
                            if (r.parsererror) self.ui.components.notify('Erreur d\'interprétation du résultat ajax', 'error');
                            else controller.apply(self.mvc, [$obj, r, r.$html]);
                            $obj.trigger('ajax-done', {"success": true});
                        },
                        error:      function (jqXHR, textStatus, errorThrown) {
                            self.ui.components.notify(jQuery.createElement('div', { "text": "Erreur ajax : " + textStatus.toString() }), 'error');
                            $obj.trigger('ajax-done', {"success": false});
                        }
                    });
                },

                /**
                 * Returns an associative array from an xml source.
                 *
                 * @param  string  xml  Xml source.
                 * @return object
                */
                xmlResponseToObject: function (xml) {
                    var o_xml = {},
                        message,
                        message_type,
                        message_index;

                    function xml_element_to_js($x, x) {
                        $x.children().each(function (index, key) {
                            var $this = jQuery(this),
                                m,
                                n,
                                text;

                            if (!$this.children().length) {
                                text = $this.text().replace('$$CDATA_START$$', '<![CDATA[', 'g').replace('$$CDATA_END$$', ']]>', 'g'); // restore inner cdata sections
                                m = this.nodeName.match(/array_index_(\d+)/);
                                if (m) {
                                    if (!x.length) {
                                        x.length = 0; // pseudo array
                                    }
                                    x[x.length++] = text;
                                } else {
                                    x[this.nodeName] = text;
                                }
                            } else {
                                m = this.nodeName.match(/array_index_(\d+)/);
                                n = this.nodeName;
                                if (m) {
                                    n = (!x.length ? 0 : x.length);
                                    x.length = n + 1;
                                }
                                x[n] = {};
                                xml_element_to_js($this, x[n]);
                            }
                        });
                    }
                    xml_element_to_js(jQuery(xml), o_xml);

                    if (o_xml.data) {
                        o_xml = o_xml.data;
                    }

                    // display messages
                    if (o_xml.messages) {
                        for (message_type in o_xml.messages) {
                            if (o_xml.messages.hasOwnProperty(message_type) && o_xml.messages[message_type].length) {
                                message = [];
                                for (message_index = 0; message_index < o_xml.messages[message_type].length; message_index++) {
                                    message.push('<li>' + o_xml.messages[message_type][message_index] + '</li>');
                                }
                                self.ui.components.notify('<ul>' + message.join('') + '</ul>', message_type);
                            }
                        }
                    }

                    // convert "html" in jQuery object
                    o_xml.$html = null;
                    if (o_xml.html) {
                        o_xml.$html = jQuery(o_xml.html);
                        self.mvc.defaultAjaxController(o_xml.$html);
                    }

                    o_xml.get = function (key, default_value) {
                        default_value = default_value || null;
                        var split_key = key.replace(/^\/+/, '').replace(/\/+$/, '').split(/\/+/),
                            e = this,
                            i;

                        for (i = 0; i < split_key.length; i++) {
                            if (i !== split_key.length - 1) {
                                if (e[split_key[i]]) {
                                    e = e[split_key[i]];
                                } else {
                                    return default_value;
                                }
                            } else {
                                return e[split_key[i]] || default_value;
                            }
                        }
                        return default_value;
                    };

                    return o_xml;
                }
            };

            // ui
            self.ui = {
                components: {

                    editor_instance: [],

                    /**
                     * Install the editor.
                     *
                     * @return array  [[editor, $object, editor index]...].
                     */
                    editor: function () {
                        // html editor
                        var $html = jQuery('.ui-editor-html'),
                            base_path,
                            base_uri_ressources_path,
                            r = [], index = self.ui.components.editor_instance.length,
                            wh = parseInt($window.height(), 10),
                            css,
                            css_tmp,
                            media,
                            $code,
                            i;

                        function getConfiguration($obj) {
                            var $config = $obj.prev('input[name^="ui-editor-config"]'),
                                config = {};

                            $config.each(function () {
                                var $this = jQuery(this),
                                    values = $this.val().split(';'),
                                    i,
                                    value;

                                for (i = 0; i < values.length; i++) {
                                    value = values[i].split(':');
                                    config[value[0]] = value[1];
                                }
                            });
                            return config;
                        }

                        if ($html.length) {
                            base_path = '/js/back/modules/weEd/';
                            self.ajax.loadJs(base_path + 'weEd.js', false, function () {
                                self.ajax.loadCss(base_path + 'weEd.css');

                                $html.each(function(h_index, html) {
                                    var $this = jQuery(html),
                                        e_index = h_index + index,
                                        config = jQuery.extend(true, {
                                        "base_path":    self.getConfiguration('request/protocol') + "://" + self.getConfiguration('base_uri_ressources_path') + base_path,
                                        "buttons":      "format-block,format-inline,|,bold,italic,underline,strikethrough,subscript,superscript,nobreakspace,|,bullets,numbering,|,outdent,indent,|,alignleft,center,alignright,justify,|,undo,redo,|,rule,image,link,unlink,table,specials-characters,|,display-css,show-elements,css-class-tools,xhtml-properties,|,source,cleanup,fullscreen",
                                        "doctype":      "xhtml-1.0-strict",
                                        "height":       Math.max(parseInt(wh / 1.9, 10), 300) + "px",
                                        "width":        "auto",
                                        "plugins": [
                                            { "name": "plugins" }, // load all plugins
                                            { "name": "changes_indicator", "options": { "modified_text": "- [modifié]" } },
                                            { "name": "code_mirror", "options": { "path": self.getConfiguration('request/protocol') + '://' + self.getConfiguration('base_uri_ressources_path') + "/js/back/modules/CodeMirror/", "code_mirror_js": "codemirror-html-php-xml-compressed.js", "mode": [] }},
                                            { "name": "effects" },
                                            { "name": "text_to_paragraph" },
                                            { "name": "html_to_xhtml" },
                                            { "name": "link" },
                                            { "name": "image" },
                                            { "name": "specials_characters" },
                                            { "name": "context_menu" },
                                            { "name": "context_menu_table_tools" },
                                            { "name": "context_menu_format_blocks_tools" },
                                            { "name": "xhtml_properties" },
                                            { "name": "undo_redo" },
                                            { "name": "css_class_tools" },
                                            { "name": "format_inline" }
                                        ],
                                        "ready": function() { self.setConfiguration('beCms/editor/loaded', (self.getConfiguration('beCms/editor/loaded') || 0) + 1); }
                                    }, getConfiguration($this));

                                    if (config.css) {
                                        css = config.css.split(',');
                                        css_tmp = [];
                                        base_uri_ressources_path = self.getConfiguration('base_uri_ressources_path');
                                        for (i = 0; i < css.length; i++) {
                                            media = css[i].match(/\/(all|braille|embossed|handheld|print|projection|screen|speech|tty|tv)\//);
                                            media = media && media.length ? media[1] : 'all';
                                            if (css[i].toLowerCase().match(/\.php$/)) css[i] += '?uri=' + self.getConfiguration('request/protocol') + '://' + base_uri_ressources_path;
                                            css_tmp.push({ "href": self.getConfiguration('request/protocol') + '://' + base_uri_ressources_path + css[i], "media": media });
                                        }
                                        config.css = css_tmp;
                                    }

                                    if ($window.width() <= 900)
                                        self.ui.components.editor_instance.push(new weEd(jQuery.extend(false, config, { "width": "95%"}), $this));
                                    else
                                        self.ui.components.editor_instance.push(new weEd(config, $this));

                                    $window.bind('resize.editor-' + index, function() {
                                        self.ui.components.editor_instance[e_index].resize(1, 1);
                                        self.ui.components.editor_instance[e_index].resize(($window.width() > 900 ? config.width : '95%'), Math.max(parseInt(parseInt($window.height(), 10) / 1.9, 10), 300) + "px");
                                    });
                                    r.push(self.ui.components.editor_instance[e_index], $this, e_index);
                                });
                            });
                        }

                        index = self.ui.components.editor_instance.length;

                        $code = jQuery('.ui-editor-php, .ui-editor-xml');
                        if ($code.length) {
                            base_path = '/js/back/modules/CodeMirror/';
                            self.ajax.loadCss(base_path + 'codemirror.css');
                            self.ajax.loadJs(base_path + 'codemirror-html-php-xml-compressed.js', false, function () {
                                $code.each(function (c_index, code) {
                                        var $this = jQuery(code),
                                            e_index = index + c_index,
                                            editor;
                                        editor = CodeMirror.fromTextArea(this, {
                                            "height": Math.max(parseInt(wh / 1.9, 10), 300) + "px",
                                            "workTime": 500,
                                            "lineWrapping": true,
                                            "lineNumbers": true,
                                            "mode": $this.hasClass('ui-editor-php') ? "application/x-httpd-php" : "application/xml",
                                            "tabMode": "indent",
                                            "onChange": function (instance) {
                                                if (!instance.beCms_discard_changes) {
                                                    jQuery('title').html(self.getConfiguration('current_page/title') + ' - [modifé]');
                                                    instance.beCms_discard_changes = true;
                                                }
                                            }
                                        });
                                    editor.beCms_discard_changes = false;
                                    self.ui.components.editor_instance.push(editor);
                                    r.push(self.ui.components.editor_instance[e_index], $this, e_index);
                                    self.setConfiguration('beCms/editor/loaded', (self.getConfiguration('beCms/editor/loaded') || 0) + 1);
                                });
                            });
                        }

                        return r;
                    },

                    /**
                     * Adds a _blank target to element.
                     *
                     * @param  jQuery  $obj  jQuery element, optional.
                     * @return void
                     */
                    externalLinks: function ($obj) {
                        $obj = $obj ? ('a' === $obj[0].nodeName.toLowerCase() ? $obj : $obj.find('a[rel*="external"]')) : jQuery('a[rel*="external"]');
                        $obj.attr('target', '_blank');
                    },

                    /**
                     * Changes x-help span to tooltip help
                     *
                     * @param  jQuery  $obj  jQuery element, optional.
                     * @return void
                     */
                    help: function ($obj) {
                        $obj = $obj || $body;
                        $obj.find('.x-help').hide().each(function () {
                            var $this = jQuery(this),
                                $label = $this.prevAll('label');
                            $label.addClass('help');
                            self.ui.components.tooltip($label, $this);
                        });
                    },

                    /**
                     * Makes a lightbox with html content.
                     *
                     * @param  jQuery  $html  jQuery html element.
                     * @return void
                     */
                    htmlLightbox: function ($html) {
                        $html.css('width', '450px').find('.text-button-ico').toggleClass('text-button-ico').addClass('image-button-ico');
                        self.ui.components.lightbox($html, $html.find('legend').html() || '', { "onCloseDiscardChange": true });
                        self.tools.formFocus($html);
                    },

                    $images_lightbox_container: null,

                    /**
                     * Makes a lightbox with image content on a.ui-lightbox click.
                     *
                     * @return void
                     */
                    imagesLightbox: function () {
                        if (!this.$images_lightbox_container) {
                            this.$images_lightbox_container = jQuery.createElement('div', { "id": "ui-images_lightbox_container" }).hide();
                            $body.append(this.$images_lightbox_container);
                        }

                        var img_list = {},
                            namespace = 'beCms/ui/components/imagesLightbox';
                        jQuery('a.ui-lightbox').each(function () {
                            var $this = jQuery(this);
                            $this.bind('click', function (e) {
                                e.preventDefault();
                                e.stopPropagation();
                                e.stopImmediatePropagation();

                                namespace += '/' + self.tools.uid($this);

                                var src = $this.attr('href'),
                                    title = $this.attr('title') || '',
                                    $img = img_list[src] || jQuery.createElement('img', { "src": src, "title": title }),

                                    resize = function () {
                                        var $this = $img,
                                            d = $this.data('dimensions'),
                                            img_w,
                                            img_h,
                                            resize_w,
                                            resize_h,
                                            img_ratio,
                                            content_ratio;

                                        if (!d) {
                                            d = { "width": $this.width(), "height": $this.height() };
                                            $this.data('dimensions', d);
                                        }
                                        img_w = d.width;
                                        img_h = d.height;
                                        resize_w = (img_w > ($window.width() * 95 / 100) ? ($window.width() * 95 / 100) : img_w);
                                        resize_h = (img_h > ($window.height() * 95 / 100) ? ($window.height() * 95 / 100) : img_h);

                                        if (resize_w < img_w || resize_h < img_h) {
                                            // reduce image to content dimensions
                                            img_ratio = (img_w / img_h);

                                            //var content_ratio = (content_max_width > 0 ? content_max_width : box_max_width) / (content_max_height > 0 ? content_max_height : box_max_height);
                                            content_ratio = (resize_w / resize_h);

                                            if (img_ratio > 1) {
                                                img_w = Math.min(img_w, (content_ratio > 1 ? resize_h : resize_w));
                                                img_h = img_w / img_ratio;
                                            } else {
                                                img_h = Math.min(img_h, resize_h);
                                                img_w = img_h * img_ratio;
                                            }
                                        }

                                        $this.css({ "width": img_w, "height": img_h, "vertical-align": "middle" });
                                        self.ui.effects.centering(self.ui.components.$lightbox);
                                    };

                                img_list[src] = $img;

                                self.ui.components.lightbox($img, title, {
                                    "onLoad": resize,
                                    "onShow": function () { $window.bind('resize.' + namespace, resize); self.ui.components.$lightbox.addClass('image-lightbox'); },
                                    "onClose": function () { $window.unbind('resize.' + namespace); self.ui.components.$images_lightbox_container.append($img); self.ui.components.$lightbox.removeClass('image-lightbox'); }
                                });
                            });
                        });
                    },

                    $overlay: null,
                    $lightbox: null,

                    /**
                     * Displays a lightbox.
                     *
                     * @param  jQuery  $obj   jQuery element for lightbox content.
                     * @param  string  title  Lightbox title.
                     * @return jQuery  The lightbox
                     */
                    lightbox: function ($obj, title, config) {
                        var namespace = 'beCms/ui/components/lightbox',
                            hide,
                            show;

                        config = jQuery.extend({
                            "onLoad": null,
                            "onShow": null,
                            "onClose": null,
                            "onCloseDiscardChange": false
                        }, config, true);

                        if (!this.$lightbox) {
                            this.$lightbox = jQuery.createElement('div', { "id": "ui-lightbox", "style": "display: none" })
                                                .append(jQuery.createElement('div', { "class": "title" }))
                                                .append(jQuery.createElement('div', { "class": "main" }));

                            self.ui.events.closable(this.$lightbox);

                            hide = this.$lightbox.hide;
                            this.$lightbox.hide = function (speed, callback) {
                                var $this = jQuery(this),
                                    config = $this.data('config');

                                if (config.onClose) {
                                    config.onClose.apply(this);
                                }

                                $this.find('div.main').html('');
                                self.ui.effects.centering($this, false);
                                self.ui.components.$overlay.hide();

                                $this.data('config', {});

                                return hide.apply(this, arguments);
                            };

                            show = this.$lightbox.show;
                            this.$lightbox.show = function (duration, callback) {
                                var $this = jQuery(this),
                                    config = $this.data('config');

                                //if ($this.is(':visible')) return ;

                                self.ui.components.$overlay.show();

                                if (config.onShow) {
                                    config.onShow.apply(this);
                                }

                                return show.apply(this, [{ "duration": duration || 10, "complete": function() { self.ui.effects.centering($this); } }]);
                            };

                            $body.append(this.$lightbox);
                        }

                        if (config.onLoad) {
                            $obj.bind('load.' + namespace, function () { config.onLoad.apply($obj); $obj.unbind('load.' + namespace); });
                        }
                        if (config.onCloseDiscardChange) {
                            self.ui.events.discardChanges(this.$lightbox.find('span.ui-closable'));
                        } else {
                            this.$lightbox.find('span.ui-closable').unbind('click.beCms/ui/events/discardChanges');
                        }

                        $obj.find('.ico-cancel').bind('click.' + namespace + '/cancel', function () {
                            self.ui.components.$lightbox.hide();
                        });

                        this.$lightbox.data('config', config);
                        this.$lightbox.find('div.title').html(title || '');
                        this.$lightbox.find('div.main').html('').append($obj);

                        this.overlay({ "onClick": function () { self.ui.components.$lightbox.hide(); self.ui.components.$overlay.hide(); } });

                        this.$lightbox.show();
                        return this.$lightbox;
                    },

                    overlay: function (config) {
                        var namespace = 'beCms/ui/components/overlay',
                            hide,
                            show;

                        config = jQuery.extend({
                            "onClick": null,
                            "onShow": null,
                            "onClose": null
                        }, config, true);

                        if (!this.$overlay) {
                            this.$overlay = jQuery.createElement('div', { "id": "ui-overlay" });

                            hide = this.$overlay.hide;
                            this.$overlay.hide = function (speed, callback) {
                                var $this = jQuery(this),
                                    config = $this.data('config');

                                if (config.onClose) {
                                    config.onClose.apply(this);
                                }

                                $this.unbind('click.' + namespace);

                                $this.data('config', {});
                                $window.unbind('resize.' + namespace + ' scroll.' + namespace + ' keypress.' + namespace);
                                return hide.apply(this, arguments);
                            };

                            show = this.$overlay.show;
                            this.$overlay.show = function (speed, callback) {
                                var $this = jQuery(this),
                                    config = $this.data('config');

                                if (config.onShow) {
                                    config.onShow.apply(this);
                                }
                                if (config.onClick) {
                                    $this.bind('click.' + namespace, config.onClick);
                                }

                                $window.bind('resize.' + namespace + ' scroll.' + namespace, function () {
                                    $this.css({ "width": parseInt($window.width(), 10) + parseInt($window.scrollLeft(), 10), "height": parseInt($window.height(), 10) + parseInt($window.scrollTop(), 10) });
                                });

                                return show.apply(this, arguments);
                            };

                            $body.append(this.$overlay);
                        }

                        this.$overlay.data('config', config);
                        this.$overlay.show().css({ "width": parseInt($window.width(), 10) + parseInt($window.scrollLeft(), 10), "height": parseInt($window.height(), 10) + parseInt($window.scrollTop(), 10) });
                    },

                    $page_menu: null,

                    /**
                     * Adds a page menu from fieldset and div.section
                     *
                     * @return void
                     */
                    menu: function () {
                        var namespace = 'beCms/ui/components/menu',
                            $ul = jQuery.createElement('ul'),
                            $list = jQuery('fieldset, div.section'),
                            $nav;

                        if (self.getConfiguration(namespace)) return ;

                        $list.addClass('none').each(function () {
                            var $this = jQuery(this),
                                title = $this.find('legend, p.legend').text();

                            $ul.append(jQuery.createElement('li').append(
                                jQuery.createElement('a', { "text": title, "href": "#" })
                                    .bind('click.' + namespace, function () {
                                        var $li = jQuery(jQuery(this).closest('li'));
                                        $ul.find('li.active').removeClass('active');
                                        $li.addClass('active');
                                        $list.hide();
                                        $this.show();
                                        self.session.cookie('page_active_tab', '{"page": "' + window.location.href.replace(/#$/g, '') + '", "index": ' + $li.index() + '}', {"path": "/"});
                                    })
                            ));
                        });

                        if ($ul.children().length) {
                            $nav = jQuery.createElement('div', { "class": "nav" }).append($ul);
                            jQuery(jQuery('div.main').get(1)).before($nav);

                            this.$page_menu = $nav;
                        }

                        self.setConfiguration(namespace, true);
                    },

                    menuSelectItem: function () {
                        var namespace = 'beCms/ui/components/menu',
                            tab_index;

                        if (!this.$page_menu) return ;

                        tab_index = self.session.cookie('page_active_tab') || 0;
                        if (tab_index) {
                            tab_index = jQuery.parseJSON(tab_index);
                            tab_index = (tab_index.page !== window.location.href.replace(/#$/g, '')) ? 0 : tab_index.index;
                        }

                        this.$page_menu.find('li:eq(' + (tab_index) + ') a').trigger('click.' + namespace);
                    },

                    $messages_box_area: null,

                    /**
                     * Displays a message box.
                     *
                     * @param  string  message  The message.
                     * @param  string  type     Message type, message box type (error, warning, info, success, loading).
                     * @return jQuery element The message box.
                     */
                    messageBox: function ($message, type) { // type : error, warning, info, success, loading
                        var $obj;

                        type = type || 'info';

                        if ('string' === typeof $message) {
                            $message = (/^(\s*<)/.test($message)) ? jQuery($message) : jQuery('<div>' + $message + '</div>');
                        }
                        if ('ul' !== $message[0].nodeName.toLowerCase()) {
                            $message = jQuery.createElement('ul').append(jQuery.createElement('li').append($message));
                        }

                        $obj = jQuery.createElement('div', { "class": "ui-message-box " + type })
                                .append(jQuery.createElement('span', { "class": "ico" }))
                                .append($message)
                                .append(jQuery.createElement('div', { "class": "clear" }));

                        if (!this.$message_box_area) {
                            this.$message_box_area = $body.find('div#ui-messages-area');
                        }
                        this.$message_box_area.append($obj);

                        return $obj;
                    },

                    $notify_area: null,

                    /**
                     * Displays a notification box.
                     *
                     * @param  jQuery  $message  jQuery element for the message content.
                     * @param  string  type      The message type.
                     * @return jQuery  The notify box.
                     */
                    notify: function ($message, type) {
                        var namespace = 'beCms/ui/components/notify',
                            $obj,
                            hide,
                            show,
                            timeout,
                            $attach;

                        type = type || 'info';

                        if ('string' === typeof $message) {
                            $message = (/^(\s*<)/.test($message)) ? jQuery($message) : jQuery('<div>' + $message + '</div>');
                        }
                        if ('ul' !== $message[0].nodeName.toLowerCase()) {
                            $message = jQuery.createElement('ul').append(jQuery.createElement('li').append($message));
                        }

                        $obj = jQuery.createElement('div', { "class": "ui-notify " + type }).append($message);

                        if (!this.$notify_area) {
                            this.$notify_area = jQuery.createElement('div', { "id": "ui-notify-area" })
                                .bind(namespace + '/position', function () {
                                    self.ui.components.$notify_area.css({ "position": "absolute", "top": $window.scrollTop() + 25, "left": $window.width() - 315 });
                                })
                                .hide();

                            hide = this.$notify_area.hide;
                            this.$notify_area.hide = function (speed, callback) {
                                $window.unbind('resize.' + namespace + ' scroll.' + namespace);
                                return hide.apply(this, arguments);
                            };

                            show = this.$notify_area.show;
                            this.$notify_area.show = function (speed, callback) {
                                $window
                                    .unbind('resize.' + namespace + ' scroll.' + namespace)
                                    .bind('resize.' + namespace + ' scroll.' + namespace, function () {
                                        self.ui.components.$notify_area.trigger(namespace + '/position');
                                    });
                                var r = show.apply(this, arguments);
                                jQuery(this).trigger(namespace + '/position');
                                return r;
                            };

                            $body.append(this.$notify_area);
                        }

                        timeout = setTimeout(function () {
                            $obj.remove();
                            if (!self.ui.components.$notify_area.children().length) {
                                self.ui.components.$notify_area.hide();
                            }
                        }, 10000);

                        /*if ('error' === type) {
                            $attach = jQuery.createElement('span', { "class": "ui-attachable", "text": "<=" }).bind('click.' + namespace, function () {
                                clearTimeout(timeout);
                                $obj.remove();
                                if (!self.ui.components.$notify_area.children().length) {
                                    self.ui.components.$notify_area.hide();
                                }
                                var $msg = self.ui.components.messageBox($message.clone(), 'error');
                                self.ui.events.closable($msg, 'remove');
                            });
                            $obj.prepend($attach);
                        }*/

                        self.ui.events.closable($obj, 'remove');

                        this.$notify_area.append($obj);
                        if (!self.ui.components.$notify_area.is(':visible')) {
                            this.$notify_area.show();
                        }

                        return $obj;
                    },

                    /**
                     * Makes a double selector from two textarea with formatted content.
                     *
                     * @param  jQuery  $values  jQuery element with selected values.
                     * @param  jQuery  $list    jQuery element with availables values.
                     * @param  object  config   Configuration ("showButtonsUpDown" : true).
                     * @return void
                     */
                    toSelect: function ($values, $list, config) {
                        var $select_in = jQuery.createElement('select', { "multiple": "multiple", "class": "ui-from-to-selector" }),
                            $select_out = jQuery.createElement('select', { "multiple": "multiple", "class": "ui-from-to-selector" }),
                            $up_select,
                            $down_select,
                            $add_list,
                            $remove_list,
                            base_id,
                            $span_1,
                            $span_2;

                        if ($list && !($list instanceof jQuery)) {
                            config = $list;
                            $list = null;
                        }

                        if (!config) config = {};
                        config = jQuery.extend({
                            "showButtonsUpDown": true
                        }, config, true);


                        function init() {
                            var val = '',
                                list = [],
                                i;

                            $select_in.children().remove();
                            $select_out.children().remove();

                            if ('textarea' === $values[0].nodeName.toLowerCase() && $list) {
                                val = $values.val().replace(/\s/, '').split(';');
                                for (i = 0; i < val.length; i++) {
                                    val[i] = jQuery.trim(val[i]) || '';
                                    if (val[i].length) {
                                        $select_in.append(jQuery.createElement('option', { "value": val[i], "text": val[i] }));
                                        list.push(val[i]);
                                    }
                                }

                                val = $list.val().replace(/\s/, '').split(';');
                                for (i = 0; i < val.length; i++) {
                                    val[i] = jQuery.trim(val[i]) || '';
                                    if (val[i].length && -1 === jQuery.inArray(val[i], list)) {
                                        $select_out.append(jQuery.createElement('option', { "value": val[i], "text": val[i] }));
                                    }
                                }
                            }
                            else {
                                $values.find('option').each(function() {
                                    i = jQuery(this);
                                    if (i.is(':selected')) {
                                        $select_in.append(jQuery.createElement('option', { "value": i.val(), "text": i.text() }));
                                        list.push(i.val());
                                    }
                                    else {
                                        $select_out.append(jQuery.createElement('option', { "value": i.val(), "text": i.text() }));
                                    }
                                });
                            }
                        }

                        function update_values() {
                            var content = [];
                            if ('textarea' === $values[0].nodeName.toLowerCase()) {
                                $select_in.find('option').each(function () { content.push(jQuery(this).val()); });
                                $values.val(content.join(';')).trigger('change');
                            }
                            else {
                                $values.find('option:selected').removeAttr('selected');
                                $select_in.find('option').each(function() {
                                    $values.find('option[value="' + jQuery(this).val() + '"]').attr('selected', 'selected');
                                }).trigger('change');
                            }
                        }

                        init();
                        $values.unbind('update').bind('update', function (e) { init(); });

                        if (config.showButtonsUpDown) {
                            $up_select = jQuery.createElement('a', { "href": "#", "class": "ico ico-up image-button-ico ui-from-to-selector-up" })
                                .bind('click', function (e) {
                                    e.preventDefault();
                                    e.stopPropagation();
                                    var selected = $select_in.find(':selected'),
                                        prev;
                                    if (selected.length) {
                                        prev = jQuery(selected).first().prev();
                                        jQuery(selected).insertBefore(prev);
                                    }
                                    update_values();
                                });
                            $down_select = jQuery.createElement('a', { "href": "#", "class": "ico ico-down image-button-ico ui-from-to-selector-down" })
                                .bind('click', function (e) {
                                    e.preventDefault();
                                    e.stopPropagation();
                                    var selected = $select_in.find(':selected'),
                                        next;
                                    if (selected.length) {
                                        next = jQuery(selected).last().next();
                                        jQuery(selected).insertAfter(next);
                                    }
                                    update_values();
                                });
                        }
                        $add_list = jQuery.createElement('a', { "href": "#", "class": "ico ico-up image-button-ico ui-from-to-selector-add" })
                            .bind('click', function (e) {
                                e.preventDefault();
                                e.stopPropagation();
                                $select_out.find(':selected').each(function () { $select_in.append(jQuery(this)); });
                                $select_in.focus();
                                update_values();
                            });
                        $remove_list = jQuery.createElement('a', { "href": "#", "class": "ico ico-down image-button-ico ui-from-to-selector-remove" })
                            .bind('click', function (e) {
                                e.preventDefault();
                                e.stopPropagation();
                                $select_in.find(':selected').each(function () { $select_out.append(jQuery(this)); });
                                update_values();
                            });

                        base_id = $values.attr('id');
                        if (base_id && '' !== base_id) {
                            $select_in.attr('id', base_id + '_select_in');
                            $select_out.attr('id', base_id + '_select_out');
                            if (config.showButtonsUpDown) $up_select.attr('id', base_id + '_up_select');
                            if (config.showButtonsUpDown) $down_select.attr('id', base_id + '_down_select');
                            $add_list.attr('id', base_id + '_add_list');
                            $remove_list.attr('id', base_id + '_remove_list');
                        }

                        $span_1 = jQuery.createElement('span', { "class": "form-block" })
                                        .append($select_in,
                                                (config.showButtonsUpDown) ? jQuery.createElement('span', { "class": "ui-from-to-selector-up-down" }).append($up_select, $down_select) : '');

                        $span_2 = jQuery.createElement('span', { "class": "form-block" })
                                            .append(jQuery.createElement('span', { "class": "ui-from-to-selector-add-remove" }).append($add_list, $remove_list), $select_out);

                        $values.before($span_1, $span_2);
                        $values.hide();
                        if ($list) $list.hide();
                    },

                    // tooltip container
                    $tooltip: null,

                    /**
                     * Adds a tooltip to element.
                     *
                     * @param  jQuery  $obj      jQuery element.
                     * @param  mixed   contents  Tooltip content (jQuery element / string / function).
                     * @return void
                     */
                    tooltip: function ($obj, contents) {
                        var namespace = 'beCms/ui/components/tooltip',
                            $tooltip = self.ui.components.$tooltip;

                        if (!$tooltip) {
                            $tooltip = self.ui.components.$tooltip = jQuery.createElement('div', { "id": "ui-tooltip", "style": "display: none; position: absolute" });
                            $body.append($tooltip);
                        }

                        $obj
                            .unbind('mouseover.' + namespace + ' mouseout.' + namespace)
                            .bind('mouseover.' + namespace, function (e) {
                                //var $this = jQuery(this);

                                if ($obj.hasClass('ui-tooltip-active')) {
                                    return;
                                }
                                if (jQuery.isFunction(contents)) {
                                    $tooltip.html(contents()).show(); // function
                                } else if (contents instanceof jQuery) {
                                    $tooltip.append(contents.clone().show()).show(); // jQuery element
                                } else {
                                    $tooltip.html(contents).show(); // string
                                }

                                $tooltip.css({ "top": e.pageY + 15, "left": e.pageX + 15 + 'px'});
                                jQuery('.ui-tooltip-active').removeClass('ui-tooltip-active');
                                $obj.addClass('ui-tooltip-active');
                            })
                            .bind('mouseout.' + namespace, function (e) {
                                $tooltip.hide().html('');
                                jQuery('.ui-tooltip-active').removeClass('ui-tooltip-active');
                            });
                    },

                    /**
                     * Adds treeview functionalities to ul.ui-treeview.
                     *
                     * @return void
                     */
                    treeview: function ($obj) {
                        $obj = $obj || $body;
                        var namespace = 'beCms/ui/components/treeview',
                            $ul = 'ul' === $obj[0].nodeName.toLowerCase() ? $obj : $obj.find('ul.ui-treeview');


                        function close($span) {
                            function open($span) {
                                $span.parent('p').nextAll('ul').show();
                                $span
                                    .removeClass('ui-treeview-plus').addClass('ui-treeview-minus')
                                    .unbind('click.' + namespace)
                                    .bind('click.' + namespace, function () { close(jQuery(this)); });
                            }

                            $span.parent('p').nextAll('ul').hide();
                            $span
                                .removeClass('ui-treeview-minus').addClass('ui-treeview-plus')
                                .unbind('click.' + namespace)
                                .bind('click.' + namespace, function () { open(jQuery(this)); });
                        }

                        $ul.find('span.ui-treeview-minus').bind('click.' + namespace, function () { close(jQuery(this)); });
                    }
                },

                events: {
                    /**
                     * Bind links and buttons with ajaxify class.
                     *
                     * @param  jQuery   $obj    jQuery element. $(body) if $obj is null.
                     * @return void
                     */
                    ajaxify: function ($obj) {
                        var namespace = 'beCms/ui/events/ajaxify';
                        $obj = $obj || $body;

                        // links and forms
                        $obj.find('a.ajaxify, form.ajaxify button[type="submit"]').unbind('click.' + namespace).bind('click.' + namespace, function (e) {
                            return self.mvc.ajaxDispatch(jQuery(this), e);
                        });

                        // not ajax form : preserve tabs
                        $obj.find('form:not(.ajaxify) button[type="submit"], a:not(.ajaxify)').unbind('click.' + namespace).bind('click.' + namespace, function (e) {
                            self.session.cookie('page_active_tab', '{"page": "' + window.location.href.replace(/#$/g, '') + '", "index": ' + jQuery('div#main div.nav ul li.active').index() + '}', {"path": "/"});
                        });

                        // if $obj is a form
                        if ('form' === $obj[0].nodeName.toLowerCase()) {
                            if ($obj.hasClass('ajaxify')) {
                                $obj.find('button[type="submit"]').unbind('click.' + namespace).bind('click.' + namespace, function (e) {
                                    return self.mvc.ajaxDispatch(jQuery(this), e);
                                });
                            } else {
                                $obj.find('button[type="submit"]').unbind('click.' + namespace).bind('click.' + namespace, function (e) {
                                    //self.session.cookie('page-active-tab', jQuery('div#main div.nav ul li.active').index());
                                    self.session.cookie('page_active_tab', '{"page": "' + window.location.href.replace(/#$/g, '') + '", "index": ' + jQuery('div#main div.nav ul li.active').index() + '}', {"path": "/"});                                });
                            }
                        }
                        else if ($obj.hasClass('ajaxify')) {
                            $obj.unbind('click.' + namespace).bind('click.' + namespace, function (e) {
                                return self.mvc.ajaxDispatch(jQuery(this), e);
                            });
                        }
                    },

                    /**
                     * Activates changes indications.
                     *
                     * @param  jQuery  $obj  jQuery element.
                     * @return void
                     */
                    changesIndicator: function ($obj) {
                        $obj = $obj || $body;

                        var namespace = 'beCms/ui/events/changesIndicator',
                            $title = jQuery('title'),
                            title = self.getConfiguration('current_page/title'),

                            $objs_keypress = $obj.find('input[type="text"], input[type="password"], textarea'),
                            $objs_click = $obj.find('input[type="checkbox"], input[type="radio"]'),
                            $objs_change = $obj.find('select'),

                            $buttons = $obj.find('button[type="submit"], button[type="reset"], a.ico-cancel, a.discard-changes, span.discard-changes');

                        function unbind_changes() {
                            $objs_keypress.unbind('keypress.' + namespace + ' change.' + namespace);
                            $objs_click.unbind('click.' + namespace + ' change.' + namespace);
                            $objs_change.unbind('change.' + namespace);
                        }

                        function bind_changes() {
                            var indicates = function () {
                                if (jQuery.browser.msie) document.title = title + ' - [modifié]';
                                else $title.html(title + ' - [modifié]');
                                unbind_changes();
                            };

                            $objs_keypress.bind('keypress.' + namespace + ' change.' + namespace, indicates);
                            $objs_click.bind('click.' + namespace + ' change.' + namespace, indicates);
                            $objs_change.bind('change.' + namespace, indicates);
                        }

                        unbind_changes();
                        bind_changes();
                        this.discardChanges($buttons);
                    },

                    /**
                     * Discards change when a jQuery element element is clicked.
                     *
                     * @param  jQuery  $obj  jQuery element.
                     * @return void
                     */
                    discardChanges: function ($obj) {
                        var namespace = 'beCms/ui/events/discardChanges',
                            $title = jQuery('title'),
                            title = self.getConfiguration('current_page/title');

                        $obj.unbind('click.' + namespace).bind('click.' + namespace, function () {
                            if (jQuery.browser.msie) document.title = title;
                            else $title.html(title);
                        });
                    },

                    /**
                     * Adds a button close to an element.
                     *
                     * @param  jQuery  $obj       jQuery element.
                     * @param  string  eventType  Event type when close button is clicked : hide (default) / remove.
                     * @return void
                     */
                    closable: function ($obj, eventType) {
                        var namespace = 'beCms/ui/events/closable',
                            $closable_box = jQuery.createElement('span', { "class": "ui-closable", "text": "X" });

                        $closable_box.bind('click.' + namespace, function () {
                            if (!eventType || 'hide' === eventType) {
                                $obj.hide();
                            } else {
                                $obj.remove();
                            }
                        });

                        $obj.prepend($closable_box).unbind('close').bind('close', function() {
                            $closable_box.trigger('click.' + namespace);
                        });
                    },

                    /**
                     * Adds a confirmation message when an element is clicked.
                     *
                     * @param  jQuery    $obj     jQuery element.
                     * @param  string    message  Message.
                     * @param  function  fn       A optional function to execute if response is ko.
                     * @return void
                     */
                    confirmationMessage: function ($obj, message, fn) {
                        var namespace = 'beCms/ui/events/confirmationMessage';

                        $obj
                            .unbind('click.' + namespace)
                            .bind('click.' + namespace, function (e) {
                                if (!confirm(message)) {
                                    e.preventDefault();
                                    e.stopImmediatePropagation();
                                    e.stopPropagation();
                                    if (fn && jQuery.isFunction(fn)) {
                                        fn.apply($obj);
                                    }
                                    return false;
                                }
                            });

                        $obj.each(function () {
                            var $this = jQuery(this),
                                events =  jQuery._data(this, 'events'), //$this.data('events'),
                                handlers,
                                handler;
                            if (events) {
                                handlers = events.click;
                                // take out the handler we just inserted from the end
                                handler = handlers.pop();
                                // move it at the beginning
                                handlers.splice(0, 0, handler);
                            }
                        });
                    },

                    /**
                     * Change a message box to notify box.
                     *
                     * @param  jQuery  $obj  jQuery message box. Can be empty.
                     * @return void
                     */
                    notify: function ($obj) {
                        $obj = $obj || jQuery('div.ui-message-box.error, div.ui-message-box.warning, div.ui-message-box.info, div.ui-message-box.success');
                        $obj.each(function () {
                            var $this = jQuery(this),
                                type;
                            $this.find('span.ico').remove();
                            type = ($this.hasClass('error') ? 'error' : ($this.hasClass('warning') ? 'warning' : ($this.hasClass('info') ? 'info' : 'success')));
                            self.ui.components.notify($this.html(), type);
                            $this.remove();
                        });
                    }
                },

                effects: {
                    /**
                     * Center an element.
                     *
                     * @param  jQuery  $obj    jQuery element to center.
                     * @param  object  config  Centering configuration / false to remove the effect.
                     * @return void
                     */
                    centering: function ($obj, config) {
                        var namespace = 'beCms/ui/effects/centering',
                            css,
                            w_namespace;

                        if (false === config) {
                            $obj.unbind(namespace);
                            return;
                        }

                        config = jQuery.extend({
                            "horizontalCenter": true,
                            "verticalCenter": true,
                            "windowScrollTop": true,
                            "windowScrollLeft": true,
                            "windowResize": true
                        }, config, true);

                        //$obj.data('centering.config', config);

                        css = {"position": "absolute"};

                        $obj.css(css);

                        $obj
                            .unbind(namespace)
                            .bind(namespace, function (e) {
                                var ww,
                                    ow,
                                    wh,
                                    oh;
                                if (config.horizontalCenter) {
                                    ww = $window.width();
                                    ow = parseInt($obj.width(), 10); //parseInt($obj.outerWidth(true), 10);

                                    if (ww > ow) {
                                        css.left = (ww - ow) / 2 + (config.windowScrollLeft ? $window.scrollLeft() : 0);
                                    } else {
                                        css.left = (config.windowScrollLeft ? $window.scrollLeft() : 0);
                                    }
                                }
                                if (config.verticalCenter) {
                                    wh = $window.height();
                                    oh = parseInt($obj.outerHeight(true), 10);

                                    if (wh > oh) {
                                        css.top = (wh - oh) / 2 + (config.windowScrollTop ? $window.scrollTop() : 0);
                                    } else {
                                        css.top = /*5 +*/ (config.windowScrollTop ? $window.scrollTop() : 0);
                                    }
                                }
                                $obj.css(css);
                            })
                            .trigger(namespace);

                        // add element uid to window bind namespace
                        w_namespace = namespace + '/' + self.tools.uid($obj);
                        $window.unbind('resize.' + w_namespace + ' scroll.' + w_namespace);

                        if (config.windowScrollTop || config.windowScrollLeft) {
                            $window.bind('scroll.' + w_namespace, function () { $obj.trigger(namespace); });
                        }
                        if (config.windowResize) {
                            $window.bind('resize.' + w_namespace, function () { $obj.trigger(namespace); });
                        }
                    },

                    /**
                     * Scrolls current page to a jQuery element.
                     *
                     * @param  jQuery   $obj    jQuery element.
                     * @param  integer  margin  Scroll to element position - margin. Optional.
                     * @return void
                     */
                    scrollTo: function ($obj, margin) {
                        jQuery($html, $body).animate({
                            "scrollTop": $obj.offset().top - (margin || 15)
                        }, 500);
                    }
                }
            };

            // tools
            self.tools = {
                /**
                 * Camelize a string (an_exemple => AnExemple).
                 *
                 * @param  string  s  string to camelize.
                 * @return string
                 */
                camelize: function (s) {
                    return !s ? '' : s.replace(/[^\w\d\-\_]+/g, ' ').replace(/(?:^|[\-\_\s]+)(\w)/g, function (z, c) {
                        return c ? c.toUpperCase() : '';
                    });
                },

                underscore: function (s) {
                    return !s ? '' : s.replace(/[^\w\d\_]/g, '_').replace(/_+/g, '_').replace(/^_+|_+$/g, '').toLowerCase();
                },

                /**
                 * Adds focus to first object in a form.
                 *
                 * @param  jQuery  $obj  Optional. jQuery form / parent form element. Default is first form.
                 * @return void
                 */
                formFocus: function ($obj) {
                    var $first;
                    $obj = ($obj && 'form' !== $obj[0].nodeName.toLowerCase() ? $obj.find('form:first') : $obj) || $body.find('form:first');

                    if ($obj.length) {
                        $first =
                        $obj
                            .find('input[type!=hidden]:first, select:first, textarea:first')
                            .filter(function () {
                                return !jQuery(this).attr('desabled') && jQuery(this).is(':visible');
                            });
                        if ($first.length) {
                            $first.get(0).focus();
                        }
                    }
                },

                /**
                 * Returns a valide url for a back-office page.
                 *
                 * @param  string  page        Page.
                 * @param  object  parameters  Associative array {"key": "value"}
                 * @return string
                 */
                getUri: function (page, parameters) {
                    var rewriting_mode = 1 === parseInt(self.getConfiguration('server/url_rewriting'), 10),
                        url = self.getConfiguration('request/protocol') + '://' + self.getConfiguration('server/base_uri_path') + '/' +
                                (!rewriting_mode ?
                                    self.getConfiguration('script_name') + '?' + self.getConfiguration('server/page_var_name') + '=' :
                                    '') + self.getConfiguration('administration/route') + '/' + page,
                        p,
                        i;

                    if (parameters) {
                        p = [];
                        for (i in parameters) {
                            if (parameters.hasOwnProperty(i)) {
                                p.push(i + '=' + parameters[i]);
                            }
                        }
                        url += (rewriting_mode ? '?' : '&') + p.join('&');
                    }
                    return url;
                },

                /**
                 * Returns a value from an uri.
                 *
                 * @param  string  uri            Uri.
                 * @param  string  param          Parameter name to return value.
                 * @param  string  default_value  Default value to return if param is not found.
                 * @return string
                 */
                getUriValue: function (uri, param, default_value) {
                    var re = new RegExp('[\\?&]' + param + '=([^&#]*)'),
                        results = re.exec(uri);
                    if (!results) {
                        return default_value || null;
                    }
                    return decodeURIComponent(results[1].replace(/\+/g, " "));
                },

                /**
                 * Returns the controller base name for an uri.
                 *
                 * @param  string   uri      Uri. Optional.
                 * @return string
                 */
                getControllerBaseName: function (uri) {
                    var is_page = !uri;
                    uri = (uri || window.location.href).toLowerCase();

                    // if we just have page config, uses it, otherwise uses uri
                    var cn = (!self.getConfiguration('server/url_rewriting')) ?
                        (self.getConfiguration('page') || 'default') :
                        ((1 != self.getConfiguration('server/url_rewriting')) ?
                        self.tools.getUriValue(uri, self.getConfiguration('server/page_var_name'))
                            .replace(self.getConfiguration('administration/route') + '/', '')
                            .replace(/(#|\&).*/g, '') :
                        uri.replace(self.getConfiguration('request/protocol') + '://' + self.getConfiguration('server/base_uri_path') + '/' + self.getConfiguration('administration/route') + '/', '')
                           .replace(/(#|\?).*/g, ''));

                    // for modules, we need pass by modulesController to load the module controller file
                    if (is_page && 'modules' === cn.substring(0, 7).toLowerCase()) cn = 'modules';
                    cn = self.tools.camelize(cn);
                    return cn.charAt(0).toLowerCase() + cn.substr(1);
                },

                serialize: function ($form) {
                    function serialize_array($form) {
                        return $form.map(function () {
                            return this.elements ? jQuery.makeArray(this.elements) : this;
                        })
                            .filter(function () {
                                return this.name &&
                                    (this.checked || /^(?:select|textarea)/i.test(this.nodeName) || /^(?:color|date|datetime|email|hidden|month|number|password|range|search|tel|text|time|url|week)$/i.test(this.type));
                            })
                            .map(function (i, elem) {
                                var val = jQuery(this).val();
                                return val === null ? null : jQuery.isArray(val) ? jQuery.map(val, function (val, i) { return { name: elem.name, value: val }; }) : { name: elem.name, value: val };
                            })
                            .get();
                    }
                    return jQuery.param(serialize_array($form));
                },

                // uid jQuery element counter
                uid_counter: 0,

                /**
                 * Returns the uid for an object.
                 *
                 * @param  jQuery  $obj  jQuery element.
                 * @return integer
                 */
                uid: function ($obj) {
                    if ($obj.data('beCms/uid')) {
                        return $obj.data('beCms/uid');
                    }
                    $obj.data('beCms/uid', ++self.tools.uid_counter);
                    return self.tools.uid_counter;
                },

                /**
                 * Update editor css.
                 *
                 * @param  my_editor  editor    Editor object.
                 * @param  object     css_list  Css list object.
                 * @return void
                 */
                syncEditorCss: function (editor, css_list) {
                    var $button = editor.$toolbar.find('li.button-display-css'),
                        button_selected,
                        css,
                        media,
                        i;

                    if (!$button.length) {
                        return;
                    }

                    button_selected = $button.hasClass('weed-button-selected');

                    if (button_selected) {
                        $button.click();
                    }

                    editor.removeCss('all');

                    // puts css
                    css = [];
                    for (i = 0; i < css_list.length; i++) {
                        media = css_list[i].match(/\/(all|braille|embossed|handheld|print|projection|screen|speech|tty|tv)\//);
                        media = media && media.length ? media[1] : 'all';
                        if (css_list[i].toLowerCase().match(/\.php$/)) css_list[i] += '?uri=' + beCms.getConfiguration('request/protocol') + '://' + beCms.getConfiguration('base_uri_ressources_path');
                        editor.addCss({ "href": css_list[i], "media": media });
                    }

                    if (button_selected) {
                        $button.click();
                    } else {
                        editor.$head.find('link[type="text/css"][href!="' + editor.options.wysiwym_css + '"]').attr('media', 'none');
                    }
                }
            };

            self.modules = {
            };

            // internals shortcuts
            self.mvc.dispatch();
            return this;
        }
    };

    window.beCms = beCms;
    jQuery(document).ready(function () { beCms.init(); });
})(window);
